﻿using JmpCommon;
using JmpModel.Model;
using JmpModel.Model.Repository;
using JunmpPoliceStation.App_Start;
using JunmpPoliceStation.Models.Repository;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Threading.Tasks;

namespace JunmpPoliceStation.Controllers
{
    [ApiController]
    [Route("api/[controller]")]
    public class BaseJpSupplierController : ControllerBase
    {
        //private ICabinetRepository _CabinetRepository;
        private ILogger<BaseJpSupplierController> _logger;
        UnitOfWork _unitOfWork;

        public BaseJpSupplierController(ILogger<BaseJpSupplierController> logger, UnitOfWork unitOfWork)
        {
            _logger = logger;
            //_CabinetRepository = CabinetRepository;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取供应商列表
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "name":"供应商名称"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListSupplier")]
        public async Task<HttpResponseMessage> GetListSupplier([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        //v
                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<BaseJpSupplier, bool>> expression = t => !t.Id.Equals(null);

                        if (!String.IsNullOrEmpty(entity.name))
                        {
                            string name = entity.name;
                            expression = t => t.Name.Contains(name);
                        }

                        var param = new string[] {
                       "BaseJpSupplierEquipments",
                       "BaseJpSupplierEquipments.Equipment"
                    };

                        List<BaseJpSupplier> dataList = _unitOfWork.SupplierRepository.GetList(expression, t => t.UpdateTime, false, param).ToList();


                        var data = dataList.Select(c => new
                        {
                            c.Id,
                            c.Name,
                            c.SupplierCode,
                            c.ShortName,
                            c.CreateTime,
                            c.UpdateTime,
                            c.UpdateUser,
                            c.Phone,
                            c.Contacts,
                            list = c.BaseJpSupplierEquipments.Where(f => f.State == 0).Select(t => new { t.Id, t.SupplierCode, t.State, t.EquipmentId, t.Equipment.Name }).ToList()
                        }
                        ).ToList();


                        return JsonManager.ReturnSuccessResponse(data);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取供应商(第三方使用)
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "updateTime": "",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetSupplierDevelop")]
        [ServiceFilter(typeof(ActionLimitAttribute))]
        //[NoSignAttribute]
        public async Task<HttpResponseMessage> GetSupplierDevelop([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);


                        Expression<Func<BaseJpSupplier, bool>> expression = t => !t.Id.Equals(null);

                        string updateTime = entity.updateTime;
                        if (string.IsNullOrEmpty(updateTime))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        expression = t => t.UpdateTime >= DateTime.Parse(updateTime);


                        var param = new string[] {
                       "BaseJpSupplierEquipments",
                       "BaseJpSupplierEquipments.Equipment"
                    };

                        List<BaseJpSupplier> pagedata = _unitOfWork.SupplierRepository.GetList(expression, p => p.UpdateTime, false, param).ToList();

                        var data = pagedata.Select(c => new
                        {
                            c.Id,
                            c.Name,
                            c.SupplierCode,
                            c.ShortName,
                            c.CreateTime,
                            c.UpdateTime,
                            c.UpdateUser,
                            c.Phone,
                            c.Contacts,
                            list = c.BaseJpSupplierEquipments.Where(f => f.State == 0).Select(t => new { t.Id, t.SupplierCode, t.State, t.EquipmentId, t.Equipment.Name }).ToList()
                        }
                        ).ToList();

                        var dataPage = new
                        {
                            totalElements = pagedata.Count,
                            content = data
                        };

                        return JsonManager.ReturnSuccessResponse(dataPage);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取供应商分页列表
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "page": "0",
        ///         "size": "10",
        ///         "name":"供应商名称"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetPageSupplier")]
        public async Task<HttpResponseMessage> GetPageSupplier([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }

                        Expression<Func<BaseJpSupplier, bool>> expression = t => !t.Id.Equals(null);

                        if (!String.IsNullOrEmpty(entity.name))
                        {
                            string name = entity.name;
                            expression = t => t.Name.Contains(name);
                        }

                        var param = new string[] {
                       "BaseJpSupplierEquipments",
                       "BaseJpSupplierEquipments.Equipment"
                    };

                        PageData<BaseJpSupplier> pagedata = _unitOfWork.SupplierRepository.GetPage(expression, "-UpdateTime", page, size, false, param);

                        var data = pagedata.content.Select(c => new
                        {
                            c.Id,
                            c.Name,
                            c.SupplierCode,
                            c.ShortName,
                            c.CreateTime,
                            c.UpdateTime,
                            c.UpdateUser,
                            c.Phone,
                            c.Contacts,
                            list = c.BaseJpSupplierEquipments.Where(f => f.State == 0).Select(t => new { t.Id, t.SupplierCode, t.State, t.EquipmentId, t.Equipment.Name }).ToList()
                        }
                        ).ToList();

                        var dataPage = new
                        {
                            totalElements = pagedata.totalElements,
                            content = data
                        };

                        return JsonManager.ReturnSuccessResponse(dataPage);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }


        /// <summary>
        /// 添加或修改供应商信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "编号",
        ///        "name":"供应商名称",
        ///        "supplierCode" : "供应商代码",
        ///        "shortName" : "供应商简称",
        ///        "updateUser":"操作用户ID"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("AddOrUpdateSupplier")]
        [ServiceFilter(typeof(ActionLimitAttribute))]
        public async Task<HttpResponseMessage> AddOrUpdateSupplier([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";
                        string supplierCode = entity?.supplierCode ?? "";
                        string updateUser = entity?.updateUser ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(entity.name ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"Name is require");
                        }
                        else if (string.IsNullOrEmpty(supplierCode) || supplierCode.Length != 9)
                        {
                            return JsonManager.SimpleCustResponse($"SupplierCode is require");
                        }
                        else if (string.IsNullOrEmpty(updateUser))
                        {
                            return JsonManager.SimpleCustResponse($"updateUser is require");
                        }
                        else if (!string.IsNullOrEmpty(id))
                        {
                            var supplierInfo = _unitOfWork.SupplierRepository.Get(p => p.SupplierCode.Equals(supplierCode));
                            if (supplierInfo != null && supplierInfo.Id != id)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.SUPPLIER_ERROR);

                            }
                            Expression<Func<BaseJpSupplier, bool>> expression = t => t.Id.Equals(id);
                            BaseJpSupplier updateObject = _unitOfWork.SupplierRepository.Get(expression);
                            if (updateObject != null)
                            {
                                string name = entity.name;
                                //判断重名
                                if (_unitOfWork.SupplierRepository.Any(x => x.Name == name && x.Id != updateObject.Id))
                                {
                                    return JsonManager.SimpleCustResponse($"供应商名称重复");
                                }

                                updateObject.Name = name;
                                updateObject.ShortName = entity.shortName;
                                updateObject.SupplierCode = entity.supplierCode;
                                updateObject.UpdateTime = System.DateTime.Now;
                                updateObject.UpdateUser = entity.updateUser;
                                updateObject.Phone = entity.phone;
                                updateObject.Contacts = entity.contacts;
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }

                            bool result = _unitOfWork.SupplierRepository.Update(updateObject);

                            if (result)
                            {
                                return JsonManager.ReturnSuccessResponse(new
                                {
                                    updateObject.Id,
                                    updateObject.Name,
                                    updateObject.ShortName,
                                    updateObject.SupplierCode,
                                    updateObject.UpdateTime,
                                    updateObject.UpdateUser,
                                    updateObject.Phone,
                                    updateObject.Contacts,
                                });
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {
                            var supplierInfo = _unitOfWork.SupplierRepository.Get(p => p.SupplierCode.Equals(supplierCode));
                            if (supplierInfo != null)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.SUPPLIER_ERROR);

                            }
                            //新增数据
                            BaseJpSupplier addObject = new BaseJpSupplier()
                            {
                                Id = Guid.NewGuid().ToString(),
                                Sort = null,
                                Name = entity.name,
                                ShortName = entity.shortName,
                                SupplierCode = entity.supplierCode,
                                Phone = entity.phone,
                                Contacts = entity.contacts,
                                CreateTime = System.DateTime.Now,
                                UpdateTime = System.DateTime.Now,
                                UpdateUser = entity.updateUser
                            };

                            string name = entity.name;
                            //判断重名
                            if (_unitOfWork.SupplierRepository.Any(x => x.Name == name))
                            {
                                return JsonManager.SimpleCustResponse($"供应商名称重复");
                            }

                            bool result = _unitOfWork.SupplierRepository.Insert(addObject);

                            if (result)
                            {
                                return JsonManager.ReturnSuccessResponse(new
                                {
                                    addObject.Id,
                                    addObject.Name,
                                    addObject.ShortName,
                                    addObject.SupplierCode,
                                    addObject.UpdateTime,
                                    addObject.UpdateUser,
                                    addObject.Phone,
                                    addObject.Contacts,
                                });
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }


        /// <summary>
        /// 删除供应商信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "编号"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("DeleteSupplier")]
        public async Task<HttpResponseMessage> DeleteSupplier([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse($"id is require");
                        }
                        else if (!string.IsNullOrEmpty(id))
                        {
                            Expression<Func<BaseJpSupplier, bool>> expression = t => t.Id.Equals(id);
                            BaseJpSupplier deleteObject = _unitOfWork.SupplierRepository.Get(expression);
                            if (deleteObject != null)
                            {
                                bool result = _unitOfWork.SupplierRepository.Delete(deleteObject);

                                if (result)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {

                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }


        /// <summary>
        /// 删除供应商对应资产关联。
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "编号"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("DeleteSupplierEquipment")]
        public async Task<HttpResponseMessage> DeleteSupplierEquipment([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse($"id is require");
                        }
                        else if (!string.IsNullOrEmpty(id))
                        {
                            Expression<Func<BaseJpSupplierEquipment, bool>> expression = t => t.Id.Equals(id);
                            BaseJpSupplierEquipment deleteObject = _unitOfWork.SupplierProductRepository.Get(expression);
                            if (deleteObject != null)
                            {
                                deleteObject.State = 1;
                                bool result = _unitOfWork.SupplierProductRepository.Update(deleteObject);

                                if (result)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {

                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }


        /// <summary>
        /// 添加供应商对应资产关联。
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "supplierCode" : "供应商ID",
        ///        "equipmentId" : "资产ID",
        ///        "updateUser":"操作用户ID"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("AddSupplierEquipment")]
        public async Task<HttpResponseMessage> AddSupplierEquipment([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else
                        {
                            //新增数据
                            BaseJpSupplierEquipment addObject = new BaseJpSupplierEquipment()
                            {
                                Id = Guid.NewGuid().ToString(),
                                Sort = null,
                                SupplierCode = entity.supplierCode,
                                EquipmentId = entity.equipmentId,
                                State = 0,
                                CreateTime = System.DateTime.Now,
                                UpdateTime = System.DateTime.Now,
                                UpdateUser = entity.updateUser
                            };

                            bool result = _unitOfWork.SupplierProductRepository.Insert(addObject);

                            if (result)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }
    }
}
