﻿using JmpCommon;
using JmpModel.Model;
using JmpModel.Model.Repository;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Threading.Tasks;
using JmpCommon.Tools;
using JunmpPoliceStation.Extensions;

namespace JunmpPoliceStation.Controllers
{
    /// <summary>
    /// 阈值
    /// </summary>
    [ApiController]
    [Route("api/[controller]")]
    public class ThresholdController : ControllerBase
    {
        private ILogger<PrintController> _logger;
        UnitOfWork _unitOfWork;

        public ThresholdController(ILogger<PrintController> logger, UnitOfWork unitOfWork)
        {
            _logger = logger;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取指定组织机构的阈值配置
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "orgId":"组织机构id"
        ///     }
        /// </remarks>
        [HttpPost("GetAllEquipmentThreshold")]
        public async Task<HttpResponseMessage> GetAllEquipmentThreshold([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        string orgId = entity.orgId ?? "";
                        if (string.IsNullOrEmpty(orgId))
                        {
                            return JsonManager.SimpleCustResponse($"orgId is require");
                        }

                        var thresholds = _unitOfWork.DbContext.CommonJpEquipmentThresholds
                            .Include(x => x.Org)
                            .Include(x => x.Warehouse)
                            .Include(x => x.EquipmentDetail)
                            .Include(x => x.EquipmentSize)
                            .Where(x => x.OrgId == orgId)
                            .OrderBy(x => x.EquipmentDetail.Code)
                            .ToList();
                        var orgData = thresholds.Where(x => x.ScopeType == "Organization").ToList();
                        var warehouseData = thresholds.Where(x => x.ScopeType == "Warehouse").ToList();
                        var dataList = new
                        {
                            orgData = orgData.Select(x => new
                            {
                                id = x.Id,
                                createTime = x.CreateTime,
                                orgId = x.OrgId,
                                orgName = x.Org.Name,
                                warehouseId = x.WarehouseId,
                                warehouseName = x.Warehouse?.Name,
                                equipmentDetailId = x.EquipmentDetailId,
                                equipmentDetailName = x.EquipmentDetail.Name,
                                equipmentSizeId = x.EquipmentSizeId,
                                equipmentSizeName = x.EquipmentSize.SizeName,
                                thresholdCount = x.ThresholdCount,
                                isEnabled = x.IsEnabled,
                                scopeType = x.ScopeType,
                            }),
                            warehouseData = warehouseData.GroupBy(x => x.Warehouse.Name).Select(x => new
                            {
                                x.Key,
                                warehouseId = x.FirstOrDefault()?.WarehouseId,
                                data = x.Select(y => new
                                {
                                    id = y.Id,
                                    createTime = y.CreateTime,
                                    orgId = y.OrgId,
                                    orgName = y.Org.Name,
                                    warehouseId = y.WarehouseId,
                                    warehouseName = y.Warehouse?.Name,
                                    equipmentDetailId = y.EquipmentDetailId,
                                    equipmentDetailName = y.EquipmentDetail.Name,
                                    equipmentSizeId = y.EquipmentSizeId,
                                    equipmentSizeName = y.EquipmentSize.SizeName,
                                    thresholdCount = y.ThresholdCount,
                                    isEnabled = y.IsEnabled,
                                    scopeType = y.ScopeType,
                                })

                            })
                        };
                        return JsonManager.ReturnSuccessResponse(dataList);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取指定组织机构的阈值结果
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "orgId":"组织机构id",
        ///        "type":"全部 低于 高于"
        ///     }
        /// </remarks>
        [HttpPost("GetEquipmentThresholdResult")]
        public async Task<HttpResponseMessage> GetEquipmentThresholdResult([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        string orgId = entity.orgId ?? "";
                        string type = entity.type ?? "";
                        if (string.IsNullOrEmpty(orgId))
                        {
                            return JsonManager.SimpleCustResponse($"orgId is require");
                        }

                        Expression<Func<CommonJpEquipmentThresholdResult, bool>> expression = x => x.Threshold.OrgId == orgId;
                        if (!string.IsNullOrEmpty(type) && type != "全部")
                        {
                            switch (type)
                            {

                                case "低于":
                                    {
                                        expression = expression.AndAlso(x => x.IsBelowThreshold == true);
                                        break;
                                    }
                                case "高于":
                                    {
                                        expression = expression.AndAlso(x => x.IsBelowThreshold == false);
                                        break;
                                    }
                            }
                        }

                        var thresholdResults = _unitOfWork.DbContext.CommonJpEquipmentThresholdResults
                            .Include(x => x.Threshold)
                            .Include(x => x.Threshold.Org)
                            .Include(x => x.Threshold.Warehouse)
                            .Include(x => x.Threshold.EquipmentDetail)
                            .Include(x => x.Threshold.EquipmentSize)
                            .Where(expression)
                            .OrderBy(x => x.Threshold.ScopeType)
                            .ThenBy(x => x.Threshold.EquipmentDetail.Code)
                            .ToList();
                        var orgData = thresholdResults.Where(x => x.Threshold.ScopeType == "Organization").ToList();
                        var warehouseData = thresholdResults.Where(x => x.Threshold.ScopeType == "Warehouse").ToList();

                        var dataList = new
                        {
                            orgData = orgData.Select(x => new
                            {
                                id = x.Id,
                                createTime = x.CreateTime,
                                isBelowThreshold = x.IsBelowThreshold,
                                insideCount = x.InsideCount,
                                realInsideCount = x.RealInsideCount,
                                purchaseFillCount = x.PurchaseFillCount,
                                orgId = x.Threshold.OrgId,
                                orgName = x.Threshold.Org.Name,
                                warehouseId = x.Threshold.WarehouseId,
                                warehouseName = x.Threshold.Warehouse?.Name,
                                equipmentId = x.Threshold.EquipmentDetailId,
                                equipmentDetailName = x.Threshold.EquipmentDetail.Name,
                                equipmentSizeId = x.Threshold.EquipmentSizeId,
                                equipmentSizeName = x.Threshold.EquipmentSize.SizeName,
                                thresholdCount = x.Threshold.ThresholdCount,
                                isEnabled = x.Threshold.IsEnabled,
                                scopeType = x.Threshold.ScopeType,
                                quantity = x.IsBelowThreshold ? x.Threshold.ThresholdCount - x.InsideCount - x.PurchaseFillCount : 0
                            }),
                            warehouseData = warehouseData.GroupBy(x => x.Threshold.Warehouse.Name).Select(x => new
                            {
                                x.Key,
                                warehouseId = x.FirstOrDefault()?.Threshold.WarehouseId,
                                data = x.Select(y => new
                                {
                                    id = y.Id,
                                    createTime = y.CreateTime,
                                    isBelowThreshold = y.IsBelowThreshold,
                                    insideCount = y.InsideCount,
                                    realInsideCount = y.RealInsideCount,
                                    purchaseFillCount = y.PurchaseFillCount,
                                    orgId = y.Threshold.OrgId,
                                    orgName = y.Threshold.Org.Name,
                                    warehouseId = y.Threshold.WarehouseId,
                                    warehouseName = y.Threshold.Warehouse?.Name,
                                    equipmentId = y.Threshold.EquipmentDetailId,
                                    equipmentDetailName = y.Threshold.EquipmentDetail.Name,
                                    equipmentSizeId = y.Threshold.EquipmentSizeId,
                                    equipmentSizeName = y.Threshold.EquipmentSize.SizeName,
                                    thresholdCount = y.Threshold.ThresholdCount,
                                    isEnabled = y.Threshold.IsEnabled,
                                    scopeType = y.Threshold.ScopeType,
                                    quantity = y.IsBelowThreshold ? y.Threshold.ThresholdCount - y.InsideCount - y.PurchaseFillCount : 0
                                })

                            })
                        };

                        return JsonManager.ReturnSuccessResponse(dataList);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 添加或修改阈值信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "编号",
        ///        "equipmentDetailId":"装备细类id",
        ///        "equipmentSizeId":"装备型号id",
        ///        "orgId":"组织机构id",
        ///        "warehouseId":"仓库id",
        ///        "thresholdCount":"阈值",
        ///        "isEnabled":"是否启用",
        ///        "scopeType":"范围类型 Organization ：组织机构 Warehouse：仓库"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("AddOrUpdateThreshold")]
        public async Task<HttpResponseMessage> AddOrUpdateThreshold([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";
                        string equipmentDetailId = entity?.equipmentDetailId ?? "";
                        string equipmentSizeId = entity?.equipmentSizeId ?? "";
                        string orgId = entity?.orgId ?? "";
                        string warehouseId = entity?.warehouseId ?? "";
                        string scopeType = entity?.scopeType ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(entity.equipmentDetailId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"equipmentDetailId is require");
                        }
                        else if (string.IsNullOrEmpty(entity.equipmentSizeId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"equipmentSizeId is require");
                        }
                        else if (string.IsNullOrEmpty(entity.orgId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"orgId is require");
                        }
                        else if (!int.TryParse(entity.thresholdCount, out int thresholdCount) || thresholdCount < 1)
                        {
                            return JsonManager.SimpleCustResponse($"thresholdCount is require");
                        }
                        else if (!bool.TryParse(entity.isEnabled, out bool isEnabled))
                        {
                            return JsonManager.SimpleCustResponse($"isEnabled is require");
                        }
                        else if (scopeType != "Organization" && scopeType != "Warehouse")
                        {
                            return JsonManager.SimpleCustResponse($"scopeType is require");
                        }
                        else if (scopeType == "Warehouse" && string.IsNullOrEmpty(warehouseId))
                        {
                            return JsonManager.SimpleCustResponse($"warehouseId is require");
                        }
                        else if (!string.IsNullOrEmpty(id))
                        {
                            var updateObject = _unitOfWork.DbContext.CommonJpEquipmentThresholds.FirstOrDefault(x => x.Id == id);
                            if (updateObject != null)
                            {
                                //判断是否改变装备+是否禁用阈值
                                if (updateObject.EquipmentDetailId != equipmentDetailId
                                    || updateObject.EquipmentSizeId != equipmentSizeId
                                    || updateObject.ScopeType != scopeType
                                    || updateObject.OrgId != orgId
                                    || ((scopeType != "Warehouse" || updateObject.WarehouseId != warehouseId) && (scopeType != "Organization" || updateObject.WarehouseId != null))
                                    || isEnabled == false
                                    )
                                {
                                    //删除之前阈值结果
                                    var oldResList = _unitOfWork.DbContext.CommonJpEquipmentThresholdResults
                                        .Where(x => x.ThresholdId == updateObject.Id)
                                        .ToList();
                                    _unitOfWork.DbContext.CommonJpEquipmentThresholdResults.RemoveRange(oldResList);
                                }

                                updateObject.EquipmentDetailId = equipmentDetailId;
                                updateObject.EquipmentSizeId = equipmentSizeId;
                                updateObject.OrgId = orgId;
                                updateObject.WarehouseId = scopeType == "Organization" ? null : warehouseId;
                                updateObject.ThresholdCount = thresholdCount;
                                updateObject.IsEnabled = isEnabled;
                                updateObject.ScopeType = scopeType;
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }

                            _unitOfWork.DbContext.CommonJpEquipmentThresholds.Update(updateObject);
                            bool result = _unitOfWork.DbContext.SaveChanges() > 0;
                            if (result)
                            {
                                //重新计算阈值
                                _unitOfWork.CalcThreshold(new List<string> { updateObject.EquipmentSizeId }, orgId, warehouseId);
                                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {
                            //新增数据
                            var addObject = new CommonJpEquipmentThreshold()
                            {
                                Id = Guid.NewGuid().ToString(),
                                CreateTime = DateTime.Now,
                                EquipmentDetailId = entity.equipmentDetailId,
                                EquipmentSizeId = entity.equipmentSizeId,
                                OrgId = entity.orgId,
                                WarehouseId = scopeType == "Organization" ? null : warehouseId,
                                ThresholdCount = thresholdCount,
                                IsEnabled = isEnabled,
                                ScopeType = scopeType,
                            };

                            var oldData = _unitOfWork.DbContext.CommonJpEquipmentThresholds
                                .FirstOrDefault(x => x.EquipmentDetailId == addObject.EquipmentDetailId
                                            && x.EquipmentSizeId == addObject.EquipmentSizeId
                                            && x.OrgId == addObject.OrgId
                                            && x.ScopeType == addObject.ScopeType
                                            && (x.ScopeType == "Organization" || (x.ScopeType == "Warehouse" && x.WarehouseId == addObject.WarehouseId))
                                );
                            if (oldData != null)
                            {
                                return JsonManager.SimpleCustResponse($"阈值配置已存在，保存失败");
                            }

                            _unitOfWork.DbContext.CommonJpEquipmentThresholds.Add(addObject);
                            bool result = _unitOfWork.DbContext.SaveChanges() > 0;
                            if (result)
                            {
                                //重新计算阈值
                                _unitOfWork.CalcThreshold(new List<string> { addObject.EquipmentSizeId }, orgId, warehouseId);
                                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 删除阈值信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "编号"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("DeleteThreshold")]
        public async Task<HttpResponseMessage> DeleteThreshold([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse($"id is require");
                        }

                        var deleteObject = _unitOfWork.DbContext.CommonJpEquipmentThresholds.FirstOrDefault(x => x.Id == id);
                        if (deleteObject != null)
                        {
                            //移除对应阈值计算结果
                            var oldResList = _unitOfWork.DbContext.CommonJpEquipmentThresholdResults
                                .Where(x => x.ThresholdId == deleteObject.Id)
                                .ToList();
                            _unitOfWork.DbContext.CommonJpEquipmentThresholdResults.RemoveRange(oldResList);

                            _unitOfWork.DbContext.CommonJpEquipmentThresholds.Remove(deleteObject);
                            bool result = _unitOfWork.DbContext.SaveChanges() > 0;
                            if (result)
                            {

                                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }
    }
}