﻿using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Threading.Tasks;

namespace JmpCommon.Tools
{

    public class Analyzingepc
    {
        /*---------以下为1.0、2.0版本公共参数-----------*/
        public byte Header;                 // 标头(8bits)： JY物资专用标头  0x11(00010001b)
        public byte IssuerId;               // 发行机构标识符(6bits)：GA组织机构： 0x01(000001b) 国家组织机构:0x02(000010b)
        public string OrganizationCode;     // 组织机构代码(54bits)：JY物资生产厂商使用GA或国家组织机构代码， 英文字母及数字的每一个字符转换为六位二进制
        public byte WzdmLen;                // 物资代码长度(4bits): 指定物资代码的长度(N*8bits)， 此处为0x7
        public UInt64 Wzdm;                 // 物资代码(56bits):  使用警用物资编目数据的物资代码，按十进制转换成十六进制存储
        public UInt64 Hxdm;
        public byte SerialLen;              // 序列号长度(4bits): 指定序列号长度(N*8bits), 此处为0xD

        public byte Ver;                    // 版本号(8bits): 0x01
        public byte TagType;                // 标签类型(8bits):   0x00 件标   0x01 箱标   0x02 装备包
        public byte PackingType;            // 包装类型(4bits):   0x1 单品， 0x2 外包装， 0x3 内包装， 0xF 零头配箱
        public UInt32 ProductionDate;       // 生产日期(24bits): 0x161201
        public byte ExpiryDate;             // 有效期值(6bits):  0-63
        public byte ExpiryDateUnit;         // 有效期时间单位(2bits): 1:01b-日， 2:10b-月， 3:11b-年
        public byte SubPackageNum;          // 下级包装内数量(8bits)：最大255
        public UInt16 WzCount;              // 包装数量(12bits): MAX:4095, 每箱内产品总数
        public UInt32 BoxNo;                // 包装箱序列号(20bits): MAX:65535    redis设置保存
        public UInt16 NoInBox;              // 箱内序列号(12bits):  每箱内产品序号， MAX4095

        /*---------以下为2.0版本新增参数，以下参数用于替换生产日期之后的字段，确保真唯一-----------*/
        public ulong TimeSpan { get; set; }   //当前日期时间戳，精确到毫秒
        public byte MachineNum { get; set; } //服务器识别码(4bits)，用于区分测试服务器与正式服务器生成的数据，实际1b就够了，其他3b做保留
    }

    public static class EpcConvert
    {
        private readonly static string strKeyWords = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";    // 无源标签6位字段对应表
        

        /// <summary>
        /// 此方法用于将普通字符串转换成16进制的字符串。
        /// </summary>
        /// <param name="_str">要转换的字符串。</param>
        /// <returns></returns>
        public static string StringToHex16String(string _str)
        {
            //将字符串转换成字节数组。
            byte[] buffer = System.Text.Encoding.UTF8.GetBytes(_str);
            //定义一个string类型的变量，用于存储转换后的值。
            string result = string.Empty;
            for (int i = 0; i < buffer.Length; i++)
            {
                //将每一个字节数组转换成16进制的字符串，以空格相隔开。
                result += Convert.ToString(buffer[i], 16) + " ";
            }
            return result;
        }
        /// <summary>
        /// 此方法用于将16进制的字节数组转换成16进制的字符串。
        /// </summary>
        /// <param name="_hex16Byte">要转换的16进制的字节数组。</param>
        /// <returns></returns>
        public static string Hex16ByteToHex16String(byte[] _hex16Byte)
        {
            string result = string.Empty;
            //如果字节数组不为空。
            if (_hex16Byte != null)
            {
                for (int i = 0; i < _hex16Byte.Length; i++)
                {
                    //将每一个字节数组转换成16进制string类型的字符串，用空格分隔开。
                    result += _hex16Byte[i].ToString("X2") + " ";
                }
            }
            return result;
        }

        public static byte[] ToHexByte(string hexString)
        {
            hexString = hexString.Replace(" ", "");
            if ((hexString.Length % 2) != 0)
            {
                hexString = hexString + " ";
            }
            byte[] buffer = new byte[hexString.Length / 2];
            for (int i = 0; i < buffer.Length; i++)
            {
                buffer[i] = Convert.ToByte(hexString.Substring(i * 2, 2), 0x10);
            }
            return buffer;
        }

        /// <summary>
        /// 生成1.0EPC
        /// </summary>
        /// <param name="epc"></param>
        /// <returns></returns>
        public static byte[] Epc1Gen(Analyzingepc epc)
        {
            byte[] buf = new byte[40];
            int pos = 0;
            SetData(ref buf, pos, 8, epc.Header);
            pos += 8;
            SetData(ref buf, pos, 6, epc.IssuerId);
            pos += 6;
            SetData(ref buf, pos, 54, OrganizationCodeTo6Bin(epc.OrganizationCode.ToUpper()));
            pos += 54;
            SetData(ref buf, pos, 4, epc.WzdmLen);
            pos += 4;
            var wzlen = ((epc.WzdmLen - 1) % 8) * 8;
            SetData(ref buf, pos, wzlen, epc.Wzdm);     // 物资代码暂时定义最长为64bits
            pos += wzlen;
            SetData(ref buf, pos, 8, epc.Hxdm);     // 物资代码暂时定义最长为64bits
            pos += 8;
            SetData(ref buf, pos, 4, epc.SerialLen);
            pos += 4;
            SetData(ref buf, pos, 8, epc.Ver);
            pos += 8;
            SetData(ref buf, pos, 8, epc.TagType);
            pos += 8;
            SetData(ref buf, pos, 4, epc.PackingType);
            pos += 4;
            SetData(ref buf, pos, 24, epc.ProductionDate);
            pos += 24;
            SetData(ref buf, pos, 6, epc.ExpiryDate);
            pos += 6;
            SetData(ref buf, pos, 2, epc.ExpiryDateUnit);
            pos += 2;
            SetData(ref buf, pos, 8, epc.SubPackageNum);
            pos += 8;
            SetData(ref buf, pos, 12, epc.WzCount);
            pos += 12;
            SetData(ref buf, pos, 20, epc.BoxNo);
            pos += 20;
            SetData(ref buf, pos, 12, epc.NoInBox);
            pos += 12;

            // 双字节对齐， 剩余保留区
            if (pos % 16 != 0)
            {
                pos += 16 - pos % 16;
            }

            // 数据体长度
            var len = pos / 8;
            var crc = Crc16(buf, len);
            SetData(ref buf, pos, 16, crc);

            byte[] rtn = new byte[len + 2];
            Array.Copy(buf, rtn, len + 2);

            return rtn;
        }

        /// <summary>
        /// 生成2.0EPC
        /// </summary>
        /// <param name="epc"></param>
        /// <returns></returns>
        public static byte[] Epc2Gen(Analyzingepc epc)
        {
            byte[] buf = new byte[40];
            int pos = 0;
            SetData(ref buf, pos, 8, epc.Header);
            pos += 8;
            SetData(ref buf, pos, 6, epc.IssuerId);
            pos += 6;
            SetData(ref buf, pos, 54, OrganizationCodeTo6Bin(epc.OrganizationCode.ToUpper()));
            pos += 54;
            SetData(ref buf, pos, 4, epc.WzdmLen);
            pos += 4;
            var wzlen = ((epc.WzdmLen - 1) % 8) * 8;
            SetData(ref buf, pos, wzlen, epc.Wzdm);     // 物资代码暂时定义最长为64bits
            pos += wzlen;
            SetData(ref buf, pos, 8, epc.Hxdm);     // 物资代码暂时定义最长为64bits
            pos += 8;
            SetData(ref buf, pos, 4, epc.SerialLen);
            pos += 4;

            /*----------------2.0版本字段-------------*/
            SetData(ref buf, pos, 8, epc.Ver);
            pos += 8;
            SetData(ref buf, pos, 24, epc.ProductionDate);
            pos += 24;
            SetData(ref buf, pos, 6, epc.ExpiryDate);
            pos += 6;
            SetData(ref buf, pos, 2, epc.ExpiryDateUnit);
            pos += 2;
            SetData(ref buf, pos, 48, epc.TimeSpan);
            pos += 48;
            SetData(ref buf, pos, 12, epc.NoInBox);
            pos += 12;
            SetData(ref buf, pos, 4, epc.MachineNum);

            // 双字节对齐， 剩余保留区
            if (pos % 16 != 0)
            {
                pos += 16 - pos % 16;
            }

            // 数据体长度
            var len = pos / 8;
            var crc = Crc16(buf, len);
            SetData(ref buf, pos, 16, crc);

            byte[] rtn = new byte[len + 2];
            Array.Copy(buf, rtn, len + 2);

            return rtn;
        }

        /// <summary>
        /// 解析EPC
        /// </summary>
        /// <param name="epc"></param>
        /// <returns></returns>
        public static Analyzingepc EpcAnlysing(byte[] epc)
        {
            // CRC 校验
            var crc_chk = Crc16(epc, epc.Length - 2);
            var crc_src = epc[epc.Length - 2] << 8 | epc[epc.Length - 1];
            if (crc_src != crc_chk)
            {
                throw new Exception("CRC校验失败");
            }

            // 数据解析
            Analyzingepc rtn = new Analyzingepc();
            int pos = 0;
            rtn.Header = (byte)GetData(epc, pos, 8);
            pos += 8;
            rtn.IssuerId = (byte)GetData(epc, pos, 6);
            pos += 6;
            var oc = GetData(epc, pos, 54);
            rtn.OrganizationCode = OrganizationCodeToStr(oc);
            pos += 54;
            rtn.WzdmLen = (byte)(GetData(epc, pos, 4));
            pos += 4;
            var wzlen = ((rtn.WzdmLen - 1) % 8) * 8;
            rtn.Wzdm = GetData(epc, pos, wzlen);
            pos += wzlen;
            rtn.Hxdm = GetData(epc, pos, 8);
            pos += 8;
            rtn.SerialLen = (byte)GetData(epc, pos, 4);
            pos += 4;
            rtn.Ver = (byte)GetData(epc, pos, 8);
            pos += 8;

            //2.0版本解析
            if (rtn.Ver == 0x02)
            {
                rtn.ProductionDate = (byte)GetData(epc, pos, 24);
                pos += 24;
                rtn.ExpiryDate = (byte)GetData(epc, pos, 6);
                pos += 6;
                rtn.ExpiryDateUnit = (byte)GetData(epc, pos, 2);
                pos += 2;
                rtn.TimeSpan = (byte)GetData(epc, pos, 48);
                pos += 48;
                rtn.NoInBox = (byte)GetData(epc, pos, 12);
                pos += 12;
                rtn.MachineNum = (byte)GetData(epc, pos, 4);
                pos += 4;
            }
            else
            {
                rtn.TagType = (byte)GetData(epc, pos, 8);
                pos += 8;
                rtn.PackingType = (byte)GetData(epc, pos, 4);
                pos += 4;
                rtn.ProductionDate = (UInt32)GetData(epc, pos, 24);
                pos += 24;
                rtn.ExpiryDate = (byte)GetData(epc, pos, 6);
                pos += 6;
                rtn.ExpiryDateUnit = (byte)GetData(epc, pos, 2);
                pos += 2;
                rtn.SubPackageNum = (byte)GetData(epc, pos, 8);
                pos += 8;
                rtn.WzCount = (UInt16)GetData(epc, pos, 12);
                pos += 12;
                rtn.BoxNo = (UInt32)GetData(epc, pos, 20);
                pos += 20;
                rtn.NoInBox = (UInt16)GetData(epc, pos, 12);
                pos += 12;
            }


            return rtn;
        }


        /// <summary>
        /// 向目标数组指定位置插入数据
        /// </summary>
        /// <param name="data">待操作数组</param>
        /// <param name="pos">起始位置</param>
        /// <param name="len">写入长度bits</param>
        /// <param name="value">写入的值</param>
        private static void SetData(ref byte[] data, int pos, int len, UInt64 value)
        {
            int p = pos;
            while (--len >= 0)
            {
                var bitPos = 7 - p % 8;
                data[p++ / 8] |= (byte)(((value >> len) & 1) << bitPos);
            }
        }

        /// <summary>
        /// 从目标数组获取成员变量
        /// </summary>
        /// <param name="data"></param>
        /// <param name="pos"></param>
        /// <param name="len"></param>
        /// <returns></returns>
        private static UInt64 GetData(byte[] data, int pos, int len)
        {
            UInt64 rtn = 0;
            int p = pos;
            while (len-- > 0)
            {
                var bitPos = 7 - p % 8;
                rtn <<= 1;
                rtn |= (byte)((data[p++ / 8] >> bitPos) & 1);
            }

            return rtn;
        }

        /// <summary>
        /// 转换成字符串
        /// </summary>
        /// <param name="icode"></param>
        /// <returns></returns>
        private static string OrganizationCodeToStr(UInt64 icode)
        {
            string code = "";
            for (int i = 0; i < 9; i++)
            {
                var v = (byte)((icode >> (i * 6)) & 0x3f);
                code = strKeyWords[v] + code;
            }

            return code;
        }

        /// <summary>
        /// 生成十六进制组织代码, 6Bin
        /// </summary>
        /// <param name="code"></param>
        /// <returns>0 失败， 其它：正常的组织代码编号</returns>
        private static UInt64 OrganizationCodeTo6Bin(string code)
        {
            UInt64 iCode = 0;
            if (code.Length == 9)
            {
                for (int i = 0; i < code.Length; i++)
                {
                    var index = strKeyWords.IndexOf(code[i]);
                    if (index == -1)
                    {
                        throw new Exception("输入字符非法!");
                    }
                    iCode <<= 6;
                    iCode |= (byte)index;
                }
            }
            else
            {
                throw new Exception("输入长度非法!");
            }

            return iCode;
        }

        /// <summary>
        /// CRC运算
        /// </summary>
        /// <param name="data"></param>
        /// <param name="len">必须是偶数</param>
        /// <returns></returns>
        public static UInt16 Crc16(byte[] data, int lenth)
        {
            ushort crc = 0x8791;
            for (int i = 0; i < lenth; ++i)
            {
                ushort temp = 0;
                ushort a = (ushort)(((crc >> 8) ^ (0xff & data[i])) << 8);
                for (int j = 0; j < 8; ++j)
                {
                    if (((temp ^ a) & 0x8000) != 0)
                    {
                        temp = (ushort)((temp << 1) ^ 0x1021);
                    }
                    else
                    {
                        temp <<= 1;
                    }
                    a <<= 1;
                }
                crc = (ushort)((crc << 8) ^ temp);
            }
            return crc;
        }
    }
}
