﻿using Google.Protobuf.WellKnownTypes;
using JmpCommon;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Filters;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace JunmpPoliceStation.App_Start
{
    public class ActionLimitAttribute : ActionFilterAttribute
    {
        private ILogger<ActionLimitAttribute> _logger;
        public static MemoryCache _cache = new MemoryCache(new MemoryCacheOptions());
        public ActionLimitAttribute(ILogger<ActionLimitAttribute> logger)
        {
            _logger = logger;
        }
        public override void OnActionExecuting(ActionExecutingContext filterContext)
        {
            try
            {
                //获取访问ip
                var userHostAddress = filterContext.HttpContext.Connection.RemoteIpAddress?.MapToIPv4().ToString();
                if (!string.IsNullOrEmpty(userHostAddress) && IsIP(userHostAddress))
                {
                    //有效ip
                    var url = filterContext.ActionDescriptor.RouteValues["controller"] + "/" + filterContext.ActionDescriptor.RouteValues["action"];
                    _logger.LogDebug("获取访问ip:" + userHostAddress + "地址：api/" + url);

                    if (userHostAddress == "127.0.0.1")
                    {
                        //本地ip过滤
                        return;
                    }

                    var key = userHostAddress + url;
                    if (Exists(key))
                    {
                        int value = int.Parse(Get(key).ToString() ?? string.Empty);
                        if (value > 25) //每分钟上限访问25次
                        {
                            filterContext.Result = new JsonResult(ResultCode.ACTION_LIMIT);
                            return;
                        }
                        else
                        {
                            value += 1;
                        }
                        AddMemoryCache(key, value);
                    }
                    else
                    {
                        AddMemoryCache(key, 1);
                    }
                }
            }
            catch (Exception e)
            {
                _logger.LogError("ActionLimitAttribute 错误:" + e.ToString());
                filterContext.Result = new JsonResult(e.ToString());
            }
        }


        /// <summary>
        /// 检查IP地址格式
        /// </summary>
        /// <param name="ip"></param>
        /// <returns></returns>
        private static bool IsIP(string ip)
        {
            return System.Text.RegularExpressions.Regex.IsMatch(ip, @"^((2[0-4]\d|25[0-5]|[01]?\d\d?)\.){3}(2[0-4]\d|25[0-5]|[01]?\d\d?)$");
        }

        /// <summary>
        /// 验证缓存项是否存在
        /// </summary>
        /// <param name="key">缓存Key</param>
        /// <returns></returns>
        public static bool Exists(string key)
        {
            if (key == null)
            {
                return false;
            }
            return _cache.TryGetValue(key, out _);
        }

        /// <summary>
        /// 获取缓存
        /// </summary>
        /// <param name="key">缓存Key</param>
        /// <returns></returns>
        /// <exception cref="ArgumentNullException"></exception>
        public static object Get(string key)
        {
            if (key == null)
            {
                throw new ArgumentNullException(nameof(key));
            }
            if (!Exists(key))
                throw new ArgumentNullException(nameof(key));


            return _cache.Get(key);
        }

        /// <summary>
        /// 添加缓存
        /// </summary>
        /// <param name="key"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        /// <exception cref="ArgumentNullException"></exception>
        public static bool AddMemoryCache(string key, object value)
        {
            if (key == null)
            {
                throw new ArgumentNullException(nameof(key));
            }
            if (value == null)
            {
                throw new ArgumentNullException(nameof(value));
            }
            DateTimeOffset time = DateTimeOffset.Now.AddMinutes(1); //一分钟后过期
            _cache.Set(key, value, time);
            return Exists(key);
        }
    }
}
