﻿using JmpCommon;
using JmpCommon.Tools;
using JmpModel.Model;
using JmpModel.Model.Repository;
using JunmpPoliceStation.App_Start;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.WebUtilities;
using Microsoft.Extensions.Logging;
using Microsoft.Net.Http.Headers;
using Newtonsoft.Json.Linq;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;

namespace JunmpPoliceStation.Controllers
{
    /// <summary>
    /// 临时文件上传
    /// </summary>
    [ApiController]
    [Route("api/[controller]")]
    public class TempFileController : ControllerBase
    {
        private IWebHostEnvironment _hostingEnvironment;
        private ILogger<SoftUpdateController> _logger;
        UnitOfWork _unitOfWork;
        public TempFileController(ILogger<SoftUpdateController> logger, UnitOfWork unitOfWork, IWebHostEnvironment _hostingEnvironment)
        {
            _logger = logger;
            this._hostingEnvironment = _hostingEnvironment;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取所有临时文件记录
        /// </summary>
        /// <remarks>
        ///  ## 例子
        ///      {
        ///         "name":"文件名"
        ///         "page":0,
        ///         "size":10
        ///      }
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllTempFile")]
        public async Task<HttpResponseMessage> GetAllTempFile([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }
                        Expression<Func<BaseJpTempFile, bool>> expression = t => true;
                        if (!string.IsNullOrEmpty(entity.name))
                        {
                            string name = entity.name;
                            expression = expression.AndAlso(t => t.Name.Contains(name));
                        }

                        var cfgList = _unitOfWork.BaseJpTempFileRepository.GetPage(expression, "UpdateTime", page, size, include: new[] { "UploadUser" });
                        var data = cfgList?.content.Select(s => new
                        {
                            id = s.Id,
                            files = new ArrayList()
                            {
                                new
                                {
                                    id = s.Id,
                                    url = s.Address,
                                    name = Path.GetFileName(s.Address.Replace("/", "\\"))
                                }
                            },
                            address = s.Address,
                            name = s.Name,
                            note = s.Note,
                            updateTime = s.UpdateTime?.ToString("yyyy-MM-dd HH:mm"),
                            uploadUserId = s.UploadUserId,
                            uploadUserName = s.UploadUser.NickName,
                        }).ToList();
                        var content = new
                        {
                            totalElements = cfgList?.totalElements ?? 0,
                            content = data
                        };
                        return JsonManager.ReturnSuccessResponse(content);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 新增临时文件
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "name":"", 
        ///         "address":"",
        ///         "note":"",
        ///         "uploadUserId":""
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("AddTempFile")]
        public async Task<HttpResponseMessage> AddTempFile([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";
                        bool bVer = int.TryParse(entity?.version ?? "0", out int version);
                        if (entity == null)
                        {
                            return JsonManager.SimpleCustResponse($"model is null");
                        }
                        else if (string.IsNullOrEmpty(entity.name ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"name is require");
                        }
                        else if (string.IsNullOrEmpty(entity.address ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"address is require");
                        }
                        else if (string.IsNullOrEmpty(entity.uploadUserId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"uploadUserId is require");
                        }


                        //新增数据
                        var tempFile = new BaseJpTempFile()
                        {
                            Address = entity?.address ?? "",
                            Name = entity?.name ?? "",
                            Note = entity?.note ?? "",
                            Id = Guid.NewGuid().ToString(),
                            UpdateTime = DateTime.Now,
                            UploadUserId = entity.uploadUserId ?? ""
                        };

                        var user = _unitOfWork.CommonJpUserRepository.Get(tempFile.UploadUserId);
                        if (user == null)
                        {
                            return JsonManager.SimpleCustResponse($"uploadUserId is require");
                        }

                        _unitOfWork.BaseJpTempFileRepository.Insert(tempFile, false);
                        var result = _unitOfWork.BaseJpTempFileLogRepository.Insert(new BaseJpTempFileLog
                        {
                            Id = Guid.NewGuid().ToString(),
                            CreateTime = DateTime.Now,
                            FileName = tempFile.Address,
                            Type = "add",
                            UpdateUserName = user.NickName
                        });

                        if (result)
                        {

                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }


                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 删除上传的文件
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        ///        ## 例子
        ///         {
        ///         "id": "",
        ///         "uploadUserId":""
        ///         }
        ///   </remarks>   
        [HttpPost("DeleteTempFile")]
        public async Task<HttpResponseMessage> DeleteTempFile([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";
                        string uploadUserId = entity?.uploadUserId ?? "";
                        if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse("id is require");
                        }
                        if (string.IsNullOrEmpty(uploadUserId))
                        {
                            return JsonManager.SimpleCustResponse("uploadUserId is require");
                        }

                        var tempFile = _unitOfWork.DbContext.BaseJpTempFiles.FirstOrDefault(x => x.Id == id);

                        if (tempFile == null)
                        {
                            return JsonManager.SimpleCustResponse("Invalid parameter");
                        }
                        var user = _unitOfWork.CommonJpUserRepository.Get(tempFile.UploadUserId);
                        if (user == null)
                        {
                            return JsonManager.SimpleCustResponse($"uploadUserId is require");
                        }

                        _unitOfWork.BaseJpTempFileLogRepository.Insert(new BaseJpTempFileLog
                        {
                            Id = Guid.NewGuid().ToString(),
                            CreateTime = DateTime.Now,
                            FileName = tempFile.Address,
                            Type = "delete",
                            UpdateUserName = user.NickName
                        }, false);

                        if (_unitOfWork.BaseJpTempFileRepository.Delete(tempFile, true))
                        {
                            var file = tempFile.Address.Replace("/", "\\");
                            var fullName = _hostingEnvironment.WebRootPath + file;
                            if (System.IO.File.Exists(fullName))
                            {
                                System.IO.File.Delete(fullName);
                            }
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleCustResponse(ex.ToString());
                }
            });
        }

        /// <summary>
        /// 获取临时文件操作日志
        /// </summary>
        /// <remarks>
        ///  ## 例子
        ///      {
        ///         "page":0,
        ///         "size":10
        ///      }
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetTempFileLog")]
        public async Task<HttpResponseMessage> GetTempFileLog([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }
                        Expression<Func<BaseJpTempFileLog, bool>> expression = t => true;

                        var cfgList = _unitOfWork.BaseJpTempFileLogRepository.GetPage(expression, "-CreateTime", page, size);
                        var data = cfgList?.content.Select(s => new
                        {
                            id = s.Id,
                            type = s.Type,
                            fileName = s.FileName,
                            createTime = s.CreateTime?.ToString("yyyy-MM-dd HH:mm"),
                            updateUserName = s.UpdateUserName,
                        }).ToList();
                        var content = new
                        {
                            totalElements = cfgList?.totalElements ?? 0,
                            content = data
                        };
                        return JsonManager.ReturnSuccessResponse(content);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }


        /// <summary>
        /// 大文件上传，参数“tempFile”，可自定义
        /// </summary>
        /// <returns></returns>
        [HttpPost("UploadTempFile")]
        [RequestSizeLimit(419430400)/*文件大小上限，400M，可自定义*/]
        [NoSign]
        public async Task<HttpResponseMessage> UploadTempFile()
        {
            try
            {
                //获取boundary
                var boundary = HeaderUtilities.RemoveQuotes(MediaTypeHeaderValue.Parse(Request.ContentType).Boundary).Value ?? "";
                //得到reader
                var reader = new MultipartReader(boundary, HttpContext.Request.Body);
                var section = await reader.ReadNextSectionAsync();

                var webRoot = _hostingEnvironment.WebRootPath/*IWebHostEnvironment，用于获取网站、API根目录*/;
                bool flag = false;
                long fileLen = 0;
                List<string> extList = new List<string>() { ".exe", ".apk", ".zip", ".rar", ".bin" };
                var savePath = string.Empty;
                //读取section
                while (section != null)
                {
                    var hasContentDispositionHeader = ContentDispositionHeaderValue.TryParse(section.ContentDisposition, out var contentDisposition);
                    if (hasContentDispositionHeader)
                    {
                        if (contentDisposition.Name == "tempFile")
                        {
                            var fileName = contentDisposition.FileName.Value;
                            var fileExt = Path.GetExtension(fileName);
                            if (extList.Contains(fileExt))
                            {
                                flag = true;
                                var filePath = "TempFile/" + DateTime.Now.ToString("yyyyMMdd\\/HHmmssfff");
                                var dir = Path.Combine(webRoot, filePath);
                                if (!Directory.Exists(dir))
                                {
                                    Directory.CreateDirectory(dir);
                                }
                                savePath = $"/{filePath}/{fileName}";
                                fileLen = await WriteFileAsync(section.Body, Path.Combine(dir, fileName));
                            }
                            else
                            {
                                return JsonManager.SimpleCustResponse("unsupported media type");
                            }
                        }
                    }
                    section = await reader.ReadNextSectionAsync();
                }
                if (!flag)
                {
                    return JsonManager.SimpleCustResponse("tempFile is require");
                }
                else
                {
                    if (fileLen > 0)
                    {
                        return JsonManager.ReturnSuccessResponse(savePath);
                    }
                    else
                    {
                        return JsonManager.SimpleCustResponse("file upload failed");
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex.ToString());
                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);

            }

        }

        #region private

        /// <summary>
        /// 写文件导到磁盘
        /// </summary>
        /// <param name="stream">流</param>
        /// <param name="path">文件保存路径</param>
        /// <returns></returns>
        private async Task<int> WriteFileAsync(Stream stream, string path)
        {
            const int FILE_WRITE_SIZE = 84975;//写出缓冲区大小
            int writeCount = 0;
            using (FileStream fileStream = new FileStream(path, FileMode.Create, FileAccess.Write, FileShare.Write, FILE_WRITE_SIZE, true))
            {
                byte[] byteArr = new byte[FILE_WRITE_SIZE];
                int readCount = 0;
                while ((readCount = await stream.ReadAsync(byteArr, 0, byteArr.Length)) > 0)
                {
                    await fileStream.WriteAsync(byteArr, 0, readCount);
                    writeCount += readCount;
                }
            }
            return writeCount;
        }

        #endregion
    }
}
