﻿using JmpCommon;
using JmpCommon.Tools;
using JmpModel.Model;
using JmpModel.Model.Repository;
using JunmpPoliceStation.Models.Repository;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json.Linq;
using System;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Threading.Tasks;
using JunmpPoliceStation.App_Start;
using Newtonsoft.Json.Serialization;
using Newtonsoft.Json;
using System.Text;
using System.Web.Http;

namespace JunmpPoliceStation.Controllers
{
    [ApiController]
    [Route("api/[controller]")]
    public class Screen3DShowController : ControllerBase
    {
        private ILogger<Screen3DShowController> _logger;
        UnitOfWork _unitOfWork;
        public Screen3DShowController(ILogger<Screen3DShowController> logger, UnitOfWork unitOfWork)
        {
            _logger = logger;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取库存信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "page":"page",
        ///         "size":"size",
        ///         "warehouseId":"仓库id",
        ///         "shelfId":"货架id"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpGet("GetAllInventory")]
        [NoSign]
        public HttpResponseMessage GetAllInventory(string warehouseId, string shelfId)
        {
            try
            {
                var param = new string[]
                {
                    "EquipmentCodeNavigation",
                };

                if (String.IsNullOrEmpty(warehouseId))
                {
                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                }

                Expression<Func<CommonJpEquipmentInventory, bool>> expression = x => x.WarehouseCode.Equals(warehouseId);

                if (!string.IsNullOrEmpty(shelfId))
                {
                    expression = expression.AndAlso(x => x.ShelfId == shelfId);
                }

                var equipmentStates = _unitOfWork.EquipmentInventoryRepository
                    .GetList(expression, null, false, param)
                    .OrderByDescending(x => x.CreateTime)
                    .ToList();
                var data = equipmentStates
                    .GroupBy(x => x.EquipmentCode)
                    .Select(x => new
                    {
                        key = x.FirstOrDefault()?.EquipmentCodeNavigation?.Name,
                        name = x.FirstOrDefault()?.EquipmentCodeNavigation?.Name,
                        value = x.Count(),
                        imgurl = "http://192.168.2.39:5000/Screen3DShow/1.png"
                    })
                    .ToList();

                JsonSerializerSettings setting = new JsonSerializerSettings();

                setting.DateFormatHandling = DateFormatHandling.MicrosoftDateFormat;
                setting.DateFormatString = "yyyy-MM-dd HH:mm:ss";
                setting.ContractResolver = new CamelCasePropertyNamesContractResolver();
                setting.MaxDepth = 10; //设置序列化的最大层数
                setting.ReferenceLoopHandling = ReferenceLoopHandling.Ignore;

                return new HttpResponseMessage { Content = new StringContent(JsonConvert.SerializeObject(data, Formatting.Indented, setting), Encoding.GetEncoding("UTF-8"), "application/json") };

            }
            catch (Exception ex)
            {

                //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
            }
        }

        /// <summary>
        /// 获取出入库数据(展示最新30条)
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "page":"page",
        ///         "size":"size",
        ///         "warehouseId":"仓库id"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpGet("GetInventoryInOutLog")]
        [NoSign]
        public HttpResponseMessage GetInventoryInOutLog(string warehouseId)
        {
            try
            {
                var param = new string[]
                {
                    "Eqiupment",
                    "Eqiupment.EquipmentCodeNavigation",
                    "Eqiupment.Shelf",
                };

                if (String.IsNullOrEmpty(warehouseId))
                {
                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                }

                Expression<Func<CommonJpEquipmentState, bool>> expression = x => x.WarehouseId.Equals(warehouseId) && !string.IsNullOrEmpty(x.EqiupmentId);

                var equipmentStates = _unitOfWork.EquipmentStateRepository
                    .GetList(expression, null, false, param)
                    .OrderByDescending(x => x.CreateTime)
                    .Take(30)
                    .ToList();

                var data = equipmentStates
                    .Select(x => new
                    {
                        name = x.Eqiupment?.EquipmentCodeNavigation?.Name,
                        from = x.Eqiupment?.Shelf?.Name ?? "",
                        time = x.CreateTime?.ToString("HH:mm:ss"),
                        type = x.OutInState == 0 ? "出" : "入",
                        imgurl = "http://192.168.2.39:5000/Screen3DShow/1.png"
                    })
                    .ToList();

                JsonSerializerSettings setting = new JsonSerializerSettings();

                setting.DateFormatHandling = DateFormatHandling.MicrosoftDateFormat;
                setting.DateFormatString = "yyyy-MM-dd HH:mm:ss";
                setting.ContractResolver = new CamelCasePropertyNamesContractResolver();
                setting.MaxDepth = 10; //设置序列化的最大层数
                setting.ReferenceLoopHandling = ReferenceLoopHandling.Ignore;

                return new HttpResponseMessage { Content = new StringContent(JsonConvert.SerializeObject(data, Formatting.Indented, setting), Encoding.GetEncoding("UTF-8"), "application/json") };

            }
            catch (Exception ex)
            {

                //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
            }
        }

        /// <summary>
        /// 获取物资状态
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "page":"page",
        ///         "size":"size",
        ///         "warehouseId":"仓库id"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpGet("GetInventoryState")]
        [NoSign]
        public HttpResponseMessage GetInventoryState(string warehouseId)
        {
            try
            {
                var param = new string[]
                {
                    "EquipmentCodeNavigation",
                };

                if (String.IsNullOrEmpty(warehouseId))
                {
                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                }

                Expression<Func<CommonJpEquipmentInventory, bool>> expression = x => x.WarehouseCode.Equals(warehouseId);

                expression = expression.AndAlso(x => new[] { 0, 1, 2, 3, 4, 6, 7, 8, 14 }.Contains(x.CurrentState));

                var equipmentStates = _unitOfWork.EquipmentInventoryRepository
                    .GetList(expression, null, false, param)
                    .OrderBy(x => x.CurrentState)
                    .ToList();
                var data = equipmentStates
                    .GroupBy(x => x.CurrentState)
                    .Select(x => new
                    {
                        name = ((InvState)x.Key).ToString(),
                        value = x.Count()
                    })
                    .ToList();

                JsonSerializerSettings setting = new JsonSerializerSettings();

                setting.DateFormatHandling = DateFormatHandling.MicrosoftDateFormat;
                setting.DateFormatString = "yyyy-MM-dd HH:mm:ss";
                setting.ContractResolver = new CamelCasePropertyNamesContractResolver();
                setting.MaxDepth = 10; //设置序列化的最大层数
                setting.ReferenceLoopHandling = ReferenceLoopHandling.Ignore;

                return new HttpResponseMessage { Content = new StringContent(JsonConvert.SerializeObject(data, Formatting.Indented, setting), Encoding.GetEncoding("UTF-8"), "application/json") };

            }
            catch (Exception ex)
            {

                //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
            }
        }

        /// <summary>
        /// 获取单警柜库存信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "page":"page",
        ///         "size":"size",
        ///         "orgId":"组织机构id"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpGet("GetAllInventoryByCabinet")]
        [NoSign]
        public HttpResponseMessage GetAllInventoryByCabinet(string orgId)
        {
            try
            {
                var param = new string[]
                {
                    "EquipmentCodeNavigation",
                    "Cabinet",
                };

                if (String.IsNullOrEmpty(orgId))
                {
                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                }

                Expression<Func<CommonJpEquipmentInventory, bool>> expression = x => x.Cabinet.OrganizationId == orgId;


                var equipmentStates = _unitOfWork.EquipmentInventoryRepository
                    .GetList(expression, null, false, param)
                    .OrderByDescending(x => x.CreateTime)
                    .ToList();
                var data = equipmentStates
                    .Select(x => new string[3]
                    {
                                x.EquipmentCodeNavigation?.Name ?? "",
                                x.Cabinet == null ? "" : (x.Cabinet.CabinetNum + "-" + x.Cabinet.CabinetChildNum),
                                x.CurrentState==3?"在库":"不在库"
                    })
                    .ToList();

                JsonSerializerSettings setting = new JsonSerializerSettings();

                setting.DateFormatHandling = DateFormatHandling.MicrosoftDateFormat;
                setting.DateFormatString = "yyyy-MM-dd HH:mm:ss";
                setting.ContractResolver = new CamelCasePropertyNamesContractResolver();
                setting.MaxDepth = 10; //设置序列化的最大层数
                setting.ReferenceLoopHandling = ReferenceLoopHandling.Ignore;

                return new HttpResponseMessage { Content = new StringContent(JsonConvert.SerializeObject(data, Formatting.Indented, setting), Encoding.GetEncoding("UTF-8"), "application/json") };

            }
            catch (Exception ex)
            {

                //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
            }
        }
    }

    public enum InvState
    {
        库存内 = 0,
        调拨中 = 1,
        借出中 = 2,
        单警柜中 = 3,
        维修中 = 4,
        已报废 = 6,
        单警柜出库中 = 7,
        领用中 = 8,
        本仓库借用中 = 14,
    }
}
