﻿using JmpCommon;
using JmpCommon.Tools;
using JmpModel.Model;
using JmpModel.Model.Repository;
using JunmpPoliceStation.App_Start;
using JunmpPoliceStation.Models.Repository;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Threading.Tasks;

namespace JunmpPoliceStation.Controllers
{
    [ApiController]
    [Route("api/[controller]")]
    public class EquipmentBoxMarkController : ControllerBase
    {
        //private ICabinetRepository _CabinetRepository;
        private ILogger<CommonJpEquipmentBoxMark> _logger;
        UnitOfWork _unitOfWork;

        public EquipmentBoxMarkController(ILogger<CommonJpEquipmentBoxMark> logger, UnitOfWork unitOfWork)
        {
            _logger = logger;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取箱标
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "orgId":"组织机构id",
        ///         "warehouseId":"仓库id",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetBoxMarkList")]
        public async Task<HttpResponseMessage> GetBoxMarkList([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<CommonJpEquipmentBoxMark, bool>> expression = t => t.ReloadBoxingState != 1;

                        if (!string.IsNullOrEmpty(entity.orgId))
                        {
                            string orgId = entity.orgId;
                            expression = expression.AndAlso(t => t.Warehouse.OrgizationId.Equals(orgId));
                        }
                        if (!string.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = expression.AndAlso(t => t.WarehouseId.Equals(warehouseId));
                        }


                        var param = new string[] {
                            "Warehouse",
                            "EquipmentDetail",
                            "Size",
                            "Supplier",
                        };

                        List<CommonJpEquipmentBoxMark> dataList = _unitOfWork.EquipmentBoxMarkRepository.GetList(expression, t => t.CreateTime, false, param).ToList();


                        var data = dataList.Select(c => new
                        {
                            id = c.Id,
                            createTime = c.CreateTime,
                            epc = c.Epc,
                            oneBoxNum = c.OneBoxNum,
                            warehouseId = c.WarehouseId,
                            warehouseName = c.Warehouse.Name,
                            orgizationId = c.Warehouse.OrgizationId,
                            equipmentDetailId = c.EquipmentDetailId,
                            equipmentDetailName = c.EquipmentDetail.Name,
                            equipmentDetailCode = c.EquipmentDetail.Code,
                            sizeId = c.SizeId,
                            sizeName = c.Size.SizeName,
                            sizeCode = c.Size.SizeCode,
                            sizeEpcType = c.Size.EpcType,
                            supplierId = c.SupplierId,
                            supplierCode = c.Supplier.SupplierCode,
                            supplierName = c.Supplier.Name,
                            oldEpc = c.OldEpc
                        }
                        ).ToList();


                        return JsonManager.ReturnSuccessResponse(data);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取需要打印的重装箱箱标
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "orgId":"组织机构id"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetNeedPrintBoxMarkList")]
        public async Task<HttpResponseMessage> GetNeedPrintBoxMarkList([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<CommonJpEquipmentBoxMark, bool>> expression = t => t.ReloadBoxingState == 1;

                        if (!string.IsNullOrEmpty(entity.orgId))
                        {
                            string orgId = entity.orgId;
                            expression = expression.AndAlso(t => t.Warehouse.OrgizationId.Equals(orgId));
                        }

                        var param = new string[] {
                            "Warehouse",
                            "EquipmentDetail",
                            "Size",
                            "Supplier",
                            "CommonJpEquipmentInventories"
                        };

                        List<CommonJpEquipmentBoxMark> dataList = _unitOfWork.EquipmentBoxMarkRepository.GetList(expression, t => t.CreateTime, false, param).ToList();


                        var data = dataList.Select(c => new
                        {
                            id = c.Id,
                            createTime = c.CreateTime,
                            epc = c.Epc,
                            oneBoxNum = c.OneBoxNum,
                            warehouseId = c.WarehouseId,
                            warehouseName = c.Warehouse.Name,
                            orgizationId = c.Warehouse.OrgizationId,
                            equipmentDetailId = c.EquipmentDetailId,
                            equipmentDetailName = c.EquipmentDetail.Name,
                            equipmentDetailCode = c.EquipmentDetail.Code,
                            sizeId = c.SizeId,
                            sizeName = c.Size.SizeName,
                            sizeCode = c.Size.SizeCode,
                            sizeEpcType = c.Size.EpcType,
                            supplierId = c.SupplierId,
                            supplierCode = c.Supplier.SupplierCode,
                            supplierName = c.Supplier.Name,
                            oldEpc = c.OldEpc,
                            productTime = c.CommonJpEquipmentInventories?.FirstOrDefault()?.ProductTime?.ToString("yyyy-MM-dd"),
                            useTime = c.CommonJpEquipmentInventories?.FirstOrDefault()?.UseTime?.ToString("yyyy-MM-dd"),
                            warrantyCycle = c.CommonJpEquipmentInventories?.FirstOrDefault()?.WarrantyCycle,
                            repairCycle = c.CommonJpEquipmentInventories?.FirstOrDefault()?.RepairCycle,
                            price = c.CommonJpEquipmentInventories?.FirstOrDefault()?.Price,
                            epcList = c.CommonJpEquipmentInventories?.Select(x => x.Epc)
                        }
                        ).ToList();


                        return JsonManager.ReturnSuccessResponse(data);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 通过箱标epc获取件标epc
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "epcList":[
        ///             {"epc":"箱标epc"},
        ///             {"epc":"箱标epc"},
        ///         ]
        ///         
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetEpcListByBoxMarkEpc")]
        public async Task<HttpResponseMessage> GetEpcListByBoxMarkEpc([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<CommonJpEquipmentBoxMark, bool>> expression = t => t.ReloadBoxingState != 1;

                        var epcList = new List<string>();
                        foreach (var epc in entity.epcList)
                        {
                            if (!string.IsNullOrEmpty(epc.epc))
                            {
                                epcList.Add(epc.epc);
                            }
                        }

                        if (epcList.Count == 0)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            expression = expression.AndAlso(x => epcList.Contains(x.Epc) || epcList.Contains(x.OldEpc));
                        }
                        var param = new string[] {
                            "Warehouse",
                            "EquipmentDetail",
                            "Size",
                            "Supplier",
                            "CommonJpEquipmentInventories"
                        };

                        var data = _unitOfWork.EquipmentBoxMarkRepository.GetList(expression, includes: param).ToList();
                        if (data.Count == 0)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.BOX_MARK_INVALID);
                        }

                        return JsonManager.ReturnSuccessResponse(data.Select(x =>
                            new
                            {
                                id = x.Id,
                                createTime = x.CreateTime,
                                epc = x.Epc,
                                oneBoxNum = x.OneBoxNum,
                                warehouseId = x.WarehouseId,
                                warehouseName = x.Warehouse.Name,
                                orgizationId = x.Warehouse.OrgizationId,
                                equipmentInfoId = x.EquipmentDetail.EquipmentId,
                                equipmentDetailId = x.EquipmentDetailId,
                                equipmentDetailName = x.EquipmentDetail.Name,
                                equipmentDetailCode = x.EquipmentDetail.Code,
                                sizeId = x.SizeId,
                                sizeName = x.Size.SizeName,
                                sizeCode = x.Size.SizeCode,
                                sizeEpcType = x.Size.EpcType,
                                supplierId = x.SupplierId,
                                supplierCode = x.Supplier.SupplierCode,
                                supplierName = x.Supplier.Name,
                                oldEpc = x.OldEpc,
                                epcList = x.CommonJpEquipmentInventories.Select(y => y.Epc)
                            }));
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 通过箱标epc获取件标详细信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "epc":"箱标epc"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetInventoryListByBoxMarkEpc")]
        public async Task<HttpResponseMessage> GetInventoryListByBoxMarkEpc([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => t.BoxMark.ReloadBoxingState != 1;

                        string epc = entity.epc;

                        if (string.IsNullOrEmpty(epc))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            expression = expression.AndAlso(x => x.BoxMark.Epc == epc || x.BoxMark.OldEpc == epc);
                        }
                        var param = new string[] {
                            "BoxMark",
                            "WarehouseCodeNavigation",
                            "EquipmentCodeNavigation",
                            "EquipmentSizecodeNavigation",
                            "SupplierCodeNavigation",

                        };

                        var data = _unitOfWork.EquipmentInventoryRepository.GetList(expression, includes: param).ToList();

                        return JsonManager.ReturnSuccessResponse(data.Select(x =>
                            new
                            {
                                id = x.Id,
                                createTime = x.CreateTime,
                                epc = x.Epc,
                                warehouseId = x.WarehouseCode,
                                warehouseName = x.WarehouseCodeNavigation.Name,
                                orgizationId = x.WarehouseCodeNavigation.OrgizationId,
                                equipmentDetailId = x.EquipmentCode,
                                equipmentDetailName = x.EquipmentCodeNavigation.Name,
                                equipmentDetailCode = x.EquipmentCodeNavigation.Code,
                                sizeId = x.EquipmentSizecode,
                                sizeName = x.EquipmentSizecodeNavigation.SizeName,
                                sizeCode = x.EquipmentSizecodeNavigation.SizeCode,
                                sizeEpcType = x.EquipmentSizecodeNavigation.EpcType,
                                supplierId = x.SupplierCode,
                                supplierCode = x.SupplierCodeNavigation.SupplierCode,
                                supplierName = x.SupplierCodeNavigation.Name,
                                price = x.Price,
                                productTime = x.ProductTime,
                                repairCycle = x.RepairCycle,
                                useTime = x.UseTime,
                                warrantyCycle = x.WarrantyCycle,
                                shelfId = x.ShelfId,
                                shelfRange = x.ShelfRange,
                                shelfRow = x.ShelfRow,
                                shelfColumn = x.ShelfColumn,
                            }));
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }


    }
}
