﻿using JmpRfidLp;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Markup;
using System.Windows.Media;
using System.Xml.Linq;
using static System.Runtime.CompilerServices.RuntimeHelpers;

namespace JmpUhfService
{
    public interface IReaderHelper : IDisposable
    {
        event InvDelegate OnInventory;

        delegate void InvDelegate(TagInfoData args);

        bool Open();
        bool Close();
        bool IsOpen();
        string GetFWVerion();
        void InventoryStart();
        void InventoryStop();
        bool SetPALevel(byte level);
        bool SetTxPower(byte port, bool enable, double power);
        List<TagInfoData> InventorySingle();
        Tuple<int, string> ReadTag(ref OperatingData tag);
        Tuple<int, string> WriteTag(OperatingData tag);
    }

    public class TagInfoData
    {
        public string EPC { get; set; }
        public int RSSI { get; set; }
        public byte AntennaPort { get; set; }
        public int SeenCount { get; set; }
    }

    public class OperatingData
    {
        public string EPC { get; set; }
        public string DATA { get; set; }
        public ushort Offset { get; set; }
        public ushort Length { get; set; }
        public RfidBankType Bank { get; set; }
        public string Access { get; set; }
    }

    public enum RfidBankType
    {
        [Description("RFU")] RFU,
        [Description("EPC")] EPC,
        [Description("TID")] TID,
        [Description("USER")] USER,
    }


    /// <summary>
    /// 钧普读写器
    /// </summary>
    public class JmpRfidLpReaderHelper : IReaderHelper
    {
        public event IReaderHelper.InvDelegate OnInventory;
        private Reader _reader;

        public JmpRfidLpReaderHelper(string comPort = "COM3")
        {
            _reader = new Reader(comPort);
            _reader.OnInventory += OnOnInventory;
        }
        public bool Open()
        {
            return _reader.Open();
        }

        public bool Close()
        {
            return _reader.Close();
        }

        public bool IsOpen()
        {
            return _reader.IsOpen();
        }

        public string GetFWVerion()
        {
            return _reader.GetFWVerion();
        }

        public void InventoryStart()
        {
            _reader.InventoryStart();
        }

        public void InventoryStop()
        {
            _reader.InventoryStop();
        }

        public bool SetPALevel(byte level)
        {
            return _reader.SetPALevel(level) == 0;
        }

        public bool SetTxPower(byte port, bool enable, double power)
        {
            return _reader.SetTxPower(port, enable, power) == 0;
        }

        public List<TagInfoData> InventorySingle()
        {
            return _reader.InventorySingle(50).Select(x => new TagInfoData
            {
                AntennaPort = x.AntennaPort,
                EPC = x.EPC,
                RSSI = x.RSSI,
                SeenCount = x.SeenCount
            }).ToList();
        }

        public Tuple<int, string> ReadTag(ref OperatingData tag)
        {
            var data = new OperData
            {
                Access = tag.Access ?? "00000000",
                Bank = (BankType)tag.Bank,
                DATA = tag.DATA ?? "",
                EPC = tag.EPC ?? "",
                Length = tag.Length,
                Offset = tag.Offset
            };
            var res = _reader.ReadTag(ref data);
            tag.Access = data.Access;
            tag.Bank = (RfidBankType)data.Bank;
            tag.DATA = data.DATA;
            tag.EPC = data.EPC;
            tag.Length = data.Length;
            tag.Offset = data.Offset;
            return new Tuple<int, string>((int)res, typeof(OperResult).GetEnumDesc((int)res));
        }

        public Tuple<int, string> WriteTag(OperatingData tag)
        {
            var data = new OperData
            {
                Access = tag.Access ?? "00000000",
                Bank = (BankType)tag.Bank,
                DATA = tag.DATA ?? "",
                EPC = tag.EPC ?? "",
                Length = tag.Length,
                Offset = tag.Offset
            };
            var res = _reader.WriteTag(data);
            return new Tuple<int, string>((int)res, typeof(OperResult).GetEnumDesc((int)res));
        }

        protected virtual void OnOnInventory(TagInfo args)
        {
            OnInventory?.Invoke(new TagInfoData()
            {
                AntennaPort = args.AntennaPort,
                EPC = args.EPC,
                RSSI = args.RSSI,
                SeenCount = args.SeenCount
            });
        }

        public void Dispose()
        {
            _reader.OnInventory -= OnOnInventory;
            _reader = null;
        }
    }

    /// <summary>
    /// 索力得USBRFID读写器
    /// </summary>
    public class USBRFIDReaderHelper : IReaderHelper
    {
        private int _portHandle = -1;
        private bool _isOpen = false;
        private ManualResetEvent inventoryEvent = new ManualResetEvent(true);
        private CancellationTokenSource _token;

        public event IReaderHelper.InvDelegate OnInventory;

        public USBRFIDReaderHelper()
        {

        }
        public bool Open()
        {
            var res = OpenUSBPort(null, ref _portHandle);
            if (res == 0)
            {
                _isOpen = true;
                return true;
            }
            else
            {
                _isOpen = false;
                Log.WorkLog("连接读写器失败：0x" + res.ToString("X2"), "读写器启动失败");
                return false;
            }
        }

        public bool Close()
        {
            _isOpen = false;
            return CloseUSBPort(_portHandle) == 0;
        }

        public bool IsOpen()
        {
            return _isOpen;
        }

        public string GetFWVerion()
        {
            byte TrType = 0;
            byte[] VersionInfo = new byte[2];
            byte ReaderType = 0;
            byte ScanTime = 0;
            byte dfreband = 0;
            byte dmaxfre = 0;
            byte dminfre = 0;
            byte powerdBm = 0;
            byte Ant = 0;
            byte BeepEn = 0;
            byte CheckAnt = 0;
            var res = GetReaderInfo(VersionInfo, ref ReaderType, ref TrType, ref dfreband, ref dmaxfre, ref dminfre, ref powerdBm, ref ScanTime, ref Ant, ref BeepEn, ref CheckAnt, _portHandle);
            if (res == 0)
            {
                return string.Format("{0}.{1:d2}", VersionInfo[0], VersionInfo[1]);
            }
            else
            {
                if (res == 0x30)
                {
                    //通讯失败，掉线
                    _isOpen = false;
                }
                return "";
            }
        }

        public void InventoryStart()
        {
            //开启盘点线程
            _token = new CancellationTokenSource();
            Task.Run(() =>
            {
                int totalLen = 0, cardNum = 0, fCmdRet = 0x30;
                int cardIdx = 0, epcLen = 0;
                int i;
                byte MaskMem = 0;
                byte[] MaskAdr = new byte[2];
                byte MaskLen = 0;
                byte[] MaskData = null;
                byte MaskFlag = 0;
                byte[] EPCList = new byte[30000];
                byte Ant = 0;
                byte AdrTID = 0;
                byte LenTID = 0;
                byte TIDFlag = 0;
                byte FastFlag = 1;

                byte QValue = 4;         //Q值 4
                byte Session = 0;       //Session 0
                byte Target = 0;        //Target A
                byte Scantime = 3;     //巡查时间 10*100ms
                inventoryEvent = new ManualResetEvent(false);
                while (!_token.IsCancellationRequested)
                {
                    try
                    {
                        if (!_isOpen)
                        {
                            return;
                        }
                        fCmdRet = Inventory_G2(QValue, Session, MaskMem, MaskAdr, MaskLen, MaskData, MaskFlag, AdrTID,
                            LenTID, TIDFlag, Target, 0x80, Scantime, FastFlag, EPCList, ref Ant, ref totalLen,
                            ref cardNum, _portHandle);
                        if ((fCmdRet == 1) | (fCmdRet == 2) | (fCmdRet == 3) | (fCmdRet == 4)) //代表已查找结束，
                        {
                            if (cardNum > 0)
                            {
                                i = 0;
                                epcLen = 0;
                                for (cardIdx = 0; cardIdx < cardNum; cardIdx++)
                                {
                                    epcLen = EPCList[i];
                                    byte[] epcNo = new byte[epcLen];
                                    Array.Copy(EPCList, i + 1, epcNo, 0, epcLen);
                                    String strEpcNo = ByteArrToHexStr(epcNo);
                                    byte rssi = EPCList[i + epcLen + 1];
                                    i += epcLen + 2;
                                    OnOnInventory(new TagInfo()
                                    {
                                        AntennaPort = 0,
                                        EPC = strEpcNo,
                                        RSSI = rssi,
                                        SeenCount = 1
                                    });
                                    //AddTagToInvRealTable(strEpcNo, rssi);
                                }
                            }
                        }
                        else
                        {
                            if (fCmdRet == 0x30)
                            {
                                //通讯失败，掉线
                                _isOpen = false;
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        Log.ErrorLog(e.Message, nameof(USBRFIDReaderHelper));
                    }
                }
                inventoryEvent.Set();
            }, _token.Token);
        }

        public void InventoryStop()
        {
            //关闭盘点线程
            _token?.Cancel();
            inventoryEvent.WaitOne(1000);
        }

        public bool SetPALevel(byte level)
        {
            //索力得读写器无天线使能设置
            return true;
        }

        public bool SetTxPower(byte port, bool enable, double power)
        {
            var res = SetRfPower((byte)Convert.ToInt32(power), _portHandle);
            if (res == 0)
            {
                return true;
            }
            else
            {
                if (res == 0x30)
                {
                    //通讯失败，掉线
                    _isOpen = false;
                }
                Log.ErrorLog("设置功率失败：0x" + res.ToString("X2"), nameof(USBRFIDReaderHelper));
                return false;
            }
        }

        public List<TagInfoData> InventorySingle()
        {
            var list = new List<TagInfoData>();
            //只盘点一轮
            int totalLen = 0, cardNum = 0, fCmdRet = 0x30;
            int cardIdx = 0, epcLen = 0;
            int i;
            byte MaskMem = 0;
            byte[] MaskAdr = new byte[2];
            byte MaskLen = 0;
            byte[] MaskData = null;
            byte MaskFlag = 0;
            byte[] EPCList = new byte[30000];
            byte Ant = 0;
            byte AdrTID = 0;
            byte LenTID = 0;
            byte TIDFlag = 0;
            byte FastFlag = 1;

            byte QValue = 1;         //Q值 Q 值的设置应为场内的标签数量约等亍2的Q次方
            byte Session = 0;       //Session 0 建议单张或者少量标签查询选 S0
            byte Target = 0;        //Target A
            byte Scantime = 3;     //巡查时间 3*100ms
            try
            {
                fCmdRet = Inventory_G2(QValue, Session, MaskMem, MaskAdr, MaskLen, MaskData, MaskFlag, AdrTID, LenTID, TIDFlag, Target, 0x80, Scantime, FastFlag, EPCList, ref Ant, ref totalLen, ref cardNum, _portHandle);
                if ((fCmdRet == 1) | (fCmdRet == 2) | (fCmdRet == 3) | (fCmdRet == 4)) //代表已查找结束，
                {
                    if (cardNum > 0)
                    {
                        i = 0;
                        epcLen = 0;
                        for (cardIdx = 0; cardIdx < cardNum; cardIdx++)
                        {
                            epcLen = EPCList[i];
                            byte[] epcNo = new byte[epcLen];
                            Array.Copy(EPCList, i + 1, epcNo, 0, epcLen);
                            String strEpcNo = ByteArrToHexStr(epcNo);
                            byte rssi = EPCList[i + epcLen + 1];
                            i += epcLen + 2;
                            list.Add(new TagInfoData()
                            {
                                AntennaPort = 0,
                                EPC = strEpcNo,
                                RSSI = rssi,
                                SeenCount = 1
                            });
                        }
                    }
                }
                else
                {
                    if (fCmdRet == 0x30)
                    {
                        //通讯失败，掉线
                        _isOpen = false;
                    }
                }
            }
            catch (Exception e)
            {
                Log.ErrorLog(e.Message, nameof(USBRFIDReaderHelper));
            }
            return list;
        }

        public Tuple<int, string> ReadTag(ref OperatingData tag)
        {
            byte ENum;
            byte[] opWordPtr = null;
            byte opLen = 0;
            byte opMem = 0;
            byte[] accPwd = null;
            string str = "";
            byte[] MaskEPC = null;
            byte MaskMem = 0;
            byte[] MaskAdr = null;
            byte MaskLen = 0;
            byte[] MaskData = null;

            ENum = (byte)(str.Length / 4);
            MaskEPC = new byte[ENum * 2];
            MaskEPC = HexStrToByteArr(str);

            opMem = (byte)tag.Bank;  //0x00：保留区；0x01：EPC 存储区；0x02：TID 存储区；0x03：用户存储区。
            opWordPtr = HexStrToByteArr(tag.Offset.ToString("X4"));
            opLen = Convert.ToByte(tag.Length);
            accPwd = HexStrToByteArr(tag.Access);
            byte[] TagData = new byte[opLen * 2];

            var errcode = 0;
            var res = 0;
            for (int retry = 0; retry < 3; retry++)
            {
                res = ExtReadData_G2(MaskEPC, ENum, opMem, opWordPtr, opLen, accPwd, MaskMem, MaskAdr, MaskLen, MaskData, TagData, ref errcode, _portHandle);
                if (res == 0)
                    break;
            }
            if (res == 0)
            {
                tag.DATA = ByteArrToHexStr(TagData);
                return new Tuple<int, string>((int)res, typeof(OperResult).GetEnumDesc(res));
            }
            else
            {
                if (res == 0x30)
                {
                    //通讯失败，掉线
                    _isOpen = false;
                }
                if (res != 0xFC)
                {
                    return new Tuple<int, string>((int)res, typeof(OperResult).GetEnumDesc(res));
                }
                else
                {
                    return new Tuple<int, string>((int)res, typeof(ErrorCodeEnum).GetEnumDesc(errcode));
                }
            }
        }

        public Tuple<int, string> WriteTag(OperatingData tag)
        {
            byte ENum;
            byte WNum = 0;
            byte[] opWordPtr = null;
            byte opMem = 0;
            byte[] opData = null;
            byte[] accPwd = null;
            string str = "";
            byte[] MaskEPC = null;
            byte MaskMem = 0;
            byte[] MaskAdr = null;
            byte MaskLen = 0;
            byte[] MaskData = null;

            ENum = (byte)(str.Length / 4);
            MaskEPC = new byte[ENum * 2];
            MaskEPC = HexStrToByteArr(str);

            opMem = (byte)tag.Bank;  //0x00：保留区；0x01：EPC 存储区；0x02：TID 存储区；0x03：用户存储区。
            opWordPtr = HexStrToByteArr(tag.Offset.ToString("X4"));
            accPwd = HexStrToByteArr(tag.Access);

            opData = HexStrToByteArr(tag.DATA);
            WNum = (byte)(opData.Length / 2);

            var errcode = 0;
            var res = 0;
            for (int retry = 0; retry < 3; retry++)
            {
                res = ExtWriteData_G2(MaskEPC, WNum, ENum, opMem, opWordPtr, opData, accPwd, MaskMem, MaskAdr, MaskLen, MaskData, ref errcode, _portHandle);
                if (res == 0)
                    break;
            }
            if (res == 0)
            {
                return new Tuple<int, string>((int)res, typeof(OperResult).GetEnumDesc(res));
            }
            else
            {
                if (res == 0x30)
                {
                    //通讯失败，掉线
                    _isOpen = false;
                }
                if (res != 0xFC)
                {
                    return new Tuple<int, string>((int)res, typeof(OperResult).GetEnumDesc(res));
                }
                else
                {
                    return new Tuple<int, string>((int)res, typeof(ErrorCodeEnum).GetEnumDesc(errcode));
                }
            }
        }

        protected virtual void OnOnInventory(TagInfo args)
        {
            OnInventory?.Invoke(new TagInfoData()
            {
                AntennaPort = args.AntennaPort,
                EPC = args.EPC,
                RSSI = args.RSSI,
                SeenCount = args.SeenCount
            });
        }

        public void Dispose()
        {
        }

        /// <summary>
        /// 字符串转16进制字节数组
        /// </summary>
        /// <param name="hexString"></param>
        /// <returns></returns>
        public static byte[] HexStrToByteArr(string hexString)
        {
            hexString = hexString.Replace(" ", "");
            if ((hexString.Length % 2) != 0)
                hexString += "0";
            byte[] returnBytes = new byte[hexString.Length / 2];
            for (int i = 0; i < returnBytes.Length; i++)
                returnBytes[i] = Convert.ToByte(hexString.Substring(i * 2, 2), 16);
            return returnBytes;
        }

        /// <summary>
        /// 字节数组转16进制字符串
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static string ByteArrToHexStr(byte[] bytes)
        {
            string returnStr = "";
            if (bytes != null)
            {
                for (int i = 0; i < bytes.Length; i++)
                {
                    returnStr += bytes[i].ToString("X2");
                }
            }
            return returnStr;
        }

        #region USBUHFPlatform

        private const string DLLNAME = @"USBUHFPlatform.dll";

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int OpenUSBPort(string DevSnr, ref int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int CloseUSBPort(int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int CompDevInterfaceName(string DevInterfaceName, int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int GetReaderInfo(
                                                byte[] VersionInfo,
                                                ref byte ReaderType,
                                                ref byte TrType,
                                                ref byte dfreband,
                                                ref byte dmaxfre,
                                                ref byte dminfre,
                                                ref byte powerdBm,
                                                ref byte ScanTime,
                                                ref byte Ant,
                                                ref byte BeepEn,
                                                ref byte CheckAnt,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetRegion(
                                            byte dfreband,
                                            byte dmaxfre,
                                            byte dminfre,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetInventoryScanTime(
                                            byte ScanTime,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetRfPower(
                                            byte PowerDbm,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int GetLedStatus(
                                            ref byte LedPin,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetLedCtrl(
                                            byte LedPin,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetBeep(
                                            byte AvtiveTime,
                                            byte SilentTime,
                                            byte Times,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int GetSerialNo(
                                            StringBuilder SeriaNo,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetSaveLen(
                                            byte SaveLen,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int GetSaveLen(
                                            ref byte SaveLen,
                                            int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int ReadBuffer_G2(
                                                ref int Totallen,
                                                ref int CardNum,
                                                byte[] pEPCList,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int ClearBuffer_G2(int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int GetBufferCnt_G2(
                                                ref int Count,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetTagCustomFunction(
                                     ref byte InlayType,
                                     int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int Inventory_G2(
                                              byte QValue,
                                              byte Session,
                                              byte MaskMem,
                                              byte[] MaskAdr,
                                              byte MaskLen,
                                              byte[] MaskData,
                                              byte MaskFlag,
                                              byte AdrTID,
                                              byte LenTID,
                                              byte TIDFlag,
                                              byte Target,
                                              byte InAnt,
                                              byte Scantime,
                                              byte FastFlag,
                                              byte[] pEPCList,
                                              ref byte Ant,
                                              ref int Totallen,
                                              ref int CardNum,
                                              int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int ExtReadData_G2(
                                               byte[] EPC,
                                               byte ENum,
                                               byte Mem,
                                               byte[] WordPtr,
                                               byte Num,
                                               byte[] Password,
                                               byte MaskMem,
                                               byte[] MaskAdr,
                                               byte MaskLen,
                                               byte[] MaskData,
                                               byte[] Data,
                                               ref int Errorcode,
                                               int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int ExtWriteData_G2(
                                                byte[] EPC,
                                                byte WNum,
                                                byte ENum,
                                                byte Mem,
                                                byte[] WordPtr,
                                                byte[] Wdt,
                                                byte[] Password,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int WriteEPC_G2(
                                                byte[] Password,
                                                byte[] WriteEPC,
                                                byte ENum,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int BlockWrite_G2(
                                                byte[] EPC,
                                                byte WNum,
                                                byte ENum,
                                                byte Mem,
                                                byte WordPtr,
                                                byte[] Wdt,
                                                byte[] Password,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int BlockErase_G2(
                                                byte[] EPC,
                                                byte ENum,
                                                byte Mem,
                                                byte WordPtr,
                                                byte Num,
                                                byte[] Password,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int Lock_G2(
                                                byte[] EPC,
                                                byte ENum,
                                                byte select,
                                                byte setprotect,
                                                byte[] Password,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int KillTag_G2(
                                                byte[] EPC,
                                                byte ENum,
                                                byte[] Killpwd,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetPrivacyByEPC_G2(
                                                byte[] EPC,
                                                byte ENum,
                                                byte[] Password,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetPrivacyWithoutEPC_G2(
                                                byte[] Password,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int ResetPrivacy_G2(
                                                byte[] Password,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int CheckPrivacy_G2(
                                                ref byte readpro,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int EASConfigure_G2(
                                                byte[] EPC,
                                                byte ENum,
                                                byte[] Password,
                                                byte EAS,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int EASAlarm_G2(
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int GetMonza4QTWorkParamter_G2(
                                                byte[] EPC,
                                                byte ENum,
                                                byte[] Password,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref byte QTcontrol,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int SetMonza4QTWorkParamter_G2(
                                                byte[] EPC,
                                                byte ENum,
                                                byte QTcontrol,
                                                byte[] Password,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                ref int Errorcode,
                                                int FrmHandle);

        [DllImport(DLLNAME, CallingConvention = CallingConvention.StdCall)]
        public static extern int InventoryBuffer_G2(
                                                byte QValue,
                                                byte Session,
                                                byte MaskMem,
                                                byte[] MaskAdr,
                                                byte MaskLen,
                                                byte[] MaskData,
                                                byte MaskFlag,
                                                byte AdrTID,
                                                byte LenTID,
                                                byte TIDFlag,
                                                byte Target,
                                                byte InAnt,
                                                byte Scantime,
                                                byte FastFlag,
                                                ref int BufferCount,
                                                ref int TagNum,
                                                int FrmHandle);

        #endregion

        public enum OperResult
        {
            [Description("执行成功")] OK = 0x00,
            [Description("扫描时间结束前返回")] Error1 = 0x01,
            [Description("指定的扫描时间溢出")] Error2 = 0x02,
            [Description("本条消息之后，还有消息")] Error3 = 0x03,
            [Description("读写模块存储空间已满")] Error4 = 0x04,
            [Description("访问密码错误")] Error5 = 0x05,
            [Description("销毁密码错误")] Error9 = 0x09,
            [Description("销毁密码丌能为全 0")] Error10 = 0x0A,
            [Description("电子标签丌支持该命令")] Error11 = 0x0B,
            [Description("对亍特定命令，访问密码丌能为 0")] Error12 = 0x0C,
            [Description("电子标签已经被设置了读保护，丌能再次设置")] Error13 = 0x0D,
            [Description("电子标签没有被设置读保护，丌需要解锁")] Error14 = 0x0E,
            [Description("参数保存失败，设置的值仅在读写器断电前有效")] Error19 = 0x13,
            [Description("无法调整功率")] Error20 = 0x14,
            [Description("电子丌支持该命令或者访问密码丌能为 0")] Error25 = 0x19,
            [Description("标签自定义功能执行错误")] Error26 = 0x1A,
            [Description("返回指令错误")] Error238 = 0xEE,
            [Description("检测丌到天线连接")] Error248 = 0xF8,
            [Description("命令执行出错")] Error249 = 0xF9,
            [Description("扫描到电子标签，但由亍通信丌畅，导致操作失败")] Error250 = 0xFA,
            [Description("无电子标签可操作")] Error251 = 0xFB,
            [Description("电子标签返回错误代码")] Error252 = 0xFC,
            [Description("命令长度错误")] Error253 = 0xFD,
            [Description("命令非法")] Error254 = 0xFE,
            [Description("参数错误")] Error255 = 0xFF,
            [Description("通讯错误")] Error48 = 0x30,
            [Description("CRC 错误")] Error49 = 0x31,
            [Description("通讯繁忙，设备正在执行其他命令")] Error51 = 0x33,
            [Description("端口已打开")] Error53 = 0x35,
            [Description("端口已关闭")] Error54 = 0x36,
            [Description("无效句柄")] Error55 = 0x37,
            [Description("已打开设备过多，空间丌足")] Error57 = 0x39,

        }

        public enum ErrorCodeEnum
        {
            [Description("其它错误，全部捕捉未被其它代码覆盖的错误")] Error0 = 0x00,
            [Description("存储器超限或丌被支持的 PC 值，规定存储位置丌存在或标签丌支持 PC 值")] Error3 = 0x03,
            [Description("存储器锁定，存储位置锁定永久锁定，且丌可写入")] Error4 = 0x04,
            [Description("电源丌足，标签电源丌足，无法执行存储写入操作")] Error11 = 0x0B,
            [Description("非特定错误，标签丌支持特定错误代码")] Error15 = 0x0F,

        }

    }

    public static class EnumExtension
    {
        /// <summary>
        /// 根据值得到中文备注
        /// </summary>
        /// <param name="e"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public static string GetEnumDesc(this System.Type e, int? value)
        {
            string result;
            try
            {
                System.Reflection.FieldInfo[] fields = e.GetFields();
                int i = 1;
                int count = fields.Length;
                while (i < count)
                {
                    if ((int)System.Enum.Parse(e, fields[i].Name) == value)
                    {
                        DescriptionAttribute[] EnumAttributes = (DescriptionAttribute[])fields[i].GetCustomAttributes(typeof(DescriptionAttribute), false);
                        if (EnumAttributes.Length > 0)
                        {
                            result = EnumAttributes[0].Description;
                            return result;
                        }
                    }
                    i++;
                }
                result = "";
            }
            catch
            {
                result = "";
            }
            return result;
        }
    }
}
