﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceProcess;
using LibUsbDotNet.Main;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using JmpPrintService.Model;
using System.Threading.Tasks;
using System.Reflection;
using JmpZbModel;
using JmpPrintService.Basic;
using System.Threading;
using System.IO;
using LibUsbDotNet.DeviceNotify;
using JmpZbService.Model;
using JmpZbService.Basic;
using System.Net;
using NetworkCommsDotNet.Connections;
using NetworkCommsDotNet;
using System.Diagnostics;

namespace JmpPrintService
{
    public partial class PrintService : ServiceBase
    {
        //private Dictionary<string, IWebSocketConnection> dic_client = new Dictionary<string, IWebSocketConnection>();
        private DeviceInfo deviceInfo;
        private PrinterBase printerBase;

        public PrintService(string[] args)
        {
            InitializeComponent();

            try
            {
                var path = AppDomain.CurrentDomain.SetupInformation.ApplicationBase + "cfg.txt";
                if (File.Exists(path))
                {
                    var com = File.ReadAllLines(path)?.FirstOrDefault() ?? "";
                    if (com.ToUpper().Contains("COM"))
                    {
                        PrintCache.ComPort = com.ToUpper();
                    }
                    else { PrintCache.ComPort = "COM5"; }
                }
                else
                {
                    File.CreateText(path);
                }
                //notifier.Enabled = false;

            }
            catch (Exception ex)
            {
                Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
            }
        }

        /// <summary>
        /// 服务启动
        /// </summary>
        /// <param name="args"></param>
        protected override void OnStart(string[] args)
        {
            try
            {
                //初始化设备信息
                deviceInfo = new DeviceInfo();
                //启动网络服务
                //StartWs();
                StartTcp();

            }
            catch (Exception ex)
            {
                Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
            }
        }

        /// <summary>
        /// 打印单条任务
        /// </summary>
        /// <param name="mission"></param>
        public async void PrintSingle(SingleInfo singleInfo, Connection client)
        {
            if (printerBase == null)
            {
                //发送打印状态
                OnSendMsg(client, "RPrintLabel", "打印机未启动", false);
                return;
            }
            else if (!printerBase.SendQueryMsg())
            {
                OnSendMsg(client, "RPrintLabel", "打印机未连接", false);
            }
            else if (await GetStateInfo(PrintState.正常) != PrintState.正常)
            {
                OnSendMsg(client, "RPrintLabel", "打印机异常", false);
            }
            else
            {
                //带RFID模板
                string strWrite = @ToshibaTemplate.GetRfidPrintCmd(singleInfo, singleInfo.ModelType);
                if (string.IsNullOrEmpty(strWrite))
                {
                    //发送打印状态
                    OnSendMsg(client, "RPrintLabel", "未找到打印模板", false);
                }
                else
                {
                    printerBase.SendMsg(strWrite + "{WS|}");
                    //延时，给够足够的时间走纸；否则可能导致RFID写入失败
                    Thread.Sleep(2500);
                    var status = await GetStateInfo(PrintState.正常);
                    if (status == PrintState.正常)
                    {
                        OnSendMsg(client, "RPrintLabel", singleInfo.Info9, true);
                    }
                    else
                    {
                        OnSendMsg(client, "RPrintLabel", "标签打印失败", false);
                    }
                }
            }

        }

        /// <summary>
        /// 启动打印机
        /// </summary>
        /// <param name="client"></param>
        /// <param name="deviceInfo"></param>
        protected void StartPrint(Connection client, DeviceInfo deviceInfo)
        {
            try
            {
                if (printerBase == null && deviceInfo.PrintType == PrintType.东芝)
                {
                    printerBase = new ToshibaPrinter();
                    printerBase.IsWork = false;
                }
                if (!printerBase?.IsWork ?? false)
                {
                    if (printerBase.OnStart(deviceInfo.ConnectType))//程序启动成功
                    {
                        if (printerBase.SendQueryMsg())
                        {
                            OnSendMsg(client, "RStartPrint", "打印机启动成功", true);
                        }
                        else
                        {
                            OnSendMsg(client, "RStartPrint", "打印机启动失败", false);
                        }
                    }
                    else
                    {
                        OnSendMsg(client, "RStartPrint", "打印机启动失败", false);
                    }

                }
                else
                {
                    OnSendMsg(client, "RStartPrint", "打印机启动成功", true);
                }
            }
            catch (Exception ex)
            {
                OnSendMsg(client, "RStartPrint", "打印机启动失败", false);
                Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
            }

        }
        private async Task<string> GetTagInfo()
        {
            PrintCache.ClearTagMsg();
            DateTime startTime = DateTime.Now;
            return await Task.Run(() =>
            {

                while (true)
                {
                    if (Math.Abs(DateTime.Now.Subtract(startTime).TotalSeconds) > 5)
                    {
                        return string.Empty;
                    }
                    else if (PrintCache.EpcMsgCount() > 0)
                    {
                        var msg = PrintCache.DeTagMsg();
                        if (!string.IsNullOrEmpty(msg))
                        {
                            return msg;
                        }
                        else
                        {
                            continue;
                        }
                    }
                    else { continue; }
                }

            });

        }
        private async Task<PrintState> GetStateInfo(PrintState state = PrintState.异常)
        {
            DateTime startTime = DateTime.Now;
            return await Task.Run(() =>
            {

                while (true)
                {
                    if (Math.Abs(DateTime.Now.Subtract(startTime).Milliseconds) > 2000 || printerBase.GetPrintState() == state)
                    {
                        return printerBase.GetPrintState();
                    }
                    else { continue; }
                }

            });

        }

        /// <summary>
        /// 校验TID，EPC是否读取完整(禁用)
        /// </summary>
        /// <param name="tid"></param>
        /// <param name="epc"></param>
        /// <returns></returns>
        private bool IsGood(string tid, string epc)
        {
            return true;
            //if (string.IsNullOrEmpty(tid) || string.IsNullOrEmpty(epc))
            //{
            //    return true;
            //}
            //else
            //{
            //    if (tid.Length > 12)
            //    {
            //        if (epc.Length < 2)
            //        {
            //            epc = "FF";
            //        }
            //        TAG_INFO tag = new TAG_INFO();
            //        tag.TAG_TID = tid;
            //        JmpTagInit.JmpPrintEncrypt("02", ref tag);
            //        tag.TAG_EPC = tag.TAG_EPC.PadRight(epc.Length, '0').ToUpper();
            //        if (tag.TAG_EPC.Equals(epc))
            //        {
            //            return true;
            //        }
            //        else
            //        {
            //            return false;
            //        }

            //    }
            //    else
            //    {
            //        return false;
            //    }
            //}
        }

        public void TestStartupAndStop(string[] args)
        {
            this.OnStart(args);
            Console.ReadKey();
            this.OnStop();
        }

        /// <summary>
        /// 停止服务
        /// </summary>
        protected override void OnStop()
        {
            try
            {
                NetworkComms.CloseAllConnections();
                Connection.StopListening();
                printerBase?.OnStop();
                NetworkComms.Shutdown();
                Process.GetCurrentProcess().Kill();
            }
            catch (Exception ex)
            {
                Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
            }
        }

        /// <summary>
        /// 关闭服务
        /// </summary>
        protected override void OnShutdown()
        {
            try
            {
                NetworkComms.CloseAllConnections();
                Connection.StopListening();
                printerBase?.OnStop();
                NetworkComms.Shutdown();
                Process.GetCurrentProcess().Kill();
            }
            catch (Exception ex)
            {
                Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
            }
        }


        #region Tcp服务相关
        protected void StartTcp(int port = 8656)
        {
            //IP地址和端口
            IPEndPoint thePoint = new IPEndPoint(IPAddress.Parse("127.0.0.1"), port);
            //开始监听此IP和端口  使用TCP协议
            Connection.StartListening(ConnectionType.TCP, thePoint);
            //打印标签
            NetworkComms.AppendGlobalIncomingPacketHandler<string>("PrintLabel", OnPrintLabel);
            //
            NetworkComms.AppendGlobalIncomingPacketHandler<string>("StartPrint", OnStartPrint);
            //
            NetworkComms.AppendGlobalIncomingPacketHandler<string>("PrintNotice", OnPrintNotice);
            //客户端连接
            NetworkComms.AppendGlobalConnectionEstablishHandler(ClientLinkSuc);
            //客户端断开连接
            NetworkComms.AppendGlobalConnectionCloseHandler(ClientDisLink);
        }

        /// <summary>
        /// 打印标签
        /// </summary>
        /// <param name="srvUrl"></param>
        /// <param name="cmd"></param>
        private void OnPrintLabel(PacketHeader header, Connection connection, string message)
        {
            try
            {
                if (string.IsNullOrEmpty(message))
                {
                    SendCmdError(connection, "RPrintLabel");
                    return;
                }
                else
                {
                    JObject jObj = JObject.Parse(message);
                    if (!jObj.HasValues)
                    {
                        SendCmdError(connection, "RPrintLabel");
                        return;
                    }
                    else
                    {
                        PrintCmdInfo cmdInfo = JsonConvert.DeserializeObject<PrintCmdInfo>(message);
                        if (cmdInfo.CmdType == PrintCmdType.printMsg)
                        {
                            var singleInfo = JsonConvert.DeserializeObject<SingleInfo>(cmdInfo.Message);
                            PrintSingle(singleInfo, connection);
                        }
                        else
                        {
                            SendCmdError(connection, "RPrintLabel");
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                SendCmdError(connection, "RPrintLabel");
                Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
            }
        }

        private void OnStartPrint(PacketHeader header, Connection connection, string message)
        {
            PrintCmdInfo cmdInfo = JsonConvert.DeserializeObject<PrintCmdInfo>(message);
            if (cmdInfo.CmdType == PrintCmdType.startPrint)
            {

                deviceInfo = JsonConvert.DeserializeObject<DeviceInfo>(cmdInfo.Message);
                StartPrint(connection, deviceInfo);
            }
            else
            {
                SendCmdError(connection, "RStartPrint");
            }
        }

        private void OnPrintNotice(PacketHeader header, Connection connection, string json)
        {
        }

        /// <summary>
        /// 客户端连接成功
        /// </summary>
        /// <param name="connection"></param>
        private void ClientLinkSuc(Connection connection)
        {
            EndPoint p = connection.ConnectionInfo.RemoteEndPoint;
            //Console.WriteLine($"客户端连接{p.ToString()}");
        }

        /// <summary>
        /// 客户端断开连接
        /// </summary>
        /// <param name="connection"></param>
        private void ClientDisLink(Connection connection)
        {
            EndPoint p = connection.ConnectionInfo.RemoteEndPoint;
            printerBase?.OnStop();
        }
        private void SendCmdError(Connection connection, string route)
        {
            var json = JsonConvert.SerializeObject(new ResponeMsg
            {
                Success = false,
                Message = "不被允许的指令"
            });
            if (connection != null &&
                (connection.ConnectionInfo.ConnectionState == ConnectionState.Established
                || connection.ConnectionInfo.ConnectionState == ConnectionState.Establishing))
            {
                connection.SendObject(route, json);
            }
        }
        protected void OnSendMsg(Connection client, string route, string msg, bool flag)
        {
            try
            {
                if (client != null &&
                    client.ConnectionInfo.ConnectionState == ConnectionState.Established ||
                    client.ConnectionInfo.ConnectionState == ConnectionState.Establishing)
                {
                    {
                        var cmdInfo = new
                        {
                            Success = flag,
                            Message = msg
                        };
                        client.SendObject(route, JsonConvert.SerializeObject(cmdInfo));
                    }
                }
            }
            catch (Exception ex)
            {
                Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
            }
        }

        #endregion

        #region Websocket服务相关 （未启用）
        ///// <summary>
        ///// 启动Ws服务
        ///// </summary>
        ///// <param name="strPort"></param>
        //protected void StartWs(string strPort = "8089")
        //{
        //    var server = new WebSocketServer($"ws://0.0.0.0:{strPort}");
        //    server.Start(socket =>
        //    {
        //        socket.OnOpen = () =>
        //        {
        //            OnNewConnection($"{socket.ConnectionInfo.ClientIpAddress}:{socket.ConnectionInfo.ClientPort}", socket);
        //        };
        //        socket.OnClose = () =>
        //        {
        //            OnDisConnection($"{socket.ConnectionInfo.ClientIpAddress}:{socket.ConnectionInfo.ClientPort}");
        //        };
        //        socket.OnMessage = message =>
        //        {
        //            Task.Run(() =>
        //            {
        //                OnReciveData($"{socket.ConnectionInfo.ClientIpAddress}:{socket.ConnectionInfo.ClientPort}", message);
        //            });
        //        };
        //    });
        //}

        ///// <summary>
        /////客户端连入
        ///// </summary>
        ///// <param name="client"></param>
        ///// <param name="connectionInfo"></param>
        //protected void OnNewConnection(string client, IWebSocketConnection connectionInfo)
        //{
        //    if (dic_client.Count == 0)
        //    {
        //        dic_client.Add(client, connectionInfo);
        //    }
        //    else
        //    {
        //        connectionInfo.Send(JsonConvert.SerializeObject(new CmdInfo() { CmdType = CmdType.printUsed, Message = "打印机被占用，连接关闭", Success = false }));
        //        connectionInfo.Close();
        //    }

        //}

        ///// <summary>
        ///// 客户端断开
        ///// </summary>
        ///// <param name="client"></param>
        //protected void OnDisConnection(string client)
        //{
        //    if (dic_client.ContainsKey(client))
        //    {
        //        dic_client.Remove(client);

        //    }
        //    printerBase?.OnStop();
        //}

        ///// <summary>
        ///// 接受客户端数据
        ///// </summary>
        ///// <param name="client"></param>
        ///// <param name="message"></param>
        //protected void OnReciveData(string client, string message)
        //{
        //    try
        //    {
        //        if (string.IsNullOrEmpty(message)) { return; }
        //        JObject jObj = JObject.Parse(message);
        //        if (!jObj.HasValues) { return; }

        //        CmdInfo cmdInfo = JsonConvert.DeserializeObject<CmdInfo>(message);
        //        switch (cmdInfo.CmdType)
        //        {

        //            case CmdType.devQuery://查询打印机状态
        //                if (printerBase == null && !printerBase.IsWork)
        //                {
        //                    //发送打印机未启动
        //                    OnSendMsg(client, CmdType.devQuery, "打印机未启动", false);
        //                }
        //                else
        //                {
        //                    QueryPrint(client);
        //                }

        //                break;
        //            case CmdType.printMsg://打印单条数据
        //                var singleInfo = JsonConvert.DeserializeObject<SingleInfo>(cmdInfo.Message);
        //                PrintSingle(singleInfo, client);
        //                //打印结束，释放缓存
        //                //printerBase.ClearFlush();
        //                break;
        //            case CmdType.startPrint://启动打印机
        //                deviceInfo = JsonConvert.DeserializeObject<DeviceInfo>(cmdInfo.Message);
        //                StartPrint(client, deviceInfo);
        //                break;
        //            default:
        //                break;

        //        }

        //    }
        //    catch (Exception ex)
        //    {
        //        Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
        //    }

        //}

        //protected void OnSendMsg(string client, CmdType cmdType, string msg, bool flag)
        //{
        //    try
        //    {
        //        if (dic_client.ContainsKey(client))
        //        {
        //            var ws_connection = dic_client[client];
        //            if (ws_connection != null)
        //            {
        //                CmdInfo cmdInfo = new CmdInfo()
        //                {
        //                    CmdType = cmdType,
        //                    Message = msg,
        //                    Success = flag
        //                };

        //                ws_connection.Send(JsonConvert.SerializeObject(cmdInfo));
        //            }
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        Log.ErrorLog(ex.ToString(), MethodBase.GetCurrentMethod().Name);
        //    }
        //}
        #endregion
    }
}
