﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Data;
using JmpZbModel;
using JmpZbPrint.Model;
using NetworkCommsDotNet;
using NetworkCommsDotNet.Connections;
using NetworkCommsDotNet.Connections.TCP;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace JmpZbPrint.Common
{
    public interface IPrintServerHelper : IDisposable
    {
        delegate void EnableControlDelegate(bool flag, bool startEnable, bool stopEnable, string msg = "");
        abstract event EnableControlDelegate EnableControlEvent;
        void Init();
        bool SendPrintCmd(SingleInfo entity);
    }

    /// <summary>
    /// 东芝打印机
    /// </summary>
    public class ToShiBaPrintServerHelper : IPrintServerHelper
    {
        public event IPrintServerHelper.EnableControlDelegate EnableControlEvent;

        private CancellationTokenSource cTcpServiceToken;
        private Task taskTcpService;
        private Connection tcpConn;
        private bool isPrintError = true;
        private bool flushPrintState = false;

        private AutoResetEvent resetEvent = new AutoResetEvent(false);

        /// <summary>
        /// 初始化
        /// </summary>
        public void Init()
        {
            if (taskTcpService == null)
            {
                cTcpServiceToken = new CancellationTokenSource();
                taskTcpService = new Task(() =>
                {
                    while (!cTcpServiceToken.IsCancellationRequested)
                    {
                        try
                        {
                            // 已连接的不处理
                            if (tcpConn?.ConnectionInfo?.ConnectionState == ConnectionState.Established
                                    || tcpConn?.ConnectionInfo?.ConnectionState == ConnectionState.Establishing)
                            {
                                if (isPrintError)
                                {
                                    try
                                    {
                                        if (!tcpConn?.ConnectionAlive() ?? false
                                            || tcpConn?.ConnectionInfo?.ConnectionState == ConnectionState.Shutdown
                                            || tcpConn?.ConnectionInfo?.ConnectionState == ConnectionState.Undefined)
                                        {
                                            return;
                                        }
                                        else
                                        {
                                            PrintCmdInfo cmdInfo = new PrintCmdInfo()
                                            {
                                                CmdType = PrintCmdType.startPrint,
                                                Message = JsonConvert.SerializeObject(new DeviceInfo() { ConnectType = ConnectType.USB, PrintType = PrintType.东芝 }),
                                                Success = false,
                                            };
                                            string msg = tcpConn?.SendReceiveObject<string, string>("StartPrint", "RStartPrint", 60000, JsonConvert.SerializeObject(cmdInfo));
                                            JObject obj = JObject.Parse(msg ?? string.Empty);
                                            if (obj["Success"]?.ToObject<bool>() == false)
                                            {
                                                isPrintError = true;
                                                EnableControlEvent?.Invoke(false, false, false, "打印机启动失败");
                                            }
                                            else
                                            {
                                                isPrintError = false;
                                                EnableControlEvent?.Invoke(true, true, false, "打印机启动成功");
                                            }
                                        }
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.ErrorLog(ex.ToString(), this.GetType().Name);
                                    }
                                }
                                else
                                {
                                    if (flushPrintState)
                                    {
                                        flushPrintState = false;
                                        EnableControlEvent?.Invoke(true, true, false, "打印机启动成功");
                                    }
                                }
                            }
                            else
                            {
                                try
                                {
                                    tcpConn = TCPConnection.GetConnection(new ConnectionInfo("127.0.0.1", 8656));
                                    tcpConn.AppendShutdownHandler(new NetworkComms.ConnectionEstablishShutdownDelegate(OnDisConnection));
                                    EnableControlEvent?.Invoke(true, true, false, "打印服务已启动");
                                    //StartPrint();
                                }
                                catch
                                {
                                    EnableControlEvent?.Invoke(false, false, false, "打印机连接失败");
                                }
                            }
                            if (resetEvent.WaitOne(3000)) { continue; }
                        }
                        catch (System.Exception ex)
                        {
                            EnableControlEvent?.Invoke(false, false, false, "打印机连接失败");
                            Log.ErrorLog(ex.ToString(), this.GetType().Name);
                        }
                    }
                });
                taskTcpService.Start();
            }
            else
            {
                //刷新状态
                flushPrintState = true;
                resetEvent.Set();
            }
        }

        /// <summary>
        /// 打印
        /// </summary>
        /// <param name="entity"></param>
        /// <returns></returns>
        public bool SendPrintCmd(SingleInfo entity)
        {
            try
            {
                if (!tcpConn?.ConnectionAlive() ?? false
                    || tcpConn?.ConnectionInfo?.ConnectionState == ConnectionState.Shutdown
                    || tcpConn?.ConnectionInfo?.ConnectionState == ConnectionState.Undefined)
                {
                    return false;
                }
                else
                {
                    PrintCmdInfo cmdInfo = new PrintCmdInfo()
                    {
                        CmdType = PrintCmdType.printMsg,
                        Message = JsonConvert.SerializeObject(entity),
                        Success = false,
                    };
                    var json = JsonConvert.SerializeObject(cmdInfo);
                    string msg = tcpConn?.SendReceiveObject<string, string>("PrintLabel", "RPrintLabel", 60000, json);
                    JObject obj = JObject.Parse(msg ?? string.Empty);
                    if (obj["Success"]?.ToObject<bool>() == true)
                    {
                        EnableControlEvent?.Invoke(true, false, true, "数据打印成功");
                        return true;
                    }
                    else
                    {
                        var err_msg = obj["Message"]?.ToString() ?? "";
                        if (!string.IsNullOrEmpty(err_msg) && (err_msg.Contains("未连接") || err_msg.Contains("未启动")))
                        {
                            isPrintError = true;
                        }
                        EnableControlEvent?.Invoke(false, false, false, err_msg);
                        return false;
                    }
                }

            }
            catch (Exception ex)
            {
                Log.ErrorLog(ex.ToString(), this.GetType().Name);
                return false;
            }
        }

        /// <summary>
        /// 失去连接，重连
        /// </summary>
        private void OnDisConnection(Connection connection)
        {
            EnableControlEvent?.Invoke(false, false, false);
        }

        public void Dispose()
        {
            resetEvent.Set();
            cTcpServiceToken?.Cancel();
            tcpConn?.CloseConnection(false);
        }
    }

    /// <summary>
    /// 博思得打印机
    /// </summary>
    public class PostekPrintServerHelper : IPrintServerHelper
    {
        public event IPrintServerHelper.EnableControlDelegate EnableControlEvent;

        private CancellationTokenSource cTcpServiceToken;
        private Task taskTcpService;
        private Connection tcpConn;
        private bool isPrintError = true;
        private bool flushPrintState = false;

        private AutoResetEvent resetEvent = new AutoResetEvent(false);

        /// <summary>
        /// 初始化
        /// </summary>
        public void Init()
        {
            if (taskTcpService == null)
            {
                cTcpServiceToken = new CancellationTokenSource();
                taskTcpService = new Task(() =>
                {
                    while (!cTcpServiceToken.IsCancellationRequested)
                    {
                        try
                        {
                            if (isPrintError)
                            {
                                try
                                {
                                    //重连
                                    var json = HttpHelper.PostUrlencoded("http://127.0.0.1:888/postek/print", new List<KeyValuePair<string, string>>()
                                    {
                                        new KeyValuePair<string, string>("reqParam", "1"),
                                        new KeyValuePair<string, string>("printparams", "[{\"PTK_OpenUSBPort\":\"255\"},{\"PTK_CloseUSBPort\":\"\"}]"),
                                    });
                                    if (string.IsNullOrEmpty(json))
                                    {
                                        EnableControlEvent?.Invoke(false, false, false, "打印服务连接失败，重连中");
                                    }
                                    else
                                    {
                                        var res_data = JsonConvert.DeserializeObject<respone_postek_msg>(json);
                                        if (res_data == null || res_data.retval != "0")
                                        {
                                            EnableControlEvent?.Invoke(false, false, false, "打印机连接失败");
                                        }
                                        else
                                        {
                                            isPrintError = false;
                                            EnableControlEvent?.Invoke(true, true, false, "打印机启动成功");
                                        }
                                    }
                                }
                                catch (Exception ex)
                                {
                                    Log.ErrorLog(ex.ToString(), this.GetType().Name);
                                }
                            }
                            else
                            {
                                if (flushPrintState)
                                {
                                    flushPrintState = false;
                                    EnableControlEvent?.Invoke(true, true, false, "打印机启动成功");
                                }
                            }
                            if (resetEvent.WaitOne(3000)) { continue; }
                        }
                        catch (System.Exception ex)
                        {
                            EnableControlEvent?.Invoke(false, false, false, "打印机连接失败");
                            Log.ErrorLog(ex.ToString(), this.GetType().Name);
                        }
                    }
                });
                taskTcpService.Start();
            }
            else
            {
                //刷新状态
                flushPrintState = true;
                resetEvent.Set();
            }
        }

        /// <summary>
        /// 打印
        /// </summary>
        /// <param name="entity"></param>
        /// <returns></returns>
        public bool SendPrintCmd(SingleInfo entity)
        {
            try
            {
                var strTemplate = (string)(typeof(PostekPrintServerHelper).GetField(entity.ModelType + "Template")?.GetValue(this));
                if (string.IsNullOrEmpty(strTemplate))
                {
                    EnableControlEvent?.Invoke(false, false, false, "无效模板");
                }
                var property = typeof(SingleInfo);
                foreach (var property_item in property.GetProperties())
                {
                    var property_value = property_item.GetValue(entity)?.ToString() ?? "";
                    strTemplate = strTemplate.Replace("{" + property_item.Name + "}", property_value);
                }
                //发送指令
                var pushData = @"[{""PTK_OpenUSBPort"":""255""},
{""PTK_ClearBuffer"":""""},
{""PTK_SetDirection"":""B""},
{""PTK_SetPrintSpeed"":""4""},
{""PTK_SetDarkness"":""10""},
{""PTK_SetLabelHeight"":""600,84,0,false""},
{""PTK_SetLabelWidth"":1284},
" + strTemplate + @"
{""PTK_PrintLabel"":""1,1""},
{""PTK_CloseUSBPort"":""""}]
";
                var json = HttpHelper.PostUrlencoded("http://127.0.0.1:888/postek/print", new List<KeyValuePair<string, string>>()
                {
                    new KeyValuePair<string, string>("reqParam", "1"),
                    new KeyValuePair<string, string>("printparams", pushData),
                });

                if (string.IsNullOrEmpty(json))
                {
                    isPrintError = true;
                    EnableControlEvent?.Invoke(false, false, false, "打印服务连接失败，重连中");
                    return false;
                }
                else
                {
                    var res_data = JsonConvert.DeserializeObject<respone_postek_msg>(json);
                    if (res_data == null || res_data.retval != "0")
                    {
                        Log.ErrorLog("数据打印失败:" + res_data?.msg, this.GetType().Name);
                        if (res_data?.msg.Contains("打开USB端口文件失败") == true || res_data?.msg.Contains("没有打开端口") == true)
                        {
                            isPrintError = true;
                            EnableControlEvent?.Invoke(false, false, false, "打印服务连接失败，重连中");
                        }
                        else
                        {
                            EnableControlEvent?.Invoke(true, false, true, "数据打印失败");
                        }

                        return false;
                    }
                    else
                    {
                        EnableControlEvent?.Invoke(true, false, true, "数据打印成功");
                        return true;
                    }
                }

            }
            catch (Exception ex)
            {
                Log.ErrorLog(ex.ToString(), this.GetType().Name);
                return false;
            }
        }

        public void Dispose()
        {
            resetEvent.Set();
            cTcpServiceToken?.Cancel();
            tcpConn?.CloseConnection(false);
        }

        #region template

        public const string pieceTemplate = @"
{""PTK_DrawText_TrueTypeEx"":""214,120,54,0,黑体,1,550,0,0,0,0,0,0,0,装备名称：{Info1}""},
{""PTK_DrawText_TrueTypeEx"":""214,190,54,0,黑体,1,550,0,0,0,0,0,0,0,装备型号：{Info2}""},
{""PTK_DrawText_TrueTypeEx"":""214,260,54,0,黑体,1,550,0,0,0,0,0,0,0,件序号：{Info4}""},
{""PTK_DrawText_TrueTypeEx"":""214,330,54,0,黑体,1,550,0,0,0,0,0,0,0,生产日期：{Info5}""},
{""PTK_DrawText_TrueTypeEx"":""214,400,54,0,黑体,1,550,0,0,0,0,0,0,0,供应商：{Info3}""},
{""PTK_RWRFIDLabel"":""1,0,2,24,1,{Info9}""},
";
        public const string pieceXBTemplate = @"
{""PTK_DrawText_TrueTypeEx"":""214,120,54,0,黑体,1,550,0,0,0,0,0,0,0,装备名称：{Info1}""},
{""PTK_DrawText_TrueTypeEx"":""214,190,54,0,黑体,1,550,0,0,0,0,0,0,0,装备型号：{Info2}""},
{""PTK_DrawText_TrueTypeEx"":""214,260,54,0,黑体,1,550,0,0,0,0,0,0,0,箱序号：{Info5} 箱内数量：{Info4}""},
{""PTK_DrawText_TrueTypeEx"":""214,330,54,0,黑体,1,550,0,0,0,0,0,0,0,打印日期：{Info6}""},
{""PTK_DrawText_TrueTypeEx"":""214,400,54,0,黑体,1,550,0,0,0,0,0,0,0,供应商：{Info3}""},
{""PTK_DrawText_TrueTypeEx"":""980,84,100,0,黑体,1,550,0,0,0,0,0,0,0,箱""},
{""PTK_RWRFIDLabel"":""1,0,2,24,1,{Info9}""},
";
        public const string pieceBagTemplate = @"
{""PTK_DrawText_TrueTypeEx"":""214,190,54,0,黑体,1,550,0,0,0,0,0,0,0,装备名称：{Info1}""},
{""PTK_DrawText_TrueTypeEx"":""214,260,54,0,黑体,1,550,0,0,0,0,0,0,0,编号：{Info5}""},
{""PTK_DrawText_TrueTypeEx"":""214,330,54,0,黑体,1,550,0,0,0,0,0,0,0,打印日期：{Info6}""},
{""PTK_DrawText_TrueTypeEx"":""980,84,100,0,黑体,1,550,0,0,0,0,0,0,0,包""},
{""PTK_RWRFIDLabel"":""1,0,2,24,1,{Info9}""},
";
        #endregion
    }
}
