﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;
using JmpCommon;
using JmpCommon.Tools;
using JmpModel.Model;
using JmpModel.Model.Enumerates;
using JmpModel.Model.Repository;
using JmpModel.Print;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.EntityFrameworkCore.Metadata.Internal;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json.Linq;

namespace JunmpPoliceStation.Controllers
{
    [ApiController]
    [Route("api/[controller]")]
    public class NoticeController : ControllerBase
    {
        private ILogger<NoticeController> _logger;
        UnitOfWork _unitOfWork;
        public NoticeController(ILogger<NoticeController> logger, UnitOfWork unitOfWork)
        {
            _logger = logger;
            //_CabinetRepository = CabinetRepository;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取报废通知
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        ///        ## 例子
        ///        {
        ///         "orgId": "",
        ///         "page":0,
        ///         "size":10,
        ///         "days":7,
        ///         "scrap":true,
        ///         "position":"全部 仓库 库外 单警柜",
        ///         }
        ///   </remarks>   
        [HttpPost("GetScrapMsg")]
        public async Task<HttpResponseMessage> GetScrapMsg([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        string orgId = entity?.orgId ?? "";
                        string id = entity?.id ?? "";
                        bool bDays = int.TryParse(entity?.days ?? "0", out int days);
                        DateTime startTime = default;
                        DateTime endTime = default;
                        bool.TryParse(entity?.scrap ?? "false", out bool scrap);
                        if (string.IsNullOrEmpty(orgId))
                        {
                            return JsonManager.SimpleCustResponse("orgId is require");
                        }
                        else if (!bDays || days <= 0)
                        {
                            return JsonManager.SimpleCustResponse("days is require");
                        }
                        else
                        {
                            Expression<Func<VScrapView, bool>> expression = t => t.OrgId.Contains(orgId);

                            if (size == 0)
                            {
                                size = 10;
                            }

                            if (string.IsNullOrEmpty(id))
                            {
                                if (scrap == true)
                                {
                                    startTime = DateTime.Now.Date.AddDays(-days).AddSeconds(-1);
                                    endTime = DateTime.Now;
                                    expression = LambdaExtensions.AndAlso(expression, t => t.IsScrap.Contains("true"));
                                }
                                else
                                {
                                    startTime = DateTime.Now;
                                    endTime = DateTime.Now.Date.AddDays(days).AddSeconds(-1);
                                    expression = LambdaExtensions.AndAlso(expression, t => (t.WarrantyTime >= startTime && t.WarrantyTime <= endTime) && t.IsScrap.Contains("false"));
                                }

                                if (!string.IsNullOrEmpty(entity.position))
                                {
                                    string position = entity.position;
                                    switch (position)
                                    {
                                        case "全部":
                                            {
                                                break;
                                            }
                                        case "仓库":
                                            {
                                                expression = expression.AndAlso(t => t.CurrentState == 0);
                                                break;
                                            }
                                        case "库外":
                                            {
                                                expression = expression.AndAlso(t => t.CurrentState != 0 && t.CurrentState != 3);
                                                break;
                                            }
                                        case "单警柜":
                                            {
                                                expression = expression.AndAlso(t => t.CurrentState == 3);
                                                break;
                                            }
                                    }
                                }

                                var scrapList = _unitOfWork.ScrapViewRepository.GetPage(expression, "WarrantyTime", page, size);
                                var data = from s in scrapList?.content
                                           select new
                                           {
                                               id = s.Id,
                                               epc = s.Epc,
                                               sizeId = s.SizeId,
                                               sizeName = s.SizeName,
                                               supplierId = s.SupplierId,
                                               supplierName = s.SupplierName,
                                               equName = s.EquName,
                                               equId = s.EquId,
                                               warehouseName = s.WarehouseName,

                                               shelfId = s.ShelfId,
                                               shelfCode = s.ShelfCode,
                                               shelfName = s.ShelfName,
                                               shelfRange = s.ShelfRange,
                                               shelfRow = s.ShelfRow,
                                               shelfColumn = s.ShelfColumn,

                                               WarrantyTime = s.WarrantyTime == default ? null : s.WarrantyTime.Value.ToString("yyyy-MM-dd")
                                           };
                                var content = new
                                {
                                    scrapList.totalElements,
                                    content = data
                                };
                                return JsonManager.ReturnSuccessResponse(content);

                            }
                            else
                            {
                                var scrapList = _unitOfWork.ScrapViewRepository.GetPage(p => p.Id.Contains(id), "WarrantyTime", page, size, false);
                                var data = from s in scrapList?.content
                                           select new
                                           {
                                               id = s.Id,
                                               epc = s.Epc,
                                               sizeName = s.SizeName,
                                               supplierName = s.SupplierName,
                                               equName = s.EquName,
                                               warehouseName = s.WarehouseName,
                                               WarrantyTime = s.WarrantyTime == default ? null : s.WarrantyTime.Value.ToString("yyyy-MM-dd")
                                           };
                                var content = new
                                {
                                    scrapList.totalElements,
                                    content = data
                                };
                                return JsonManager.ReturnSuccessResponse(content);
                            }
                        }

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    return JsonManager.SimpleCustResponse(ex.ToString());
                }
            });
        }

        /// <summary>
        /// 获取报废通知(树状）
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        ///        ## 例子
        ///        {
        ///         "orgId": "",
        ///         "page":0,
        ///         "size":10,
        ///         "days":7,
        ///         "scrap":true,
        ///         "position":"全部 仓库 库外 单警柜",
        ///         }
        ///   </remarks>   
        [HttpPost("GetTreeScrapMsg")]
        public async Task<HttpResponseMessage> GetTreeScrapMsg([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        string orgId = entity?.orgId ?? "";
                        string id = entity?.id ?? "";
                        bool bDays = int.TryParse(entity?.days ?? "0", out int days);
                        DateTime startTime = default;
                        DateTime endTime = default;
                        bool.TryParse(entity?.scrap ?? "false", out bool scrap);
                        if (string.IsNullOrEmpty(orgId))
                        {
                            return JsonManager.SimpleCustResponse("orgId is require");
                        }
                        else if (!bDays || days <= 0)
                        {
                            return JsonManager.SimpleCustResponse("days is require");
                        }
                        else
                        {
                            Expression<Func<VScrapView, bool>> expression = t => t.OrgId.Contains(orgId);

                            if (size == 0)
                            {
                                size = 10;
                            }
                            if (scrap == true)
                            {
                                startTime = DateTime.Now.Date.AddDays(-days).AddSeconds(-1);
                                endTime = DateTime.Now;
                                expression = LambdaExtensions.AndAlso(expression, t => t.IsScrap.Contains("true"));
                            }
                            else
                            {
                                startTime = DateTime.Now;
                                endTime = DateTime.Now.Date.AddDays(days).AddSeconds(-1);
                                expression = LambdaExtensions.AndAlso(expression, t => (t.WarrantyTime >= startTime && t.WarrantyTime <= endTime) && t.IsScrap.Contains("false"));
                            }

                            if (!string.IsNullOrEmpty(entity.position))
                            {
                                string position = entity.position;
                                switch (position)
                                {
                                    case "全部":
                                        {
                                            break;
                                        }
                                    case "仓库":
                                        {
                                            expression = expression.AndAlso(t => t.CurrentState == 0);
                                            break;
                                        }
                                    case "库外":
                                        {
                                            expression = expression.AndAlso(t => t.CurrentState != 0 && t.CurrentState != 3);
                                            break;
                                        }
                                    case "单警柜":
                                        {
                                            expression = expression.AndAlso(t => t.CurrentState == 3);
                                            break;
                                        }
                                }
                            }

                            var scrapList = _unitOfWork.ScrapViewRepository.GetPage(expression, "WarrantyTime", 0, 999999);
                            var data = scrapList.content
                                .GroupBy(x => new { x.WarehouseName, x.EquName, x.SizeName })
                                .Select(s => new
                                {
                                    warehouseName = s.Key.WarehouseName,
                                    equName = s.Key.EquName,
                                    sizeName = s.Key.SizeName,
                                    count = s.Count(),
                                    children = s.Select(y => new
                                    {
                                        id = y.Id,
                                        epc = y.Epc,
                                        sizeId = y.SizeId,
                                        sizeName = y.SizeName,
                                        supplierId = y.SupplierId,
                                        supplierName = y.SupplierName,
                                        equName = y.EquName,
                                        warehouseName = y.WarehouseName,
                                        shelfId = y.ShelfId,
                                        shelfCode = y.ShelfCode,
                                        shelfName = y.ShelfName,
                                        range = y.ShelfRange,
                                        row = y.ShelfRow,
                                        column = y.ShelfColumn,
                                        WarrantyTime = y.WarrantyTime?.ToString("yyyy-MM-dd")
                                    })
                                });
                            var content = new
                            {
                                totalElements = data.Count(),
                                content = data.Skip(page * size).Take(size).ToList()
                            };
                            return JsonManager.ReturnSuccessResponse(content);

                        }

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    return JsonManager.SimpleCustResponse(ex.ToString());
                }
            });
        }

        /// <summary>
        /// 获取借用超期明细
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        ///        ## 例子
        ///        {
        ///         "id": "",
        ///         }
        ///   </remarks>   
        [HttpPost("GetOverdueMsgDetail")]
        public async Task<HttpResponseMessage> GetOverdueMsgDetail([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        string id = entity?.id ?? "";
                        string borrowType = entity?.borrowType ?? "";
                        if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse("id is require");
                        }
                        else if (string.IsNullOrEmpty(borrowType))
                        {
                            return JsonManager.SimpleCustResponse("borrowType is require");
                        }
                        else
                        {
                            if (borrowType == "本库借用")
                            {
                                var orderInfo = _unitOfWork.DbContext.CommonJpBorrowReturnDetailRealities
                                    .Include(t => t.EquipmentDetail)
                                    .Include(t => t.Estimate)
                                    .Include(t => t.Size)
                                    .Include(t => t.Supplier)
                                    .Where(t => t.EstimateId.Contains(id)
                                    && t.RealCount > 0
                                    && t.Estimate.ProcessCurrent.FlowState.Equals("true")
                                    && t.State == 1 && t.Reality.ActionType == 0);
                                if (orderInfo != null)
                                {
                                    var data = from s in orderInfo
                                               select new
                                               {
                                                   id = s.Id,
                                                   sizeName = s.Size.SizeName,
                                                   supplierName = s.Supplier.Name,
                                                   equName = s.EquipmentDetail.Name,
                                                   warehouseName = s.Estimate.Warehouse.Name,
                                                   quantity = s.RealCount - s.InsideRealCount,
                                                   total = s.RealCount
                                               };
                                    return JsonManager.ReturnSuccessResponse(data);
                                }
                                else
                                {
                                    return JsonManager.ReturnSuccessResponse(new { });
                                }
                            }
                            else if (borrowType == "跨库借用")
                            {
                                var orderInfo = _unitOfWork.DbContext.CommonJpTransferDetailRealities
                                    .Include(t => t.EquipmentDetail)
                                    .Include(t => t.Estimate)
                                    .Include(t => t.Size)
                                    .Include(t => t.Supplier)
                                    .Where(t => t.EstimateId.Contains(id)
                                    && t.RealCount > 0
                                    && t.Estimate.ProcessCurrent.FlowState.Equals("true")
                                    && t.State != 2 && t.Reality.ActionType == 1);

                                if (orderInfo != null)
                                {
                                    var data = from s in orderInfo
                                               select new
                                               {
                                                   id = s.Id,
                                                   sizeName = s.Size.SizeName,
                                                   supplierName = s.Supplier.Name,
                                                   equName = s.EquipmentDetail.Name,
                                                   warehouseName = s.Estimate.TransferWarehouseNavigation.Name,
                                                   quantity = s.RealCount - s.InsideRealCount ?? 0,
                                                   total = s.RealCount
                                               };
                                    return JsonManager.ReturnSuccessResponse(data);
                                }
                                else
                                {
                                    return JsonManager.ReturnSuccessResponse(new { });
                                }
                            }
                            else
                            {
                                return JsonManager.ReturnSuccessResponse(new { });
                            }

                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    return JsonManager.SimpleCustResponse(ex.ToString());
                }
            });
        }

        /// <summary>
        /// 获取超期未归还通知
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        ///        ## 例子
        ///        {
        ///         "orgId": "",
        ///         "page":0,
        ///         "size":10,
        ///         "id"://可为空
        ///         }
        ///   </remarks>   
        [HttpPost("GetOverdueMsg")]
        public async Task<HttpResponseMessage> GetOverdueMsg([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        string orgId = entity?.orgId ?? "";
                        bool bDays = int.TryParse(entity?.days ?? "0", out int days);
                        bool.TryParse(entity?.overdue ?? "false", out bool overdue);
                        DateTime startTime = default;
                        DateTime endTime = DateTime.Now.Date.AddDays(365).AddSeconds(-1);
                        if (string.IsNullOrEmpty(orgId))
                        {
                            return JsonManager.SimpleCustResponse("orgId is require");
                        }
                        else if (!bDays || days <= 0)
                        {
                            return JsonManager.SimpleCustResponse("days is require");
                        }
                        else
                        {
                            if (size == 0)
                            {
                                size = 10;
                            }

                            string id = entity?.id ?? "";
                            if (string.IsNullOrEmpty(id))
                            {
                                Expression<Func<VBorrowView, bool>> expression = t => t.OrgId.Contains(orgId);
                                if (overdue == true)
                                {
                                    startTime = DateTime.Now.Date.AddDays(-days).AddSeconds(-1);
                                    endTime = DateTime.Now;
                                    expression = LambdaExtensions.AndAlso(expression, t => t.IsOverdue.Contains("true"));
                                }
                                else
                                {
                                    startTime = DateTime.Now;
                                    endTime = DateTime.Now.Date.AddDays(days).AddSeconds(-1);
                                    expression = LambdaExtensions.AndAlso(expression, t => (t.ExpectedTime >= startTime && t.ExpectedTime <= endTime) && t.IsOverdue.Contains("false"));
                                }

                                var expectedList = _unitOfWork.BorrowViewRepository.GetPage(expression, "-ExpectedTime", page, size, false);

                                var data = from s in expectedList?.content
                                           select new
                                           {
                                               id = s.Id,
                                               borrowType = s.BorrowType,
                                               applyName = s.ApplyUserName,
                                               applyTime = s.ApplyTime == null ? null : s.ApplyTime.Value.ToString("yyyy-MM-dd HH:mm"),
                                               warehouseName = s.WarehouseName,
                                               expectedTime = s.ExpectedTime == default ? null : s.ExpectedTime.Value.ToString("yyyy-MM-dd HH:mm")
                                           };
                                var content = new
                                {
                                    expectedList.totalElements,
                                    content = data
                                };
                                return JsonManager.ReturnSuccessResponse(content);
                            }
                            else
                            {
                                var expectedList = _unitOfWork.BorrowViewRepository.GetPage(p => p.Id.Contains(id), "-ExpectedTime", page, size, false);
                                var data = from s in expectedList?.content
                                           select new
                                           {
                                               id = s.Id,
                                               borrowType = s.BorrowType,
                                               applyName = s.ApplyUserName,
                                               applyTime = s.ApplyTime == null ? null : s.ApplyTime.Value.ToString("yyyy-MM-dd HH:mm"),
                                               warehouseName = s.WarehouseName,
                                               expectedTime = s.ExpectedTime == default ? null : s.ExpectedTime.Value.ToString("yyyy-MM-dd HH:mm")
                                           };
                                var content = new
                                {
                                    expectedList.totalElements,
                                    content = data
                                };
                                return JsonManager.ReturnSuccessResponse(content);
                            }

                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    return JsonManager.SimpleCustResponse(ex.ToString());
                }
            });
        }

    }
}