﻿using JmpModel.Model;
using JmpModel.Model.DataContext;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using MQTTnet;
using MQTTnet.Client.Receiving;
using MQTTnet.Protocol;
using MQTTnet.Server;
using System;
using System.Text;
using Microsoft.Extensions.DependencyInjection;
using Newtonsoft.Json.Linq;

namespace JunmpPoliceStation.Extensions
{
    public class MQTTServer : IDisposable
    {
        public MqttServer mqttServer { get; set; }

        private IConfiguration _configuration;
        public MQTTServer(IConfiguration configuration)
        {
            _configuration = configuration;
            StartMqttServer();
        }

        public async void StartMqttServer()
        {
            try
            {
                if (mqttServer == null)
                {
                    var optionsBuilder = new MqttServerOptionsBuilder()
                        .WithDefaultEndpoint()
                        .WithDefaultEndpointPort(int.Parse(_configuration.GetSection("MQTT:Port").Value))
                        .WithConnectionValidator(
                            c =>
                            {
                                var currentUser = _configuration["MQTT:UserName"].ToString();
                                var currentPWD = _configuration["MQTT:Password"].ToString();

                                if (string.IsNullOrEmpty(currentUser) || string.IsNullOrEmpty(currentPWD))
                                {
                                    c.ReasonCode = MqttConnectReasonCode.BadUserNameOrPassword;
                                    return;
                                }

                                if (c.Username != currentUser)
                                {
                                    c.ReasonCode = MqttConnectReasonCode.BadUserNameOrPassword;
                                    return;
                                }

                                if (c.Password != currentPWD)
                                {
                                    c.ReasonCode = MqttConnectReasonCode.BadUserNameOrPassword;
                                    return;
                                }

                                c.ReasonCode = MqttConnectReasonCode.Success;
                            })
                        .WithSubscriptionInterceptor(
                            c =>
                            {
                                c.AcceptSubscription = true;
                            })
                        .WithApplicationMessageInterceptor(
                            c =>
                            {
                                c.AcceptPublish = true;
                            });

                    mqttServer = new MqttFactory().CreateMqttServer() as MqttServer;
                    mqttServer.StartedHandler = new MqttServerStartedHandlerDelegate(OnMqttServerStarted);
                    mqttServer.StoppedHandler = new MqttServerStoppedHandlerDelegate(OnMqttServerStopped);

                    mqttServer.ClientConnectedHandler = new MqttServerClientConnectedHandlerDelegate(OnMqttServerClientConnected);
                    mqttServer.ClientDisconnectedHandler = new MqttServerClientDisconnectedHandlerDelegate(OnMqttServerClientDisconnected);
                    mqttServer.ClientSubscribedTopicHandler = new MqttServerClientSubscribedTopicHandlerDelegate(OnMqttServerClientSubscribedTopic);
                    mqttServer.ClientUnsubscribedTopicHandler = new MqttServerClientUnsubscribedTopicHandlerDelegate(OnMqttServerClientUnsubscribedTopic);
                    mqttServer.ApplicationMessageReceivedHandler = new MqttApplicationMessageReceivedHandlerDelegate(OnMqttServer_ApplicationMessageReceived);


                    await mqttServer.StartAsync(optionsBuilder.Build());
                    //lbxMonitor.BeginInvoke(_updateMonitorAction,
                    //    Logger.TraceLog(Logger.Level.Info, "MQTT Server is started."));

                }
            }
            catch (Exception ex)
            {
                //lbxMonitor.BeginInvoke(_updateMonitorAction,
                //            Logger.TraceLog(Logger.Level.Fatal, $"MQTT Server start fail.>{ex.Message}"));
            }
        }

        public async void StopMqttServer()
        {
            if (mqttServer == null) return;
            try
            {
                await mqttServer?.StopAsync();
                mqttServer = null;
                //lbxMonitor.BeginInvoke(_updateMonitorAction,
                //        Logger.TraceLog(Logger.Level.Info, "MQTT Server is stopped."));
            }
            catch (Exception ex)
            {
                //lbxMonitor.BeginInvoke(_updateMonitorAction,
                //            Logger.TraceLog(Logger.Level.Fatal, $"MQTT Server stop fail.>{ex.Message}"));
            }
        }

        public async void ServerPublishMqttTopic(string topic, string payload)
        {
            var message = new MqttApplicationMessage()
            {
                Topic = topic,
                Payload = Encoding.UTF8.GetBytes(payload)
            };
            await mqttServer.PublishAsync(message);
            //lbxMonitor.BeginInvoke(_updateMonitorAction,
            //        Logger.TraceLog(Logger.Level.Info, string.Format("MQTT Broker发布主题[{0}]成功！", topic)));
        }

        public void OnMqttServerStarted(EventArgs e)
        {
            //Console.WriteLine("MQTT服务启动完成！");
        }
        public void OnMqttServerStopped(EventArgs e)
        {
            //Console.WriteLine("MQTT服务停止完成！");
        }

        public void OnMqttServerClientConnected(MqttServerClientConnectedEventArgs e)
        {
            using (var scope = Startup.ServiceLocator.Instance.CreateScope())
            {
                var dbContext = scope.ServiceProvider.GetService<JunmppolicesqlContext>();

                dbContext?.ApiMqttMessages.Add(new ApiMqttMessage
                {
                    Id = Guid.NewGuid().ToString(),
                    CreateTime = DateTime.Now,
                    DeviceNo = e.ClientId,
                    DeviceType = e.ClientId.Substring(0, 2),
                    Topic = "Connected",
                    Payload = $"客户端[{e.ClientId}]已连接"
                });
                dbContext?.SaveChanges();
            }

            //Console.WriteLine($"客户端[{e.ClientId}]已连接");
        }
        public void OnMqttServerClientDisconnected(MqttServerClientDisconnectedEventArgs e)
        {
            using (var scope = Startup.ServiceLocator.Instance.CreateScope())
            {
                var dbContext = scope.ServiceProvider.GetService<JunmppolicesqlContext>();

                dbContext?.ApiMqttMessages.Add(new ApiMqttMessage
                {
                    Id = Guid.NewGuid().ToString(),
                    CreateTime = DateTime.Now,
                    DeviceNo = e.ClientId,
                    DeviceType = e.ClientId.Substring(0, 2),
                    Topic = "Disconnected",
                    Payload = $"客户端[{e.ClientId}]已断开连接！"
                });
                dbContext?.SaveChanges();
            }

            //Console.WriteLine($"客户端[{e.ClientId}]已断开连接！");
        }
        public void OnMqttServerClientSubscribedTopic(MqttServerClientSubscribedTopicEventArgs e)
        {
            using (var scope = Startup.ServiceLocator.Instance.CreateScope())
            {
                var dbContext = scope.ServiceProvider.GetService<JunmppolicesqlContext>();

                var message = new ApiMqttMessage
                {
                    Id = Guid.NewGuid().ToString(),
                    CreateTime = DateTime.Now,
                    DeviceNo = e.ClientId,
                    DeviceType = e.ClientId.Substring(0, 2),
                    Topic = e.TopicFilter.Topic,
                    Payload = $"客户端[{e.ClientId}]已成功订阅主题[{e.TopicFilter}]！"
                };
                dbContext?.ApiMqttMessages.Add(message);
                dbContext?.SaveChanges();
            }

            //Console.WriteLine($"客户端[{e.ClientId}]已成功订阅主题[{e.TopicFilter}]！");
        }
        public void OnMqttServerClientUnsubscribedTopic(MqttServerClientUnsubscribedTopicEventArgs e)
        {
            //Console.WriteLine($"客户端[{e.ClientId}]已成功取消订阅主题[{e.TopicFilter}]！");
        }
        public void OnMqttServer_ApplicationMessageReceived(MqttApplicationMessageReceivedEventArgs e)
        {
            try
            {
                if (string.IsNullOrEmpty(e.ClientId))
                {
                    return;
                }
                var message = new ApiMqttMessage
                {
                    Id = Guid.NewGuid().ToString(),
                    CreateTime = DateTime.Now,
                    DeviceNo = e.ClientId,
                    DeviceType = e.ClientId.Substring(0, 2),
                    Topic = e.ApplicationMessage.Topic,
                    Payload = Encoding.UTF8.GetString(e.ApplicationMessage.Payload)
                };
                if (message.Payload.Length > 2 && message.Payload.Substring(0, 2) == "/C")    //控制消息
                {
                    var data = message.Payload.Split('|');
                    if (data.Length == 3)
                    {
                        message.Topic = data[1];
                        var jObj = JObject.Parse(data[2]);
                        message.Token = jObj["token"]?.Value<string>();
                    }
                }

                using (var scope = Startup.ServiceLocator.Instance.CreateScope())
                {
                    var dbContext = scope.ServiceProvider.GetService<JunmppolicesqlContext>();
                    dbContext?.ApiMqttMessages.Add(message);
                    dbContext?.SaveChanges();
                }

                //Console.WriteLine($"客户端[{e.ClientId}]>> 主题：{e.ApplicationMessage.Topic} 负荷：{Encoding.UTF8.GetString(e.ApplicationMessage.Payload)} Qos：{e.ApplicationMessage.QualityOfServiceLevel} 保留：{e.ApplicationMessage.Retain}");

            }
            catch (Exception exception)
            {
                //catch
            }
        }

        public void Dispose()
        {
            StopMqttServer();
        }
    }
}
