﻿using JmpCommon;
using JmpCommon.Tools;
using JmpModel.Model;
using JmpModel.Model.Repository;
using JunmpPoliceStation.App_Start;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.WebUtilities;
using Microsoft.Extensions.Logging;
using Microsoft.Net.Http.Headers;
using Newtonsoft.Json.Linq;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;

namespace JunmpPoliceStation.Controllers
{
    /// <summary>
    /// 打印模块
    /// </summary>
    [ApiController]
    [Route("api/[controller]")]
    public class SoftUpdateController : ControllerBase
    {
        private IWebHostEnvironment _hostingEnvironment;
        private ILogger<SoftUpdateController> _logger;
        UnitOfWork _unitOfWork;
        public SoftUpdateController(ILogger<SoftUpdateController> logger, UnitOfWork unitOfWork, IWebHostEnvironment _hostingEnvironment)
        {
            _logger = logger;
            this._hostingEnvironment = _hostingEnvironment;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取所有通道配置
        /// </summary>
        /// <remarks>
        ///  ## 例子
        ///      {
        ///         "name":"软件名"
        ///         "page":0,
        ///         "size":10
        ///      }
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllUpdateCfg")]
        public async Task<HttpResponseMessage> GetAllUpdateCfg([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }
                        Expression<Func<BaseJpSoftUpdate, bool>> expression = t => t.AppCode != "";
                        if (!string.IsNullOrEmpty(entity.name))
                        {
                            string name = entity.name;
                            expression = expression.AndAlso(t => t.Name.Contains(name));
                        }

                        var cfgList = _unitOfWork.BaseJpSoftUpdateRepository.GetPage(expression, "UpdateTime", page, size);
                        var data = cfgList?.content.Select(s => new
                        {
                            id = s.Id,
                            files = new ArrayList()
                        {
                            new
                            {
                                id=s.Id,
                                url=s.Address,
                                name=Path.GetFileName( s.Address.Replace("/","\\"))
                            }
                        },
                            autoFiles = string.IsNullOrEmpty(s.AutoUpdateAddress) ? null :
                                new ArrayList()
                            {
                                new
                                {
                                    id=s.Id,
                                    url=s.AutoUpdateAddress,
                                    name=Path.GetFileName( s.AutoUpdateAddress.Replace("/","\\"))
                                }
                            },
                            address = s.Address,
                            name = s.Name,
                            note = s.Note,
                            updateTime = s.UpdateTime?.ToString("yyyy-MM-dd HH:mm"),
                            version = s.Version,
                            version2 = s.Version2,
                            appCode = s.AppCode,
                            fileMd5 = s.FileMd5,

                            isAutoUpdate = s.IsAutoUpdate,
                            autoUpdateAddress = s.AutoUpdateAddress,
                            autoUpdateFileMd5 = s.AutoUpdateFileMd5,
                            autoUpdateVersion2 = s.AutoUpdateVersion2,

                        }).ToList();
                        var content = new
                        {
                            totalElements = cfgList?.totalElements ?? 0,
                            content = data
                        };
                        return JsonManager.ReturnSuccessResponse(content);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 新增/更新配置
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "id":"", //新增可不填
        ///         "name":"", 
        ///         "appCode":"",
        ///         "version":100,
        ///         "version2":1.0.0.0,
        ///         "address":"",
        ///         "note":"",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("AddOrUpdateCfg")]
        public async Task<HttpResponseMessage> AddOrUpdateCfg([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";
                        bool bVer = int.TryParse(entity?.version ?? "0", out int version);
                        if (entity == null)
                        {
                            return JsonManager.SimpleCustResponse($"model is null");
                        }
                        else if (string.IsNullOrEmpty(entity.name ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"name is require");
                        }
                        else if (string.IsNullOrEmpty(entity.appCode ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"appCode is require");
                        }
                        else if (!bVer || version <= 0)
                        {
                            return JsonManager.SimpleCustResponse($"version is require");
                        }
                        else if (string.IsNullOrEmpty(entity.address ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"address is require");
                        }

                        if (bool.TryParse(entity.isAutoUpdate, out bool isAutoUpdate) && isAutoUpdate)
                        {
                            if (string.IsNullOrEmpty(entity.autoUpdateAddress))
                            {
                                return JsonManager.SimpleCustResponse($"autoUpdateAddress is require");
                            }
                        }

                        if (!string.IsNullOrEmpty(id))
                        {
                            var cfg = _unitOfWork.DbContext.BaseJpSoftUpdates.FirstOrDefault(t => t.Id == id);
                            if (cfg == null)
                            {
                                return JsonManager.SimpleCustResponse($"id:{id}数据不存在");
                            }
                            else
                            {


                                cfg.Name = entity?.name ?? "";
                                cfg.AppCode = entity?.appCode ?? "";
                                cfg.Note = entity?.note ?? "";
                                cfg.Version = version;
                                cfg.Version2 = entity?.version2 ?? "";
                                cfg.Address = entity?.address ?? "";
                                cfg.UpdateTime = DateTime.Now;

                                cfg.IsAutoUpdate = isAutoUpdate;
                                cfg.AutoUpdateAddress = entity.autoUpdateAddress;

                                //校验编号是否存在
                                var exits = _unitOfWork.DbContext.BaseJpSoftUpdates.FirstOrDefault(t => t.AppCode == cfg.AppCode && t.Id != cfg.Id);
                                if (exits != null)
                                {
                                    return JsonManager.SimpleCustResponse("appCode already exist", ResultCode.CHANNEL_CODE_EXIST.Code);
                                }
                                else
                                {
                                    var file = cfg.Address.Replace("/", "\\");
                                    var fullName = _hostingEnvironment.WebRootPath + file;
                                    //计算文件md5
                                    if (System.IO.File.Exists(fullName))
                                    {
                                        cfg.FileMd5 = GetMD5HashFromFile(fullName);
                                        //获取程序版本号（文件版本号一致）
                                        cfg.Version2 = System.Diagnostics.FileVersionInfo.GetVersionInfo(fullName).FileVersion ?? "1.0.0.0";
                                    }
                                    else
                                    {
                                        return JsonManager.SimpleCustResponse("file not found");
                                    }

                                    //自动更新
                                    if (cfg.IsAutoUpdate == true)
                                    {
                                        var file2 = cfg.AutoUpdateAddress.Replace("/", "\\");
                                        var fullName2 = _hostingEnvironment.WebRootPath + file2;
                                        //计算文件md5
                                        if (System.IO.File.Exists(fullName2))
                                        {
                                            cfg.AutoUpdateFileMd5 = GetMD5HashFromFile(fullName2);
                                            //获取程序版本号（文件版本号一致）
                                            cfg.AutoUpdateVersion2 = System.Diagnostics.FileVersionInfo.GetVersionInfo(fullName2).FileVersion ?? "1.0.0.0";
                                        }
                                        else
                                        {
                                            return JsonManager.SimpleCustResponse("autoUpdateFile not found");
                                        }
                                    }
                                    else
                                    {
                                        cfg.AutoUpdateFileMd5 = "";
                                        cfg.AutoUpdateAddress = "";
                                        cfg.AutoUpdateVersion2 = "";
                                    }

                                    bool result = _unitOfWork.BaseJpSoftUpdateRepository.Update(cfg);
                                    if (result)
                                    {
                                        return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                    }
                                    else
                                    {
                                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                    }
                                }
                            }
                        }
                        else
                        {

                            //新增数据
                            BaseJpSoftUpdate cfg = new BaseJpSoftUpdate()
                            {
                                Address = entity?.address ?? "",
                                Name = entity?.name ?? "",
                                AppCode = entity?.appCode ?? "",
                                Note = entity?.note ?? "",
                                Version = version,
                                Version2 = entity?.version2 ?? "",
                                Id = Guid.NewGuid().ToString(),
                                UpdateTime = DateTime.Now,
                                IsAutoUpdate = isAutoUpdate,
                                AutoUpdateAddress = entity.autoUpdateAddress ?? ""
                            };

                            //校验编号是否存在
                            var exits = _unitOfWork.DbContext.BaseJpSoftUpdates.FirstOrDefault(t => t.AppCode == cfg.AppCode);
                            if (exits != null)
                            {
                                return JsonManager.SimpleCustResponse("appCode already exist", ResultCode.CHANNEL_CODE_EXIST.Code);
                            }
                            else
                            {
                                var file = cfg.Address.Replace("/", "\\");
                                var fullName = _hostingEnvironment.WebRootPath + file;
                                //计算文件md5
                                if (System.IO.File.Exists(fullName))
                                {
                                    cfg.FileMd5 = GetMD5HashFromFile(fullName);
                                    //获取程序版本号（文件版本号一致）
                                    cfg.Version2 = System.Diagnostics.FileVersionInfo.GetVersionInfo(fullName).FileVersion ?? "1.0.0.0";
                                }
                                else
                                {
                                    return JsonManager.SimpleCustResponse("file not found");
                                }
                                //自动更新
                                if (cfg.IsAutoUpdate == true)
                                {
                                    var file2 = cfg.AutoUpdateAddress.Replace("/", "\\");
                                    var fullName2 = _hostingEnvironment.WebRootPath + file2;
                                    //计算文件md5
                                    if (System.IO.File.Exists(fullName2))
                                    {
                                        cfg.AutoUpdateFileMd5 = GetMD5HashFromFile(fullName2);
                                        //获取程序版本号（文件版本号一致）
                                        cfg.AutoUpdateVersion2 = System.Diagnostics.FileVersionInfo.GetVersionInfo(fullName2).FileVersion ?? "1.0.0.0";
                                    }
                                    else
                                    {
                                        return JsonManager.SimpleCustResponse("autoUpdateFile not found");
                                    }
                                }
                                else
                                {
                                    cfg.AutoUpdateFileMd5 = "";
                                    cfg.AutoUpdateAddress = "";
                                    cfg.AutoUpdateVersion2 = "";
                                }

                                bool result = _unitOfWork.BaseJpSoftUpdateRepository.Insert(cfg);
                                if (result)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }
                            }
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取软件版本更新
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        ///        ## 例子
        ///         {
        ///         "appCode": ""
        ///         }
        ///   </remarks>   
        [HttpPost("GetLastUpdate")]
        [NoSignAttribute]
        public async Task<HttpResponseMessage> GetLastUpdate([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string appCode = entity?.appCode ?? "";
                        if (string.IsNullOrEmpty(appCode))
                        {
                            return JsonManager.SimpleCustResponse("appCode is require");
                        }
                        else
                        {
                            Expression<Func<BaseJpSoftUpdate, bool>> expression = t => t.AppCode == appCode;

                            var softUpdate = _unitOfWork.DbContext.BaseJpSoftUpdates.FirstOrDefault(expression);

                            if (softUpdate == null)
                            {
                                return JsonManager.SimpleCustResponse("Invalid parameter");
                            }
                            else
                            {
                                return JsonManager.ReturnSuccessResponse(new
                                {
                                    version2 = softUpdate?.Version2,
                                    version = softUpdate?.Version,
                                    name = softUpdate?.Name,
                                    appCode = softUpdate?.AppCode,
                                    note = softUpdate?.Note,
                                    address = softUpdate?.Address,
                                    updateTime = softUpdate.UpdateTime,
                                    fileMd5 = softUpdate.FileMd5,

                                    isAutoUpdate = softUpdate.IsAutoUpdate,
                                    autoUpdateAddress = softUpdate.AutoUpdateAddress,
                                    autoUpdateFileMd5 = softUpdate.AutoUpdateFileMd5,
                                    autoUpdateVersion2 = softUpdate.AutoUpdateVersion2,
                                });

                            }

                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    return JsonManager.SimpleCustResponse(ex.ToString());
                }
            });
        }

        /// <summary>
        /// 获取软件版本更新
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        /// <remarks>
        ///        ## 例子
        ///         {
        ///         "id": "",
        ///         "type":"Normal 、AutoUpdate"
        ///         }
        ///   </remarks>   
        [HttpPost("DeletePackage")]
        public async Task<HttpResponseMessage> DeletePackage([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";
                        string type = entity?.type ?? "";
                        if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse("id is require");
                        }
                        if (string.IsNullOrEmpty(type))
                        {
                            return JsonManager.SimpleCustResponse("type is require");
                        }


                        Expression<Func<BaseJpSoftUpdate, bool>> expression = t => t.Id == id;

                        var softUpdate = _unitOfWork.DbContext.BaseJpSoftUpdates.FirstOrDefault(expression);

                        if (softUpdate == null)
                        {
                            return JsonManager.SimpleCustResponse("Invalid parameter");
                        }
                        else
                        {
                            var file = "";
                            switch (type)
                            {
                                case "Normal":
                                    {
                                        file = softUpdate.Address.Replace("/", "\\");
                                        break;
                                    }
                                case "AutoUpdate":
                                    {
                                        file = softUpdate.AutoUpdateAddress.Replace("/", "\\");
                                        break;
                                    }
                            }
                            var fullName = _hostingEnvironment.WebRootPath + file;

                            if (System.IO.File.Exists(fullName))
                            {
                                System.IO.File.Delete(fullName);
                            }
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);

                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleCustResponse(ex.ToString());
                }
            });
        }


        /// <summary>
        /// 大文件上传，参数“softPackage”，可自定义
        /// </summary>
        /// <returns></returns>
        [HttpPost("UploadSoftPackage")]
        [RequestSizeLimit(4194304000)/*文件大小上限，4000M，可自定义*/]
        [NoSign]
        public async Task<HttpResponseMessage> UploadSoftPackage()
        {

            try
            {
                //获取boundary
                var boundary = HeaderUtilities.RemoveQuotes(MediaTypeHeaderValue.Parse(Request.ContentType).Boundary).Value ?? "";
                //得到reader
                var reader = new MultipartReader(boundary, HttpContext.Request.Body);
                var section = await reader.ReadNextSectionAsync();

                var webRoot = _hostingEnvironment.WebRootPath/*IWebHostEnvironment，用于获取网站、API根目录*/;
                bool flag = false;
                long fileLen = 0;
                List<string> extList = new List<string>() { ".exe", ".apk", ".zip", ".rar", ".bin" };
                var savePath = string.Empty;
                //读取section
                while (section != null)
                {
                    var hasContentDispositionHeader = ContentDispositionHeaderValue.TryParse(section.ContentDisposition, out var contentDisposition);
                    if (hasContentDispositionHeader)
                    {
                        if (contentDisposition.Name == "softPackage")
                        {
                            var fileName = contentDisposition.FileName.Value;
                            var fileExt = Path.GetExtension(fileName);
                            if (extList.Contains(fileExt))
                            {
                                flag = true;
                                var dir = Path.Combine(webRoot, "SoftPackage");
                                if (!Directory.Exists(dir))
                                {
                                    Directory.CreateDirectory(dir);
                                }
                                savePath = $"/SoftPackage/{fileName}";
                                fileLen = await WriteFileAsync(section.Body, Path.Combine(dir, fileName));
                            }
                            else
                            {
                                return JsonManager.SimpleCustResponse("unsupported media type");
                            }
                        }
                    }
                    section = await reader.ReadNextSectionAsync();
                }
                if (!flag)
                {
                    return JsonManager.SimpleCustResponse("softPackage is require");
                }
                else
                {
                    if (fileLen > 0)
                    {
                        return JsonManager.ReturnSuccessResponse(savePath);
                    }
                    else
                    {
                        return JsonManager.SimpleCustResponse("file upload failed");
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex.ToString());
                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);

            }

        }

        /// <summary>
        /// 写文件导到磁盘
        /// </summary>
        /// <param name="stream">流</param>
        /// <param name="path">文件保存路径</param>
        /// <returns></returns>
        private async Task<int> WriteFileAsync(Stream stream, string path)
        {
            const int FILE_WRITE_SIZE = 84975;//写出缓冲区大小
            int writeCount = 0;
            using (FileStream fileStream = new FileStream(path, FileMode.Create, FileAccess.Write, FileShare.Write, FILE_WRITE_SIZE, true))
            {
                byte[] byteArr = new byte[FILE_WRITE_SIZE];
                int readCount = 0;
                while ((readCount = await stream.ReadAsync(byteArr, 0, byteArr.Length)) > 0)
                {
                    await fileStream.WriteAsync(byteArr, 0, readCount);
                    writeCount += readCount;
                }
            }
            return writeCount;
        }

        /// <summary>
        /// 获取文件的MD5码
        /// </summary>
        /// <param name="fileName">传入的文件名（含路径及后缀名）</param>
        /// <returns></returns>
        private string GetMD5HashFromFile(string fileName)
        {
            try
            {
                FileStream file = new FileStream(fileName, System.IO.FileMode.Open);
                MD5 md5 = new MD5CryptoServiceProvider();
                byte[] retVal = md5.ComputeHash(file);
                file.Close();
                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < retVal.Length; i++)
                {
                    sb.Append(retVal[i].ToString("x2"));
                }
                return sb.ToString();
            }
            catch (Exception ex)
            {
                throw new Exception("GetMD5HashFromFile() fail,error:" + ex.Message);
            }
        }
    }
}
