﻿using JmpCommon;
using JmpCommon.Tools;
using JmpModel.Model;
using JmpModel.Model.Repository;
using JunmpPoliceStation.App_Start;
using JunmpPoliceStation.Models.Repository;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Threading.Tasks;
using Z.EntityFramework.Plus;

namespace JunmpPoliceStation.Controllers
{
    [ApiController]
    [Route("api/[controller]")]
    public class ShelfController : ControllerBase
    {
        //private ICabinetRepository _CabinetRepository;
        private ILogger<ShelfController> _logger;
        UnitOfWork _unitOfWork;
        public List<BaseJpOrganization> ListEq = new List<BaseJpOrganization>();

        public ShelfController(ILogger<ShelfController> logger, UnitOfWork unitOfWork)
        {
            _logger = logger;
            //_CabinetRepository = CabinetRepository;
            _unitOfWork = unitOfWork;
        }

        /// <summary>
        /// 获取指定仓库货架列表
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "name":"货架名称",
        ///         "warehouseId":"仓库id",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListShelf")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetListShelf([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<BaseJpShelfInfo, bool>> expression = t => true;
                        if (!String.IsNullOrEmpty(entity.name))
                        {
                            string Name = entity.name;
                            expression = LambdaExtensions.AndAlso(expression, t => t.Name.Contains(Name));
                        }
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseId.Equals(warehouseId));
                        }

                        var param = new string[] {
                            "Warehouse"
                        };

                        List<BaseJpShelfInfo> dataList = _unitOfWork.BaseJpShelfInfoRepository.GetList(expression, t => t.CreateTime, false, param).ToList();


                        var data = dataList.Select(c => new
                        {
                            c.Id,
                            c.CreateTime,
                            c.Name,
                            c.Code,
                            c.Type,
                            c.WarehouseId,
                            c.ShelfSize,
                            c.Range,
                            c.Row,
                            c.Column,
                            c.Url,
                            Warehouse = new
                            {
                                Id = c.Warehouse?.Id,
                                Sort = c.Warehouse?.Sort,
                                Name = c.Warehouse?.Name,
                                Location = c.Warehouse?.Location,
                                LocationDetail = c.Warehouse?.LocationDetail,
                                OrgizationId = c.Warehouse?.OrgizationId,
                                Phone = c.Warehouse?.Phone,
                                CreateTime = c.Warehouse?.CreateTime,
                                UpdateTime = c.Warehouse?.UpdateTime,
                                UpdateUser = c.Warehouse?.UpdateUser,
                            }
                        }
                        ).ToList();


                        return JsonManager.ReturnSuccessResponse(data);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取指定仓库全部货架（智能）+物资
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseId":"仓库id",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListShelfAndInventory")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetListShelfAndInventory([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<BaseJpShelfInfo, bool>> expression = t => !string.IsNullOrEmpty(t.Url) && t.Type == "Smart";
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = expression.AndAlso(t => t.WarehouseId.Equals(warehouseId));
                        }

                        var param = new string[] {
                            "Warehouse",
                            "Warehouse.Orgization",
                            "CommonJpEquipmentInventories",
                            "CommonJpEquipmentInventories.EquipmentCodeNavigation",
                            "CommonJpEquipmentInventories.EquipmentCodeNavigation.Equipment",
                        };

                        List<BaseJpShelfInfo> dataList = _unitOfWork.DbContext.BaseJpShelfInfos
                            .IncludeFilter(x => x.Warehouse)
                            .IncludeFilter(x => x.Warehouse.Orgization)
                            .IncludeFilter(x => x.CommonJpEquipmentInventories.Where(y => y.CurrentState == 0))
                            .IncludeFilter(x => x.CommonJpEquipmentInventories.Where(y => y.CurrentState == 0).Select(y => y.EquipmentCodeNavigation))
                            .IncludeFilter(x => x.CommonJpEquipmentInventories.Where(y => y.CurrentState == 0).Select(y => y.EquipmentCodeNavigation).Select(z => z.Equipment))
                            .Where(expression)
                            .OrderBy(x => x.Url)
                            .ToList();

                        var data = dataList.Select(c => new
                        {
                            c.Name,
                            c.Url,
                            OrganizationName = c.Warehouse.Orgization.Name,
                            infoList = c.CommonJpEquipmentInventories
                                .GroupBy(y => y.EquipmentCodeNavigation.EquipmentId)
                                .Select(y => new
                                {
                                    y.FirstOrDefault()?.EquipmentCodeNavigation.Equipment.Name,
                                    InventoryList = y
                                        .GroupBy(x => x.EquipmentCode)
                                        .Select(x => new
                                        {
                                            Name = x.FirstOrDefault()?.EquipmentCodeNavigation.Name,
                                            Count = x.Count()
                                        })
                                })
                        }
                        ).ToList();

                        return JsonManager.ReturnSuccessResponse(data);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取指定仓库对应装备类型（号型选填）的装备存放货架位置（多个）
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseId":"仓库id",
        ///         "equipmentDetailId":"装备类型id",
        ///         "sizeId":"装备号型id",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquipmentShelfPosition")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetListEquipmentShelfPosition([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        string warehouseId = entity.warehouseId;
                        string equipmentDetailId = entity.equipmentDetailId;
                        string sizeId = entity.sizeId;


                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => t.CurrentState != 6 && !string.IsNullOrEmpty(t.ShelfId);
                        if (!string.IsNullOrEmpty(warehouseId))
                        {
                            expression = expression.AndAlso(t => t.WarehouseCode.Equals(warehouseId));
                        }
                        else
                        {
                            return JsonManager.SimpleCustResponse($"warehouseId is require");
                        }
                        if (!string.IsNullOrEmpty(equipmentDetailId))
                        {
                            expression = expression.AndAlso(t => t.EquipmentCode.Equals(equipmentDetailId));
                        }
                        else
                        {
                            return JsonManager.SimpleCustResponse($"equipmentDetailId is require");
                        }
                        if (!string.IsNullOrEmpty(sizeId))
                        {
                            expression = expression.AndAlso(t => t.EquipmentSizecode.Equals(sizeId));
                        }

                        var param = new string[] {
                            "Shelf",
                        };

                        var dataList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, includes: param).ToList();

                        var data = dataList
                            .GroupBy(x => new { x.ShelfId, x.ShelfRange, x.ShelfRow, x.ShelfColumn })
                            .Select(c => new
                            {
                                Id = c.Key.ShelfId,
                                Name = c.FirstOrDefault()?.Shelf?.Name,
                                Range = c.Key.ShelfRange,
                                Row = c.Key.ShelfRow,
                                Column = c.Key.ShelfColumn,
                            })
                            .OrderBy(x => x.Name)
                            .ThenBy(x => x.Range)
                            .ThenBy(x => x.Row)
                            .ThenBy(x => x.Column)
                            .ToList();

                        return JsonManager.ReturnSuccessResponse(data);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 通过id获取指定货架
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "id":"货架id",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetShelfById")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetShelfById([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<BaseJpShelfInfo, bool>> expression = t => true;
                        if (!string.IsNullOrEmpty(entity.id))
                        {
                            string id = entity.id;
                            expression = expression.AndAlso(t => t.Id.Equals(id));
                        }
                        else
                        {
                            return JsonManager.SimpleCustResponse($"id is require");
                        }

                        var param = new string[] {
                            "Warehouse"
                        };

                        BaseJpShelfInfo data = _unitOfWork.BaseJpShelfInfoRepository.Get(expression);

                        return JsonManager.ReturnSuccessResponse(new
                        {
                            data.Id,
                            data.CreateTime,
                            data.Name,
                            data.Code,
                            data.Type,
                            data.WarehouseId,
                            data.ShelfSize,
                            data.Range,
                            data.Row,
                            data.Column,
                            data.Url,
                            Warehouse = new
                            {
                                Id = data.Warehouse?.Id,
                                Sort = data.Warehouse?.Sort,
                                Name = data.Warehouse?.Name,
                                Location = data.Warehouse?.Location,
                                LocationDetail = data.Warehouse?.LocationDetail,
                                OrgizationId = data.Warehouse?.OrgizationId,
                                Phone = data.Warehouse?.Phone,
                                CreateTime = data.Warehouse?.CreateTime,
                                UpdateTime = data.Warehouse?.UpdateTime,
                                UpdateUser = data.Warehouse?.UpdateUser,
                            }
                        });

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取货架分页列表
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "page": "0",
        ///         "size": "10",
        ///         "name":"货架名称",
        ///         "warehouseId":"仓库id",
        ///         "orgId":"组织机构id",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetPageShelf")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetPageShelf([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }

                        Expression<Func<BaseJpShelfInfo, bool>> expression = t => true;
                        if (!String.IsNullOrEmpty(entity.orgId))
                        {
                            string orgId = entity.orgId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.Warehouse.OrgizationId.Equals(orgId));
                        }
                        else
                        {
                            return JsonManager.SimpleCustResponse($"orgId is require");
                        }
                        if (!String.IsNullOrEmpty(entity.name))
                        {
                            string Name = entity.name;
                            expression = LambdaExtensions.AndAlso(expression, t => t.Name.Contains(Name));
                        }
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseId.Equals(warehouseId));
                        }


                        var param = new string[] {
                            "Warehouse"
                        };

                        PageData<BaseJpShelfInfo> pagedata = _unitOfWork.BaseJpShelfInfoRepository.GetPage(expression, "-CreateTime", page, size, false, param);

                        var data = pagedata.content.Select(c => new
                        {
                            c.Id,
                            c.CreateTime,
                            c.Name,
                            c.Code,
                            c.Type,
                            c.WarehouseId,
                            c.ShelfSize,
                            c.Range,
                            c.Row,
                            c.Column,
                            c.Url,
                            Warehouse = new
                            {
                                Id = c.Warehouse?.Id,
                                Sort = c.Warehouse?.Sort,
                                Name = c.Warehouse?.Name,
                                Location = c.Warehouse?.Location,
                                LocationDetail = c.Warehouse?.LocationDetail,
                                OrgizationId = c.Warehouse?.OrgizationId,
                                Phone = c.Warehouse?.Phone,
                                CreateTime = c.Warehouse?.CreateTime,
                                UpdateTime = c.Warehouse?.UpdateTime,
                                UpdateUser = c.Warehouse?.UpdateUser,
                            }
                        }
                        ).ToList();

                        var dataPage = new
                        {
                            totalElements = pagedata.totalElements,
                            content = data
                        };

                        return JsonManager.ReturnSuccessResponse(dataPage);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }


        /// <summary>
        /// 添加或修改货架信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "编号",
        ///        "name":"货架名称",
        ///        "code":"货架编号",
        ///        "type" : "货架类型   Normal:普通货架 Smart:智能货架",
        ///        "warehouseId" : "仓库id",
        ///        "shelfSize" : "货架尺寸",
        ///        "range":"货架排数",
        ///        "row":"货架层数",
        ///        "column":"单层货架列数",
        ///        "url":"智能货架url",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("AddOrUpdateShelf")]
        //[NoSign]
        public async Task<HttpResponseMessage> AddOrUpdateShelf([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(entity.name ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"Name is require");
                        }
                        else if (string.IsNullOrEmpty(entity.warehouseId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"warehouseId is require");
                        }
                        else if (string.IsNullOrEmpty(entity.type ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"type is require");
                        }
                        else if (!int.TryParse(entity.range, out int range) || range < 1)
                        {
                            return JsonManager.SimpleCustResponse($"range is require");
                        }
                        else if (!int.TryParse(entity.row, out int row) || row < 1)
                        {
                            return JsonManager.SimpleCustResponse($"row is require");
                        }
                        else if (!int.TryParse(entity.column, out int column) || column < 1)
                        {
                            return JsonManager.SimpleCustResponse($"column is require");
                        }
                        else if (!string.IsNullOrEmpty(id))
                        {
                            Expression<Func<BaseJpShelfInfo, bool>> expression = t => t.Id.Equals(id);
                            BaseJpShelfInfo updateObject = _unitOfWork.BaseJpShelfInfoRepository.Get(expression);
                            if (updateObject != null)
                            {
                                //当货架上存放装备时禁止修改货架信息(在库状态下的物品）
                                var invList =
                                    _unitOfWork.EquipmentInventoryRepository.Count(x => x.ShelfId == updateObject.Id && x.CurrentState == 0);
                                if (invList > 0)
                                {
                                    //检查字段修改
                                    if (updateObject.WarehouseId != entity.warehouseId
                                        || updateObject.Range != Convert.ToInt32(entity.range)
                                        || updateObject.Row != Convert.ToInt32(entity.row)
                                        || updateObject.Column != Convert.ToInt32(entity.column)
                                        )
                                    {
                                        return JsonManager.SimpleCustResponse("当前货架已存放装备，禁止修改货架物理信息");
                                    }
                                }
                                updateObject.Name = entity.name;
                                updateObject.Code = entity.code;
                                updateObject.Type = entity.type;
                                updateObject.WarehouseId = entity.warehouseId;
                                updateObject.ShelfSize = entity.shelfSize;
                                updateObject.Range = Convert.ToInt32(entity.range);
                                updateObject.Row = Convert.ToInt32(entity.row);
                                updateObject.Column = Convert.ToInt32(entity.column);
                                updateObject.Url = entity.url;
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }

                            bool result = _unitOfWork.BaseJpShelfInfoRepository.Update(updateObject);

                            if (result)
                            {
                                return JsonManager.ReturnSuccessResponse(updateObject);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {
                            //新增数据
                            BaseJpShelfInfo addObject = new BaseJpShelfInfo()
                            {
                                Id = Guid.NewGuid().ToString(),
                                CreateTime = DateTime.Now,
                                Name = entity.name,
                                Code = entity.code,
                                Type = entity.type,
                                WarehouseId = entity.warehouseId,
                                ShelfSize = entity.shelfSize,
                                Range = Convert.ToInt32(entity.range),
                                Row = Convert.ToInt32(entity.row),
                                Column = Convert.ToInt32(entity.column),
                                Url = entity.url,
                            };

                            bool result = _unitOfWork.BaseJpShelfInfoRepository.Insert(addObject);

                            if (result)
                            {
                                return JsonManager.ReturnSuccessResponse(addObject);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 更新货架物品存放位置
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "货架id",
        ///        "dataList":
        ///         [
        ///             {
        ///                 "epc":"物品epc",
        ///                 "range":"货架排位置",
        ///                 "row":"货架行位置",
        ///                 "column":"货架列位置",
        ///             }
        ///         ]
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("UpdateInventoryAddress")]
        //[NoSign]
        public async Task<HttpResponseMessage> UpdateInventoryAddress([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse($"id is require");
                        }
                        else if (entity.dataList == null || !(entity.dataList is IList) || entity.dataList.Count == 0)
                        {
                            return JsonManager.SimpleCustResponse($"dataList is require");
                        }
                        else if (!string.IsNullOrEmpty(id))
                        {
                            Expression<Func<BaseJpShelfInfo, bool>> expression = t => t.Id.Equals(id);
                            BaseJpShelfInfo shelfInfo = _unitOfWork.BaseJpShelfInfoRepository.Get(expression);
                            if (shelfInfo != null)
                            {
                                var successList = new List<string>();
                                var failList = new List<string[]>();
                                //更新
                                foreach (dynamic item in (IList)entity.dataList)
                                {
                                    if (string.IsNullOrEmpty(item.epc))
                                    {
                                        //过滤空的epc
                                        continue;
                                    }
                                    //if (string.IsNullOrEmpty(item.state))
                                    //{
                                    //    //过滤空的出入库状态
                                    //    continue;
                                    //}
                                    string epc = item.epc;
                                    var inventory = _unitOfWork.EquipmentInventoryRepository.Get(x => x.Epc == epc);

                                    if (inventory != null)
                                    {
                                        inventory.ShelfId = shelfInfo.Id;
                                        //inventory.Address = item.address;
                                        inventory.ShelfRange = Convert.ToInt32(item.range);
                                        inventory.ShelfRow = Convert.ToInt32(item.row);
                                        inventory.ShelfColumn = Convert.ToInt32(item.column);

                                        _unitOfWork.EquipmentInventoryRepository.Update(inventory, false);
                                    }
                                    else
                                    {
                                        failList.Add(new[] { epc, "无效物资" });
                                    }
                                }

                                return JsonManager.SimpleStatusResponse(_unitOfWork.SaveChanges() > 0
                                    ? ResultCode.OPERATE_SUCCESS
                                    : ResultCode.OPERATE_FAILED);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {

                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }


        /// <summary>
        /// 删除货架信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///        "id" : "编号"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("DeleteShelf")]
        //[NoSign]
        public async Task<HttpResponseMessage> DeleteShelf([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                        }
                        else if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse($"id is require");
                        }
                        else if (!string.IsNullOrEmpty(id))
                        {
                            Expression<Func<BaseJpShelfInfo, bool>> expression = t => t.Id.Equals(id);
                            BaseJpShelfInfo deleteObject = _unitOfWork.BaseJpShelfInfoRepository.Get(expression);
                            if (deleteObject != null)
                            {
                                //当货架上存放装备时禁止删除货架(在库状态下的物品）
                                var invList =
                                    _unitOfWork.EquipmentInventoryRepository.Count(x => x.ShelfId == deleteObject.Id && x.CurrentState == 0);
                                if (invList > 0)
                                {
                                    return JsonManager.SimpleCustResponse("当前货架已存放装备，禁止删除货架");
                                }

                                //清理可能的异常数据
                                var errorData =
                                    _unitOfWork.EquipmentInventoryRepository.GetList(x => x.ShelfId == deleteObject.Id)
                                        .ToList();

                                foreach (var inventory in errorData)
                                {
                                    inventory.ShelfId = null;
                                    inventory.ShelfRange = null;
                                    inventory.ShelfRow = null;
                                    inventory.ShelfColumn = null;
                                }
                                _unitOfWork.EquipmentInventoryRepository.Update(errorData);

                                bool result = _unitOfWork.BaseJpShelfInfoRepository.Delete(deleteObject);

                                if (result)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {

                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

    }
}
