﻿using JmpCommon;
using JmpCommon.Tools;
using JmpModel.Model;
using JmpModel.Model.Enumerates;
using JmpModel.Model.Repository;
using JunmpPoliceStation.Extensions;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Net;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;
using JunmpPoliceStation.App_Start;
using Rextec.SOA.Infrastructure;
using static JmpModel.Model.Enumerates.BorrowEnum;

namespace JunmpPoliceStation.Controllers
{
    [Route("api/[controller]")]
    [ApiController]
    public class InventoryController : ControllerBase
    {
        private HttpHelper _httpHelper;
        private ILogger<InventoryController> _logger;
        UnitOfWork _unitOfWork;
        public List<BaseJpOrganization> ListEq = new List<BaseJpOrganization>();
        public IConfiguration Configuration { get; }

        public InventoryController(ILogger<InventoryController> logger, UnitOfWork unitOfWork, IConfiguration configuration, HttpHelper httpHelper)
        {
            _httpHelper = httpHelper;
            _logger = logger;

            _unitOfWork = unitOfWork;
            Configuration = configuration;
        }

        //递归取数据
        private List<BaseJpOrganization> GetClassID(String id)
        {

            var query = from c in ListEq
                        where c.ParentId == id
                        select c;
            return query.ToList().Concat(query.ToList().SelectMany(t => GetClassID(t.Id))).ToList();
        }

        /// <summary>
        /// 第三方生成单据信息
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "type":"订单类型"，purchase采购,borrow借用,recipients领用,transfer调拨，fix维修，scrap报废,
        ///         "warehouseId":"仓库ID"调拨时为请求仓库，其他业务为当前业务仓库,
        ///         "orgId":"",//调拨时为申请调拨的组织机构
        ///         "transferId":""//调拨出库的组织机构//仅限调拨使用
        ///         "orderNo":"订单号，
        ///         "expectedReturnTime":"借用归还时间"//借用时使用，其他业务不做处理
        ///         "detailList":[//实际出货
        ///         {  
        ///            "detailQuantity":""
        ///            "epc":""//维修，报废时使用，其他业务不做处理
        ///            "sendWarehouseId":"发货仓库代码",//调拨时使用，其他业务不做处理
        ///            "equipmentSizeId":"",
        ///            "equipmentDetailId":"",
        ///            "supplierId":""//采购不做处理
        ///            "warehouseList":[{//调拨时使用
        ///               "warehouseId":"",
        ///               "quantity":""
        ///             }]
        ///         }]
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("CreateOrder")]
        //[NoSign]
        public async Task<HttpResponseMessage> CreateOrder([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        //第三方调用接口验证orgId正确性
                        if (!string.IsNullOrEmpty(entity._PUSHORG))
                        {
                            if (entity.orgId?.ToString() != entity._PUSHORG.ToString())
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.ORGANIZATION_ERROR);
                            }
                        }

                        string note = entity.note;
                        string type = entity.type;
                        string warehouseId = entity.warehouseId;
                        string orgId = entity.orgId;
                        string orderNo = entity.orderNo;
                        string expectedReturnTime = entity.expectedReturnTime;
                        string orderState = entity.orderState;
                        //List<string> tmpList = JsonManager.GetDetailIds(entity.detailList);
                        //if (entity.detailList == null)
                        //{
                        //    return JsonManager.SimpleCustResponse($"detailList is require");
                        //}

                        switch (type)
                        {
                            case "purchase":
                                CommonJpPurchaseOrder queryOrder = _unitOfWork.PurchaseOrderyRepository.Get(p => p.OrderCode.Equals(orderNo));
                                if (queryOrder != null)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_IS_EXIST);
                                }
                                if (!orderNo.Contains("CG"))
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_ERROR);
                                }
                                CommonJpPurchaseOrder purchaseOrder = new CommonJpPurchaseOrder()
                                {
                                    ApplyTime = DateTime.Now,
                                    ApprovalResult = (int)ApproveType.已审批,
                                    ApprovalTime = null,
                                    CreateTime = DateTime.Now,
                                    ApplyId = entity.applyId,
                                    WarehouseId = entity.warehouseId,
                                    Id = Guid.NewGuid().ToString(),
                                    OrgId = entity.orgId,
                                    ApplyName = entity.applyName,
                                    OrderCode = orderNo,
                                    CurrentState = 0,

                                };
                                List<CommonJpPurchaseDetail> detailsList = new List<CommonJpPurchaseDetail>();
                                foreach (var item in entity.detailList)
                                {
                                    if (!int.TryParse(item.detailQuantity, out int detailQuantity))
                                    {
                                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                    }
                                    detailsList.Add(new CommonJpPurchaseDetail()
                                    {
                                        Id = Guid.NewGuid().ToString(),
                                        EquipmentId = item.equipmentDetailId,
                                        SizeId = item.equipmentSizeId,
                                        Note = item.note,
                                        OrderId = purchaseOrder.Id,
                                        Quantity = detailQuantity,
                                        SupplierId = null,
                                        CreateTime = DateTime.Now,
                                        State = 0
                                    });
                                }
                                purchaseOrder.CommonJpPurchaseDetails = detailsList;
                                bool result = _unitOfWork.PurchaseOrderyRepository.Insert(purchaseOrder);
                                if (result)
                                {
                                    return JsonManager.ReturnSuccessResponse(new
                                    {
                                        purchaseOrder.ApplyTime,
                                        purchaseOrder.ApprovalResult,
                                        purchaseOrder.ApprovalTime,
                                        purchaseOrder.CreateTime,
                                        purchaseOrder.ApplyId,
                                        purchaseOrder.WarehouseId,
                                        purchaseOrder.Id,
                                        purchaseOrder.OrgId,
                                        purchaseOrder.ApplyName,
                                        purchaseOrder.OrderCode,
                                        purchaseOrder.CurrentState,
                                        Details = purchaseOrder.CommonJpPurchaseDetails.Select(x => new
                                        {
                                            x.Id,
                                            x.EquipmentId,
                                            x.SizeId,
                                            x.Note,
                                            x.OrderId,
                                            x.Quantity,
                                            x.SupplierId,
                                            x.CreateTime,
                                            x.State
                                        })
                                    });
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }

                            case "borrow":
                            case "recipients":
                                CommonJpBorrowReturnApply borrowQuery = _unitOfWork.BorrowReturnApplyRepository.Get(p => p.OrderCode.Equals(orderNo));
                                if (borrowQuery != null)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_IS_EXIST);
                                }
                                if (type.Equals("borrow"))
                                {
                                    if (!orderNo.Contains("JY"))
                                    {
                                        return JsonManager.SimpleStatusResponse(ResultCode.ORDER_ERROR);
                                    }
                                }
                                else if (type.Equals("recipients"))
                                {
                                    if (!orderNo.Contains("LY"))
                                    {
                                        return JsonManager.SimpleStatusResponse(ResultCode.ORDER_ERROR);
                                    }
                                }
                                if (string.IsNullOrEmpty(expectedReturnTime) && type.Equals("borrow"))
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }
                                CommonJpBorrowReturnApply borrowApplyOrder = new CommonJpBorrowReturnApply();
                                if (type.Equals("borrow"))
                                {
                                    borrowApplyOrder = new CommonJpBorrowReturnApply()
                                    {
                                        ExpectedReturnTime = DateTime.Parse(expectedReturnTime.ToString()),
                                        ChangeFlag = 0,
                                        ActionType = 1,
                                        ApplyTime = DateTime.Now,
                                        CreateTime = DateTime.Now,
                                        WarehouseId = entity.warehouseId,
                                        Id = Guid.NewGuid().ToString(),
                                        OrgId = entity.orgId,
                                        OrderCode = orderNo,
                                    };
                                }
                                else if (type.Equals("recipients"))
                                {
                                    borrowApplyOrder = new CommonJpBorrowReturnApply()
                                    {
                                        ChangeFlag = 0,
                                        ActionType = 1,
                                        ApplyTime = DateTime.Now,
                                        CreateTime = DateTime.Now,
                                        WarehouseId = entity.warehouseId,
                                        Id = Guid.NewGuid().ToString(),
                                        OrgId = entity.orgId,
                                        OrderCode = orderNo,
                                    };
                                }
                                CommonJpBorrowReturn borrowOrder = new CommonJpBorrowReturn()
                                {

                                    Id = Guid.NewGuid().ToString(),
                                    ActionType = type.Equals("borrow") ? 0 : 1,
                                    WarehouseId = entity.warehouseId,
                                    OrderCode = orderNo,
                                    CurrentState = (int)CurrentState.未出去,
                                    OrgId = entity.orgId,
                                    CreateTime = DateTime.Now,
                                    IsWork = false,

                                };
                                List<CommonJpBorrowReturnDetailEstimate> borrowEsDetailsList = new List<CommonJpBorrowReturnDetailEstimate>();
                                List<CommonJpBorrowReturnDetailReality> borrowReDetailsList = new List<CommonJpBorrowReturnDetailReality>();
                                foreach (var item in entity.detailList)
                                {
                                    borrowEsDetailsList.Add(new CommonJpBorrowReturnDetailEstimate()
                                    {
                                        Id = Guid.NewGuid().ToString(),
                                        EquipmentId = item.equipmentDetailId,
                                        SizeId = item.equipmentSizeId,
                                        EstimateId = borrowApplyOrder.Id,
                                        EquipmentCount = int.Parse(item.detailQuantity),
                                        //SupplierId = item.supplierId==null?null:item.supplierId
                                    });
                                    borrowReDetailsList.Add(new CommonJpBorrowReturnDetailReality()
                                    {
                                        Id = Guid.NewGuid().ToString(),
                                        SizeId = item.equipmentSizeId,
                                        EquipmentDetailId = item.equipmentDetailId,
                                        RealityId = borrowOrder.Id,
                                        State = (int)CurrentState.未出去,
                                        EstimateId = borrowApplyOrder.Id,
                                        EquipmentCount = int.Parse(item.detailQuantity),

                                        //SupplierId = item.supplierId == null ? null : item.supplierId

                                    });

                                }
                                borrowOrder.CommonJpBorrowReturnDetailRealities = borrowReDetailsList;
                                borrowApplyOrder.CommonJpBorrowReturnDetailEstimates = borrowEsDetailsList;
                                bool borrowResult = _unitOfWork.BorrowReturnApplyRepository.Insert(borrowApplyOrder, false);
                                borrowResult = _unitOfWork.BorrowReturnRepository.Insert(borrowOrder, true);
                                if (borrowResult)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }



                            case "transfer":
                                CommonJpTransferApply transferQuery = _unitOfWork.TransferApplyRepository.Get(p => p.Order.Equals(orderNo));
                                if (transferQuery != null)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_IS_EXIST);
                                }

                                if (!orderNo.Contains("DB"))
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_ERROR);
                                }

                                CommonJpTransferApply transferApplyOrder = new CommonJpTransferApply()
                                {
                                    Id = Guid.NewGuid().ToString(),
                                    ActionType = 0,
                                    TransferId = entity.transferId,
                                    TargetId = orgId,
                                    TransferWarehouse = warehouseId,
                                    ApplyTime = DateTime.Now,
                                    CreateTime = DateTime.Now,
                                    UpdateTime = DateTime.Now,
                                    Order = orderNo,
                                };
                                CommonJpTransfer transferOrder = new CommonJpTransfer()
                                {
                                    Id = Guid.NewGuid().ToString(),
                                    ActionType = 0,
                                    ReceiveOrgizationId = orgId,
                                    SendOrgizationId = entity.transferId,
                                    ReceiveWarehouseId = entity.warehouseId,
                                    Order = orderNo,
                                    CurrentState = (int)CurrentState.未出去,
                                    UpdateTime = DateTime.Now,
                                    CreateTime = DateTime.Now,
                                    IsWork = false,
                                };
                                List<CommonJpTransferDetailEstimate> transferEsDetailsList = new List<CommonJpTransferDetailEstimate>();
                                List<CommonJpTransferDetailReality> transferReDetailsList = new List<CommonJpTransferDetailReality>();
                                foreach (var item in entity.detailList)
                                {


                                    var warehouseList = item.warehouseList;
                                    foreach (var warehouseItem in warehouseList)
                                    {

                                        string sendWarehouseId = warehouseItem.warehouseId;
                                        string count = warehouseItem.quantity;
                                        if (count != null && count != "0")
                                        {

                                            transferEsDetailsList.Add(new CommonJpTransferDetailEstimate()
                                            {
                                                Id = Guid.NewGuid().ToString(),
                                                EquipmentDetailId = item.equipmentDetailId,
                                                EquipmentSizeId = item.equipmentSizeId,
                                                EstimateId = transferApplyOrder.Id,
                                                EquipmentCount = int.Parse(count),


                                            });
                                            transferReDetailsList.Add(new CommonJpTransferDetailReality()
                                            {
                                                Id = Guid.NewGuid().ToString(),
                                                SizeId = item.equipmentSizeId,
                                                EquipmentDetailId = item.equipmentDetailId,
                                                RealityId = transferOrder.Id,
                                                State = (int)CurrentState.未出去,
                                                EstimateId = transferApplyOrder.Id,
                                                EquipmentCount = int.Parse(count),
                                                RealCount = 0,
                                                IsWork = false,
                                                WarehouseId = sendWarehouseId,
                                                InsideRealCount = 0,
                                            });
                                        }

                                    }


                                }
                                transferOrder.CommonJpTransferDetailRealities = transferReDetailsList;
                                transferApplyOrder.CommonJpTransferDetailEstimates = transferEsDetailsList;
                                bool transferResult = _unitOfWork.TransferApplyRepository.Insert(transferApplyOrder, false);
                                transferResult = _unitOfWork.TransferRepository.Insert(transferOrder, true);
                                if (transferResult)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }

                            case "fix":
                            case "scrap":

                                using (var scope = _unitOfWork.BeginTransaction())
                                {
                                    try
                                    {
                                        CommonJpFixReceiveApply fixQuery = _unitOfWork.FixReceiveApplyRepository.Get(p => p.OrderCode.Equals(orderNo));
                                        if (fixQuery != null)
                                        {
                                            scope.Rollback();
                                            return JsonManager.SimpleStatusResponse(ResultCode.ORDER_IS_EXIST);
                                        }

                                        if (type.Equals("fix"))
                                        {
                                            if (!orderNo.Contains("WX"))
                                            {
                                                scope.Rollback();
                                                return JsonManager.SimpleStatusResponse(ResultCode.ORDER_ERROR);
                                            }
                                        }
                                        else if (type.Equals("scrap"))
                                        {
                                            if (!orderNo.Contains("BF"))
                                            {
                                                scope.Rollback();
                                                return JsonManager.SimpleStatusResponse(ResultCode.ORDER_ERROR);
                                            }
                                        }

                                        CommonJpFixReceiveApply FixOrder = new CommonJpFixReceiveApply()
                                        {
                                            Action = type.Equals("fix") ? 0 : 1,
                                            ApplyTime = DateTime.Now,
                                            CreateTime = DateTime.Now,
                                            ApprovalResult = (int)ApproveType.已审批,
                                            WarehouseId = warehouseId,
                                            Id = Guid.NewGuid().ToString(),
                                            OrgId = entity.orgId,
                                            OrderCode = orderNo,
                                            ApprovalTime = DateTime.Now,
                                            CurrentState = 1

                                        };
                                        CommonJpFixUseless useless = new CommonJpFixUseless()
                                        {
                                            Id = Guid.NewGuid().ToString(),
                                            Action = type.Equals("fix") ? 0 : 1,
                                            CreateTime = DateTime.Now,
                                            CreateUser = "",
                                            FixTime = DateTime.Now,
                                            EstimateId = FixOrder.Id,
                                            OrderCode = orderNo,
                                            OrgId = entity.orgId,
                                            WarehouseId = warehouseId,
                                            CommonJpFixUselessDetails = new List<CommonJpFixUselessDetail>(),
                                            CurrentState = 1
                                        };


                                        List<CommonJpFixReceiveApplyDetail> FixDetailsList = new List<CommonJpFixReceiveApplyDetail>();
                                        foreach (var item in entity.detailList)
                                        {
                                            if (string.IsNullOrEmpty(item.supplierId) || string.IsNullOrEmpty(item.epc))
                                            {
                                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                            }
                                            FixDetailsList.Add(new CommonJpFixReceiveApplyDetail()
                                            {
                                                Id = Guid.NewGuid().ToString(),
                                                EquipmentId = item.equipmentDetailId,
                                                SizeId = item.equipmentSizeId,
                                                OrderId = FixOrder.Id,
                                                Quantity = 1,
                                                SupplierId = item.supplierId,
                                                Epc = item.epc ?? "",
                                                State = 1
                                            });

                                            var tmpEntity = new CommonJpFixUselessDetail()
                                            {
                                                Id = Guid.NewGuid().ToString(),
                                                OrderId = useless.Id,
                                                Quantity = item.Quantity,
                                                State = 1,
                                                Epc = item.epc,
                                                EquipmentId = item.equipmentDetailId,
                                                SizeId = item.equipmentSizeId,
                                                SupplierId = item.supplierId,
                                                RealCount = 1
                                            };
                                            useless.CommonJpFixUselessDetails.Add(tmpEntity);
                                            var tmpState = GenerateState(tmpEntity.Epc, useless.Action, tmpEntity.Id, useless.WarehouseId, useless.OrgId, useless.OrderCode);
                                            if (tmpState != null)
                                            {
                                                tmpEntity.CommonJpEquipmentStates.Add(tmpState);
                                            }
                                        }
                                        //反写库存状态
                                        var updateList = UpdateInvState(FixDetailsList.Select(x => x.Epc).ToList(), type.Equals("fix") ? 0 : 1, orderNo);
                                        FixOrder.CommonJpFixReceiveApplyDetails = FixDetailsList;
                                        bool FixResult = _unitOfWork.FixReceiveApplyRepository.Insert(FixOrder, false);
                                        FixResult = _unitOfWork.FixUselessRepository.Insert(useless, true);

                                        if (FixResult)
                                        {
                                            //上报数据
                                            var pushData = new List<object>();
                                            pushData.AddRange(updateList.Select(x => new
                                            {
                                                editType = "UPDATE",
                                                id = x.Id,
                                                supplierCode = x.SupplierCode,
                                                epc = x.Epc,
                                                policeCode = x.PoliceCode,
                                                equipmentCode = x.EquipmentCode,
                                                warehouseCode = x.WarehouseCode,
                                                cabinetId = x.CabinetId,
                                                equipmentSizecode = x.EquipmentSizecode,
                                                equipmentLocation = x.EquipmentLocation,
                                                currentState = x.CurrentState,
                                                createTime = x.CreateTime,
                                                warrantyCycle = x.WarrantyCycle,
                                                repairCycle = x.RepairCycle,
                                                productTime = x.ProductTime,
                                                useTime = x.UseTime,
                                                price = x.Price,
                                                isBorrowed = x.IsBorrowed,
                                                lostFlag = x.LostFlag,
                                                inventoryState = x.InventoryState,
                                                isInBox = x.IsInBox,
                                                boxMarkId = x.BoxMarkId,
                                                instantiationState = x.InstantiationState,
                                            }));

                                            var httpResult = _httpHelper.GetHtml(new HttpItem()
                                            {
                                                URL = _httpHelper._centerServerAddress + "/api/Print/EditInventory",
                                                Encoding = Encoding.UTF8,
                                                Method = "POST",
                                                ContentType = "application/json",
                                                Timeout = 5000,
                                                Postdata = _httpHelper.CreatePostData(JsonConvert.SerializeObject(new
                                                {
                                                    data = pushData
                                                }))
                                            });
                                            if (httpResult.StatusCode != HttpStatusCode.OK || string.IsNullOrEmpty(httpResult.Html))
                                            {
                                                scope.Rollback();
                                                return JsonManager.SimpleCustResponse("远端上报数据失败");
                                            }
                                            var obj = JsonConvert.DeserializeObject<JmpBaseResponse<object>>(httpResult.Html);
                                            if (obj.code != "10000")
                                            {
                                                scope.Rollback();
                                                return JsonManager.SimpleCustResponse(obj.msg);
                                            }
                                            scope.Commit();

                                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                        }
                                        else
                                        {
                                            scope.Rollback();
                                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                        }
                                    }
                                    catch (Exception e)
                                    {
                                        scope.Rollback();
                                        throw;
                                    }
                                }
                            default:
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                    }

                }
                catch (Exception ex)
                {

                    _logger.LogError("Inventory/CreateOrder 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }
        /// <summary>
        /// 更新库存信息
        /// </summary>
        /// <param name="epcList"></param>
        /// <param name="action"></param>
        /// <param name="orderCode"></param>
        private List<CommonJpEquipmentInventory> UpdateInvState(List<string> epcList, int action, string orderCode)
        {
            if (epcList == null || epcList.Count == 0)
            {
                _logger.LogError($"[维修报废]未找到维修报废明细，{orderCode}");
            }
            var wz_info = _unitOfWork.DbContext.CommonJpEquipmentInventories.Where(t => epcList.Contains(t.Epc)).ToList();
            if (wz_info != null && wz_info.Count() > 0)
            {
                wz_info.ForEach(x =>
                {

                    x.CurrentState = action == 0 ? 4 : 6;
                    _unitOfWork.EquipmentInventoryRepository.Update(x, false);
                });

            }
            return wz_info;
        }

        /// <summary>
        /// 状态更新
        /// </summary>
        /// <param name="epc"></param>
        /// <param name="action"></param>
        /// <param name="detailId"></param>
        /// <param name="warehouseId"></param>
        /// <param name="orgId"></param>
        /// <returns></returns>
        private CommonJpEquipmentState GenerateState(string epc, int action, string detailId, string warehouseId, string orgId, string ordercode)
        {
            var invInfo = _unitOfWork.DbContext.CommonJpEquipmentInventories.FirstOrDefault(t => t.Epc.Contains(epc));
            if (invInfo != null)
            {
                CommonJpEquipmentState equState = new CommonJpEquipmentState()
                {
                    Id = Guid.NewGuid().ToString(),
                    State = 2,
                    OrderCode = ordercode,
                    ActionState = action == 0 ? 6 : 7,
                    EqiupmentId = invInfo.Id,
                    OutInState = 0,
                    FixId = detailId,
                    WarehouseId = warehouseId,
                    CreateTime = DateTime.Now,
                    OrgId = orgId,
                    OutTime = DateTime.Now,
                    HistorySafeLevel = invInfo.SafeLevel
                };
                return equState;
            }
            else { return default; }

        }

        /// <summary>
        /// 获取出入库表单的详细信息
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "id":"当前出入库表单Id", 
        ///         "type":1 0调拨, 1采购，2借用，3领用，4调拨,5,维修,6报废,7跨库借用，9跨库归还
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetInOutDetail")]
        public async Task<HttpResponseMessage> GetInOutDetail([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = string.Empty;
                        int type = -1;
                        if (!String.IsNullOrEmpty(entity.id))
                        {
                            id = entity.id;

                        }
                        if (!String.IsNullOrEmpty(entity.type))
                        {

                            type = int.Parse(entity.type);
                        }
                        var param = new string[] { };
                        if (type.Equals(0) || type.Equals(4))//调拨
                        {
                            param = new string[] {
                            "Size" ,
                            "EquipmentDetail" ,
                            "Supplier" ,
                            "Reality" ,
                            "Warehouse"
                            //"Reality.ReceiveWarehouse" ,
                             };
                            var dataList = _unitOfWork.TransferDetailRealityRepository.GetList(p => p.EstimateId.Equals(id) && p.Reality.ActionType.Equals(0), null, false, param).ToList();
                            if (type.Equals(0))
                            {
                                var newDataList = dataList.Select(p => new
                                {
                                    p.State,
                                    p.Id,
                                    p.Size?.SizeName,
                                    equipmentDetailName = p.EquipmentDetail?.Name,
                                    p.EquipmentCount,
                                    p.RealCount,
                                    supplierName = p.Supplier?.Name,
                                    warehouseName = p.Warehouse?.Name
                                    //sendWarehouseName =p.Warehouse,
                                    //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                                });
                                return JsonManager.ReturnSuccessResponse(newDataList);
                            }
                            else
                            {
                                var newDataList = dataList.Select(p => new
                                {
                                    p.State,
                                    p.Id,
                                    p.Size?.SizeName,
                                    equipmentDetailName = p.EquipmentDetail?.Name,
                                    p.EquipmentCount,
                                    realCount = p.InsideRealCount,
                                    supplierName = p.Supplier?.Name,
                                    warehouseName = p.Warehouse?.Name
                                    //sendWarehouseName =p.Warehouse,
                                    //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                                });
                                return JsonManager.ReturnSuccessResponse(newDataList);
                            }


                        }
                        else if (type.Equals(1))//采购
                        {
                            param = new string[] {
                            "Size" ,
                            "Equipment" ,
                            "Supplier" ,
                            "CommonJpPurchaseEpcs",
                            "Order",
                            "Order.ProcessCurrent"
                             };

                            var dataList = _unitOfWork.PurchaseDetailRepository.GetList(p => p.OrderId.Equals(id) && p.Order.ProcessCurrent.FlowState.Equals("true"), null, false, param).ToList();
                            var newDataList = dataList.Select(p => new
                            {
                                p.State,
                                p.Id,
                                p.Size?.SizeName,
                                equipmentDetailName = p.Equipment?.Name,
                                equipmentCount = p.Quantity,
                                p.RealCount,
                                supplierName = p.CommonJpPurchaseEpcs?.FirstOrDefault()?.SupplierName,
                                repairCycle = p.CommonJpPurchaseEpcs?.FirstOrDefault()?.RepairCycle,
                                warrantyCycle = p.CommonJpPurchaseEpcs?.FirstOrDefault()?.WarrantyCycle,
                                useDirection = p.CommonJpPurchaseEpcs?.FirstOrDefault()?.UseDirection,
                                useTime = p.CommonJpPurchaseEpcs?.FirstOrDefault()?.UseTime?.ToString("yyyy-MM-dd"),
                                productTime = p.CommonJpPurchaseEpcs?.FirstOrDefault()?.ProductTime?.ToString("yyyy-MM-dd"),
                                price = p.CommonJpPurchaseEpcs?.FirstOrDefault()?.Price
                                //sendWarehouseName =p.Warehouse,
                                //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                            });
                            return JsonManager.ReturnSuccessResponse(newDataList);
                        }

                        else if (type.Equals(2))//借用
                        {
                            param = new string[] {
                            "Size" ,
                            "EquipmentDetail" ,
                            "Supplier" ,

                             };

                            var dataList = _unitOfWork.BorrowReturnDetailRealityRepository.GetList(p => p.EstimateId.Equals(id) && p.Reality.ActionType.Equals(0), null, false, param).ToList();
                            var newDataList = dataList.Select(p => new
                            {
                                p.State,
                                p.Id,
                                p.Size?.SizeName,
                                equipmentDetailName = p.EquipmentDetail?.Name,
                                equipmentCount = p.EquipmentCount,
                                p.RealCount,
                                supplierName = p.Supplier?.Name
                                //sendWarehouseName =p.Warehouse,
                                //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                            });
                            return JsonManager.ReturnSuccessResponse(newDataList);
                        }
                        else if (type.Equals(3))//领用
                        {
                            param = new string[] {
                            "Size" ,
                            "EquipmentDetail" ,
                            "Supplier" ,

                             };

                            var dataList = _unitOfWork.BorrowReturnDetailRealityRepository.GetList(p => p.EstimateId.Equals(id) && p.Reality.ActionType.Equals(1), null, false, param).ToList();
                            var newDataList = dataList.Select(p => new
                            {
                                p.State,
                                p.Id,
                                p.Size?.SizeName,
                                equipmentDetailName = p.EquipmentDetail?.Name,
                                equipmentCount = p.EquipmentCount,
                                p.RealCount,
                                supplierName = p.Supplier?.Name
                                //sendWarehouseName =p.Warehouse,
                                //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                            });
                            return JsonManager.ReturnSuccessResponse(newDataList);
                        }
                        else if (type.Equals(5) || type.Equals(6))//维修
                        {
                            param = new string[] {
                            "Size" ,
                            "Equipment" ,
                            "Supplier" ,

                             };

                            var dataList = _unitOfWork.FixUselessDetailRepository.GetList(p => p.OrderId.Equals(id), null, false, param).ToList();
                            var newDataList = dataList.GroupBy(p => p.SizeId).Select(p => new
                            {

                                p.FirstOrDefault().State,
                                p.FirstOrDefault().Id,
                                p.FirstOrDefault().Size?.SizeName,
                                equipmentDetailName = p.FirstOrDefault().Equipment?.Name,
                                equipmentCount = dataList.Count(),
                                RealCount = dataList.Where(p => p.RealCount > 0).Count(),
                                supplierName = p.FirstOrDefault().Supplier?.Name
                                //sendWarehouseName =p.Warehouse,
                                //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                            });
                            return JsonManager.ReturnSuccessResponse(newDataList);
                        }
                        else if (type.Equals(7) || type.Equals(8) || type.Equals(9) || type.Equals(10))//借用、归还
                        {
                            param = new string[] {
                            "Size" ,
                            "EquipmentDetail" ,
                            "Supplier" ,
                            "Reality",
                            "Warehouse"
                             };
                            Expression<Func<CommonJpTransferDetailReality, bool>> expression = p => p.EstimateId.Equals(id);
                            if (type.Equals(type.Equals(7) || type.Equals(8)))
                            {
                                expression = LambdaExtensions.And(expression, t => t.Reality.ActionType.Equals(1));
                            }
                            else if (type.Equals(type.Equals(9) || type.Equals(10)))
                            {
                                expression = LambdaExtensions.And(expression, t => t.Reality.ActionType.Equals(2));
                            }
                            var dataList = _unitOfWork.TransferDetailRealityRepository.GetList(expression, null, false, param).ToList();
                            if (type.Equals(7) || type.Equals(9))
                            {
                                var newDataList = dataList.Select(p => new
                                {
                                    p.State,
                                    p.Id,
                                    p.Size?.SizeName,
                                    equipmentDetailName = p.EquipmentDetail?.Name,
                                    p.EquipmentCount,
                                    p.RealCount,
                                    supplierName = p.Supplier?.Name,
                                    warehouseName = p.Warehouse?.Name
                                    //sendWarehouseName =p.Warehouse,
                                    //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                                });
                                return JsonManager.ReturnSuccessResponse(newDataList);

                            }
                            else
                            {
                                var newDataList = dataList.Select(p => new
                                {
                                    p.State,
                                    p.Id,
                                    p.Size?.SizeName,
                                    equipmentDetailName = p.EquipmentDetail?.Name,
                                    p.EquipmentCount,
                                    realCount = p.InsideRealCount,
                                    supplierName = p.Supplier?.Name,
                                    warehouseName = p.Warehouse?.Name
                                    //sendWarehouseName =p.Warehouse,
                                    //receiveWarehouseName= p.Reality?.ReceiveWarehouse?.Name
                                });
                                return JsonManager.ReturnSuccessResponse(newDataList);
                            }


                        }

                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }


                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取全部出入库表单
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "wareHouseId":"当前仓库Id", 
        ///         "outInState" : 0:出库,1入库
        ///         "type":1  1采购，2借用，3领用，4调拨,5,维修,6报废,7跨库借用，9跨库归还
        ///         "currentState":""0 已完成，1未完成
        ///         "startTime":"",
        ///         "endTime":"",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllInOutList")]
        public async Task<HttpResponseMessage> GetAllInOutList([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        string startTime = entity.startTime;
                        string endTime = entity.endTime;
                        if (size == 0)
                        {
                            size = 10;
                        }
                        string warehouseID = string.Empty;
                        string currentState = entity.currentState;
                        //var warehouse = _unitOfWork.WarehouseRepository.Get(p => p.Id.Equals(warehouseID)).Orgization;//当前仓库信息
                        Expression<Func<ViewOutinOrder, bool>> expression = t => t.Id != null;//0,调拨出库 1采购，2借用，3领用，4调拨入库,5,维修,6报废
                        if (!String.IsNullOrEmpty(entity.wareHouseId))
                        {
                            warehouseID = entity.wareHouseId;

                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseId.Equals(warehouseID));
                        }
                        if (!String.IsNullOrEmpty(currentState))//0 已完成，1未完成
                        {
                            if (currentState.Equals("0"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(2) || (t.CurrentState.Equals(1) && (t.Type.Equals(1) || t.Type.Equals(3) || t.Type.Equals(6))));
                            }
                            else if (currentState.Equals("1"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(0) || (t.CurrentState == null && t.Type.Equals(1)) || (t.CurrentState.Equals(1) && (t.Type.Equals(0) || t.Type.Equals(2) || t.Type.Equals(4) || t.Type.Equals(5) || t.Type.Equals(7) || t.Type.Equals(8) || t.Type.Equals(9) || t.Type.Equals(10) || t.Type.Equals(11))));
                            }

                        }
                        if (!String.IsNullOrEmpty(entity.outInState))//出入库状态
                        {
                            int outInState = int.Parse(entity.outInState);
                            if (outInState.Equals(0))//出库
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(0) || (t.Type.Equals(5) && (t.CurrentState.Equals(0) ||
                                t.CurrentState.Equals(2) || t.CurrentState.Equals(1))) || (t.Type.Equals(2) && (t.CurrentState.Equals(0) ||
                                t.CurrentState.Equals(2) || t.CurrentState.Equals(1))) || t.Type.Equals(6) || t.Type.Equals(3) || t.Type.Equals(7) || t.Type.Equals(9));
                            }
                            else if (outInState.Equals(1))//入库
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(4) || t.Type.Equals(1) || (t.Type.Equals(2) && (
                                t.CurrentState.Equals(2) || t.CurrentState.Equals(1))) || (t.Type.Equals(5) && (
                                t.CurrentState.Equals(2) || t.CurrentState.Equals(1))) || t.Type.Equals(8) || t.Type.Equals(10) || t.Type.Equals(11));
                            }
                        }
                        if (!string.IsNullOrEmpty(startTime) && !string.IsNullOrEmpty(endTime))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => t.CreateTime >= Convert.ToDateTime(startTime) && t.CreateTime <= Convert.ToDateTime(endTime));
                        }
                        if (!String.IsNullOrEmpty(entity.type))//业务选择
                        {
                            int type = int.Parse(entity.type);
                            if (type.Equals(4))//调拨
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(4) || t.Type.Equals(0));
                            }
                            else if (type.Equals(7))//跨库借用
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(7) || t.Type.Equals(8));
                            }
                            else if (type.Equals(9))//跨库借用
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(9) || t.Type.Equals(10));
                            }
                            else if (type.Equals(2))//跨库借用
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(2) || t.Type.Equals(11));
                            }
                            else
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(Convert.ToInt32(type)));
                            }

                        }
                        if (string.IsNullOrEmpty(warehouseID))
                        {
                            var content = new
                            {
                                totalElements = 0,
                                content = new List<string> { },
                            };
                            return JsonManager.ReturnSuccessResponse(content);
                        }
                        else
                        {
                            var outInOrders = _unitOfWork.ViewOutInOrderRepository.GetPage(expression, "-CreateTime", page, size, false);
                            //var orderList= outInOrders.totalElements
                            var content = new
                            {
                                outInOrders?.totalElements,
                                content = outInOrders.content.Select(t => new
                                {
                                    t.Id,
                                    t.Order,
                                    t.Type,
                                    t.WarehouseId,
                                    t.WarehouseName,
                                    t.CurrentState,
                                    t.CreateTime,
                                    StateType = (t.CurrentState.Equals(2) || t.CurrentState.Equals(11) || t.CurrentState.Equals(3) || (t.CurrentState.Equals(1) && (t.Type.Equals(1) || t.Type.Equals(3) || t.Type.Equals(6)))) ? "0" :
                                t.CurrentState.Equals(0) || (t.CurrentState == null && t.Type.Equals(1)) || (t.CurrentState.Equals(1) && (t.Type.Equals(0) || t.Type.Equals(2) || t.Type.Equals(11) || t.Type.Equals(4) || t.Type.Equals(5) || t.Type.Equals(7) || t.Type.Equals(8) || t.Type.Equals(9) || t.Type.Equals(10)))
                                ? "1" : null
                                })
                            };
                            return JsonManager.ReturnSuccessResponse(content);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取仓库下全部的装备列表
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseId":"仓库Id", 
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllInventory")]
        public async Task<HttpResponseMessage> GetAllInventory([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string warehouseId = entity.warehouseId;



                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "SupplierCodeNavigation" ,
                       "EquipmentCodeNavigation"

                    };

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => t.WarehouseCode == warehouseId && t.CurrentState.Equals(0);

                        var equipmentInvs = _unitOfWork.EquipmentInventoryRepository.GetList(expression, null, false, param);

                        var data = equipmentInvs.Select(p => new
                        {
                            equipmentDetailName = p.EquipmentCodeNavigation.Name,
                            equipmentDetailId = p.EquipmentCode,
                            epc = p.Epc,
                            sizeName = p.EquipmentSizecodeNavigation.SizeName,
                            sizeId = p.EquipmentSizecode,
                            Id = p.Id
                        }).ToList();




                        return JsonManager.ReturnSuccessResponse(data);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取仓库下全部的装备
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseId":"仓库Id", 
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllInventoryList")]
        public async Task<HttpResponseMessage> GetAllInventoryList([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string warehouseId = entity.warehouseId;



                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "SupplierCodeNavigation" ,
                       "EquipmentCodeNavigation"

                    };

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => t.WarehouseCode == warehouseId && t.CurrentState.Equals(0) && t.InventoryState != "loss";

                        var equipmentInvs = _unitOfWork.EquipmentInventoryRepository.GetList(expression, null, false, param);

                        var data = equipmentInvs.AsEnumerable().GroupBy(p => p.EquipmentCode).Select(p => new
                        {
                            detailCount = equipmentInvs.Where(t => t.EquipmentCode.Equals(p.Key)).Count(),
                            equipmentDetailName = p.FirstOrDefault().EquipmentCodeNavigation.Name,
                            equipmentDetailId = p.FirstOrDefault().EquipmentCode,
                            sizeData = equipmentInvs.Where(t => t.EquipmentCode.Equals(p.Key)).AsEnumerable().GroupBy(t => t.EquipmentSizecode).Select(t => new
                            {
                                sizeName = t.FirstOrDefault().EquipmentSizecodeNavigation.SizeName,
                                sizeId = t.FirstOrDefault().EquipmentSizecode,
                                sizeCount = equipmentInvs.Where(x => x.EquipmentSizecode.Equals(t.Key)).Count(),
                            })

                        }).ToList();




                        return JsonManager.ReturnSuccessResponse(data);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 遗失物资操作
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "id":""//物资ID
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("SetLostEquipment")]
        //[NoSign]
        public async Task<HttpResponseMessage> SetLostEquipment([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        string id = entity.id;
                        var equipment = _unitOfWork.EquipmentInventoryRepository.Get(p => p.Id.Equals(id));
                        if (equipment != null)
                        {
                            equipment.CurrentState = 6;
                            using (var scope = _unitOfWork.BeginTransaction())
                            {
                                try
                                {
                                    var result = _unitOfWork.EquipmentInventoryRepository.Update(equipment);
                                    if (result)
                                    {
                                        //上报数据
                                        var httpResult = _httpHelper.GetHtml(new HttpItem()
                                        {
                                            URL = _httpHelper._centerServerAddress + "/api/Print/EditInventory",
                                            Encoding = Encoding.UTF8,
                                            Method = "POST",
                                            ContentType = "application/json",
                                            Timeout = 5000,
                                            Postdata = _httpHelper.CreatePostData(JsonConvert.SerializeObject(new
                                            {
                                                data = new List<object>()
                                                {
                                                    new
                                                    {
                                                        editType = "UPDATE",
                                                        id = equipment.Id,
                                                        supplierCode = equipment.SupplierCode,
                                                        epc = equipment.Epc,
                                                        policeCode = equipment.PoliceCode,
                                                        equipmentCode = equipment.EquipmentCode,
                                                        warehouseCode = equipment.WarehouseCode,
                                                        cabinetId = equipment.CabinetId,
                                                        equipmentSizecode = equipment.EquipmentSizecode,
                                                        equipmentLocation = equipment.EquipmentLocation,
                                                        currentState = equipment.CurrentState,
                                                        createTime = equipment.CreateTime,
                                                        warrantyCycle = equipment.WarrantyCycle,
                                                        repairCycle = equipment.RepairCycle,
                                                        productTime = equipment.ProductTime,
                                                        useTime = equipment.UseTime,
                                                        price = equipment.Price,
                                                        isBorrowed = equipment.IsBorrowed,
                                                        lostFlag = equipment.LostFlag,
                                                        inventoryState = equipment.InventoryState,
                                                        isInBox = equipment.IsInBox,
                                                        boxMarkId = equipment.BoxMarkId,
                                                        instantiationState = equipment.InstantiationState,
                                                    }
                                                }
                                            }))
                                        });
                                        if (httpResult.StatusCode != HttpStatusCode.OK || string.IsNullOrEmpty(httpResult.Html))
                                        {
                                            scope.Rollback();
                                            return JsonManager.SimpleCustResponse("远端上报数据失败");
                                        }
                                        var obj = JsonConvert.DeserializeObject<JmpBaseResponse<object>>(httpResult.Html);
                                        if (obj.code != "10000")
                                        {
                                            scope.Rollback();
                                            return JsonManager.SimpleCustResponse(obj.msg);
                                        }
                                        scope.Commit();
                                        return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                    }
                                    else
                                    {
                                        scope.Rollback();
                                        return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                                    }
                                }
                                catch (Exception e)
                                {
                                    scope.Rollback();
                                    throw;
                                }
                            }
                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.EQUIPMENT_ERROR);
                        }


                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 根据仓库获取全部出库的装备
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "orgId":""//组织机构ID,
        ///         "policeName":""//警员姓名
        ///         "equipmentID":""//物资ID
        ///         "sizeID":""//物资号型ID
        ///         "page":"",
        ///         "size":""
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetLostEquipments")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetLostEquipments([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        if (size == 0)
                        {
                            size = 10;
                        }
                        string orgId = entity.orgId;
                        string name = entity.policeName;
                        string equipmentID = entity.equipmentID;
                        string sizeID = entity.sizeID;
                        var param = new string[] {
                       "WarehouseCodeNavigation" ,
                       "Cabinet",
                       "CommonJpEquipmentStates",
                       "EquipmentSizecodeNavigation",
                       "EquipmentCodeNavigation",
                       "SupplierCodeNavigation",
                       "Cabinet.BaseJpPolicemen",
                       "CommonJpEquipmentStates.Police",
                       "Shelf"

                    };
                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => t.Id != null && !t.LostFlag.Equals(true);
                        expression = LambdaExtensions.AndAlso(expression, p => (p.Cabinet.OrganizationId.Equals(orgId) ||
                        p.WarehouseCodeNavigation.OrgizationId.Equals(orgId)) &&
                        (p.CurrentState.Equals(1) || p.CurrentState.Equals(4) || p.CurrentState.Equals(7) ||
                        p.CurrentState.Equals(8) || p.CurrentState.Equals(14) || p.CurrentState.Equals(15) ||
                        p.CurrentState.Equals(17) || p.CurrentState.Equals(2)
                        ));

                        if (!string.IsNullOrEmpty(name))
                        {
                            expression = LambdaExtensions.AndAlso(expression, p => p.Cabinet.BaseJpPolicemen.Where(t => t.Name.Contains(name)).Count() > 0 || p.CommonJpEquipmentStates.OrderByDescending(t => t.CreateTime).FirstOrDefault().Police.Name.Contains(name));
                        }
                        if (!string.IsNullOrEmpty(equipmentID))
                        {
                            expression = LambdaExtensions.AndAlso(expression, p => p.EquipmentCode.Equals(equipmentID));
                        }
                        if (!string.IsNullOrEmpty(sizeID))
                        {
                            expression = LambdaExtensions.AndAlso(expression, p => p.EquipmentSizecode.Equals(sizeID));
                        }
                        var equipment = _unitOfWork.EquipmentInventoryRepository.GetPage(expression, "EquipmentCode,CurrentState,-OutTime", page, size, false, param);

                        var content = new
                        {
                            equipment?.totalElements,
                            content = equipment.content.Select(p => new
                            {
                                p.Epc,
                                p.Id,
                                sizeName = p.EquipmentSizecodeNavigation?.SizeName,
                                name = p.EquipmentCodeNavigation?.Name,
                                supplierName = p.SupplierCodeNavigation?.Name,
                                p.CurrentState,

                                shelfId = p.ShelfId,
                                shelfCode = p.Shelf?.Code,
                                shelfName = p.Shelf?.Name,
                                shelf = p.Shelf == null ? null : new
                                {
                                    p.Shelf.Id,
                                    p.Shelf.CreateTime,
                                    p.Shelf.Name,
                                    p.Shelf.Code,
                                    p.Shelf.Type,
                                    p.Shelf.WarehouseId,
                                    p.Shelf.Row,
                                    p.Shelf.Column,
                                    p.Shelf.Range,
                                },
                                range = p.ShelfRange,
                                row = p.ShelfRow,
                                column = p.ShelfColumn,
                                safeLevel = p.SafeLevel,
                                p.Address,
                                warehouseName = p.WarehouseCodeNavigation?.Name,
                                warehouseId = p.WarehouseCodeNavigation?.Id,
                                cabinetName = p.CabinetId == null ? "" : p.Cabinet?.CabinetNum + "-" + p.Cabinet?.CabinetChildNum,
                                userName = string.IsNullOrEmpty(p.CabinetId) ? (p.CommonJpEquipmentStates.Count() == 0 ? ""
                                : p.CommonJpEquipmentStates?.OrderByDescending(t => t.CreateTime).FirstOrDefault()?.Police?.Name)
                                : p.Cabinet?.BaseJpPolicemen?.FirstOrDefault()?.Name,
                                supplierId = p.SupplierCode,
                                equipmentSizeId = p.EquipmentSizecode,
                                equipmentId = p.EquipmentCode,
                            })
                        };
                        return JsonManager.ReturnSuccessResponse(content);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 根据EPC集合获取物资信息
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///         "EPCList": [{
        ///            "epc": ""
        ///             }
        ///             ],
        ///             "warehouseId":"仓库ID" 
        ///            }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetInvListByEpc")]
        public async Task<HttpResponseMessage> GetInvListByEpc([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        _logger.LogDebug("GetInvListByEpc/盘库校验:" + jdata.ToString());
                        var entity = JsonManager.GetJsonEntity(jdata);
                        if (entity == null)
                        {
                            return JsonManager.SimpleCustResponse($"model is null");
                        }

                        else
                        {
                            string warehouseId = entity.warehouseId;

                            List<string> Rfids = new List<string>();
                            foreach (var item in entity.EPCList)
                            {
                                string rfidItem = item.epc;

                                if (!string.IsNullOrEmpty(rfidItem))
                                {
                                    Rfids.Add(rfidItem);
                                }

                            }

                            string epc = entity.epc ?? "";


                            var wz_info = _unitOfWork.DbContext.CommonJpEquipmentInventories//本仓库物资
                                .Include(t => t.SupplierCodeNavigation)
                                .Include(t => t.EquipmentCodeNavigation)
                                .Include(t => t.EquipmentCodeNavigation.Equipment)
                                .Include(t => t.EquipmentSizecodeNavigation)
                                .Include(t => t.WarehouseCodeNavigation)
                                .Include(t => t.Shelf)
                                .Where(t => Rfids.Contains(t.Epc)

                                && (t.LostFlag == false || t.LostFlag == null)).ToList();
                            //var wz_infoOut = _unitOfWork.DbContext.CommonJpEquipmentInventories//非本仓库物资
                            //    .Include(t => t.SupplierCodeNavigation)
                            //    .Include(t => t.EquipmentCodeNavigation)
                            //    .Include(t => t.EquipmentCodeNavigation.Equipment)
                            //    .Include(t => t.EquipmentSizecodeNavigation)
                            //    .Include(t => t.WarehouseCodeNavigation)
                            //    .Include(t => t.Shelf)
                            //    .Where(t => Rfids.Contains(t.Epc)
                            //    && t.WarehouseCode != warehouseId
                            //    &&(t.CurrentState==0)
                            //    && (t.LostFlag == false || t.LostFlag == null)).ToList();
                            if (wz_info.Count() <= 0)
                            {
                                return JsonManager.SimpleCustResponse("未找到物资信息");
                            }
                            else
                            {
                                var newList = wz_info.Select(t => new
                                {
                                    t.Id,
                                    t.EquipmentCode,
                                    equipmentName = t.EquipmentCodeNavigation?.Name,
                                    typeId = t.EquipmentCodeNavigation.EquipmentId,
                                    typeName = t.EquipmentCodeNavigation?.Equipment?.Name,
                                    equipmentSizeId = t.EquipmentSizecode,
                                    equipmentSizeName = t.EquipmentSizecodeNavigation?.SizeName,
                                    supplierId = t.SupplierCode,
                                    supplierName = t.SupplierCodeNavigation?.Name,
                                    epc = t.Epc,
                                    warehouseId = t?.WarehouseCode,
                                    warehouseName = t?.WarehouseCodeNavigation?.Name,
                                    t.CurrentState,
                                    t.InventoryState,

                                    shelfId = t.ShelfId,
                                    shelfCode = t.Shelf?.Code,
                                    shelfName = t.Shelf?.Name,
                                    shelf = t.Shelf == null ? null : new
                                    {
                                        t.Shelf.Id,
                                        t.Shelf.CreateTime,
                                        t.Shelf.Name,
                                        t.Shelf.Code,
                                        t.Shelf.Type,
                                        t.Shelf.WarehouseId,
                                        t.Shelf.Row,
                                        t.Shelf.Column,
                                        t.Shelf.Range,
                                    },
                                    range = t.ShelfRange,
                                    row = t.ShelfRow,
                                    column = t.ShelfColumn,
                                    safeLevel = t.SafeLevel,
                                    t.Address
                                }).ToList();

                                //var outList = wz_infoOut.Select(t => new
                                //{
                                //    t.Id,
                                //    t.EquipmentCode,
                                //    equipmentName = t.EquipmentCodeNavigation?.Name,
                                //    typeId = t.EquipmentCodeNavigation.EquipmentId,
                                //    typeName = t.EquipmentCodeNavigation?.Equipment?.Name,
                                //    equipmentSizeId = t.EquipmentSizecode,
                                //    equipmentSizeName = t.EquipmentSizecodeNavigation?.SizeName,
                                //    supplierId = t.SupplierCode,
                                //    supplierName = t.SupplierCodeNavigation?.Name,
                                //    epc = t.Epc,
                                //    warehouseId = t?.WarehouseCode,
                                //    warehouseName = t?.WarehouseCodeNavigation?.Name,
                                //    t.CurrentState,
                                //    t.InventoryState,

                                //    shelfId = t.ShelfId,
                                //    shelfCode = t.Shelf?.Code,
                                //    shelfName = t.Shelf?.Name,
                                //    shelf = t.Shelf == null ? null : new
                                //    {
                                //        t.Shelf.Id,
                                //        t.Shelf.CreateTime,
                                //        t.Shelf.Name,
                                //        t.Shelf.Code,
                                //        t.Shelf.Type,
                                //        t.Shelf.WarehouseId,
                                //        t.Shelf.Row,
                                //        t.Shelf.Column,
                                //        t.Shelf.Range,
                                //    },
                                //    range = t.ShelfRange,
                                //    row = t.ShelfRow,
                                //    column = t.ShelfColumn,
                                //    safeLevel = t.SafeLevel,
                                //    t.Address
                                //}).ToList();
                                //newList.AddRange(outList);
                                return JsonManager.ReturnSuccessResponse(newList);
                            }

                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                    }
                }
                catch (Exception ex)
                {
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 根据EPC集合获取物资信息(全部物资状态）
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///         "EPCList": [{
        ///            "epc": ""
        ///             }
        ///             ] 
        ///            }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetInvListByEpcAllState")]
        public async Task<HttpResponseMessage> GetInvListByEpcAllState([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        if (entity == null)
                        {
                            return JsonManager.SimpleCustResponse($"model is null");
                        }

                        else
                        {

                            List<string> Rfids = new List<string>();
                            foreach (var item in entity.EPCList)
                            {
                                string rfidItem = item.epc;

                                if (!string.IsNullOrEmpty(rfidItem))
                                {
                                    Rfids.Add(rfidItem);
                                }

                            }

                            string epc = entity.epc ?? "";


                            var wz_info = _unitOfWork.DbContext.CommonJpEquipmentInventories
                                .Include(t => t.SupplierCodeNavigation)
                                .Include(t => t.EquipmentCodeNavigation)
                                .Include(t => t.EquipmentSizecodeNavigation)
                                .Include(t => t.WarehouseCodeNavigation)
                                .Include(t => t.Shelf)
                                .Where(t => Rfids.Contains(t.Epc)
                                //&& t.CurrentState == 0
                                && (t.LostFlag == false || t.LostFlag == null || t.InventoryState != "loss")).ToList();
                            if (wz_info.Count() <= 0)
                            {
                                return JsonManager.SimpleCustResponse("未找到物资信息");
                            }
                            else
                            {
                                var newList = wz_info.Select(t => new
                                {
                                    t.Id,
                                    t.EquipmentCode,
                                    equipmentName = t.EquipmentCodeNavigation?.Name,
                                    typeId = t.EquipmentCodeNavigation.EquipmentId,
                                    typeName = t.EquipmentCodeNavigation?.Equipment?.Name,
                                    equipmentSizeId = t.EquipmentSizecode,
                                    equipmentSizeName = t.EquipmentSizecodeNavigation?.SizeName,
                                    supplierId = t.SupplierCode,
                                    supplierName = t.SupplierCodeNavigation?.Name,
                                    epc = t.Epc,
                                    warehouseId = t?.WarehouseCode,
                                    warehouseName = t?.WarehouseCodeNavigation?.Name,
                                    t.CurrentState,
                                    t.InventoryState,

                                    shelfId = t.ShelfId,
                                    shelfCode = t.Shelf?.Code,
                                    shelfName = t.Shelf?.Name,
                                    shelf = t.Shelf == null ? null : new
                                    {
                                        t.Shelf.Id,
                                        t.Shelf.CreateTime,
                                        t.Shelf.Name,
                                        t.Shelf.Code,
                                        t.Shelf.Type,
                                        t.Shelf.WarehouseId,
                                        t.Shelf.Row,
                                        t.Shelf.Column,
                                        t.Shelf.Range,
                                    },
                                    range = t.ShelfRange,
                                    row = t.ShelfRow,
                                    column = t.ShelfColumn,
                                    safeLevel = t.SafeLevel,
                                    t.Address
                                });

                                return JsonManager.ReturnSuccessResponse(newList);
                            }

                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                    }
                }
                catch (Exception ex)
                {
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 根据EPC获取物资出入流水记录
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "epc":""
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetInfoByEpc")]
        //[NoSignAttribute]
        public async Task<HttpResponseMessage> GetInfoByEpc([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        //int.TryParse(entity.page, out int page);
                        //int.TryParse(entity.size, out int size);
                        //if (size == 0)
                        //{
                        //    size = 10;
                        //}
                        string epc = entity.epc;

                        var param = new string[]
                    {
                            "EquipmentSizecodeNavigation",
                            "EquipmentCodeNavigation",
                            "WarehouseCodeNavigation",
                            "WarehouseCodeNavigation.Orgization",
                            "PoliceCodeNavigation",
                            "SupplierCodeNavigation",
                            "Cabinet",
                            "CommonJpEquipmentStates",
                            "CommonJpEquipmentStates.Police",
                            "Shelf",
                    };
                        var epcList = new List<CommonJpEquipmentInventory>();

                        var equipment = _unitOfWork.EquipmentInventoryRepository.GetList(p => p.Epc.Equals(epc), null, false, param).FirstOrDefault();
                        if (equipment == null)
                        {
                            param = new string[] {
                                "CommonJpEquipmentInventories",
                                "CommonJpEquipmentInventories.EquipmentSizecodeNavigation" ,
                                "CommonJpEquipmentInventories.EquipmentCodeNavigation" ,
                                "CommonJpEquipmentInventories.WarehouseCodeNavigation" ,
                                "CommonJpEquipmentInventories.WarehouseCodeNavigation.Orgization" ,
                                "CommonJpEquipmentInventories.PoliceCodeNavigation" ,
                                "CommonJpEquipmentInventories.SupplierCodeNavigation",
                                "CommonJpEquipmentInventories.Cabinet",
                                "CommonJpEquipmentInventories.CommonJpEquipmentStates",
                                "CommonJpEquipmentInventories.CommonJpEquipmentStates.Police",
                                "CommonJpEquipmentInventories.Shelf",
                            };
                            var boxMark = _unitOfWork.EquipmentBoxMarkRepository
                                .GetList(p => p.Epc.Equals(epc), null, false, param).FirstOrDefault();

                            if (boxMark == null || boxMark.CommonJpEquipmentInventories.Count == 0)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.EPC_ERROR);
                            }
                            else
                            {
                                epcList.AddRange(boxMark.CommonJpEquipmentInventories);
                            }
                        }
                        else
                        {
                            epcList.Add(equipment);
                        }

                        return JsonManager.ReturnSuccessResponse(epcList.Select(x => new
                        {
                            x.Id,
                            x.Epc,
                            x.UseDirection,
                            x.WarrantyCycle,
                            x.ProductTime,
                            x.UseTime,
                            x.CurrentState,

                            shelfId = x.ShelfId,
                            shelfCode = x.Shelf?.Code,
                            shelfName = x.Shelf?.Name,
                            shelf = x.Shelf == null ? null : new
                            {
                                x.Shelf.Id,
                                x.Shelf.CreateTime,
                                x.Shelf.Name,
                                x.Shelf.Code,
                                x.Shelf.Type,
                                x.Shelf.WarehouseId,
                                x.Shelf.Row,
                                x.Shelf.Column,
                                x.Shelf.Range,
                            },
                            range = x.ShelfRange,
                            row = x.ShelfRow,
                            column = x.ShelfColumn,
                            safeLevel = x.SafeLevel,
                            x.Address,
                            sizeName = x.EquipmentSizecodeNavigation?.SizeName,
                            x.EquipmentSizecode,
                            equipmentName = x.EquipmentCodeNavigation?.Name,
                            x.EquipmentCode,
                            warehouseName = x.WarehouseCodeNavigation?.Name,
                            x.WarehouseCode,
                            x.WarehouseCodeNavigation?.Orgization?.Name,
                            x.WarehouseCodeNavigation?.OrgizationId,
                            policeName = x.PoliceCodeNavigation?.Name,
                            x.PoliceCode,
                            supplierName = x.SupplierCodeNavigation?.Name,
                            x.SupplierCode,
                            cabinetID = x.Cabinet?.Id,
                            CabinetChildNum = x.Cabinet?.CabinetChildNum + "号柜",
                            x.Cabinet?.CabinetNum,
                            x.Price,
                            x.RepairCycle,
                            x.IsInBox,
                            useCount = x.CommonJpEquipmentStates.Where(p => p.OutInState == 0 && p.State != 2).Count(),
                            fixCount = x.CommonJpEquipmentStates.Where(p => p.ActionState == 6).Count(),

                            stateList = x.CommonJpEquipmentStates?.Select(t => new
                            {
                                t.Id,
                                t.OutInState,
                                t.CreateTime,
                                t.ActionState,
                                policeName = t.Police?.Name,
                                equipmentName = x.EquipmentCodeNavigation?.Name,
                                sizeName = x.EquipmentSizecodeNavigation?.SizeName,
                                x.WarehouseCodeNavigation.Name
                            }).OrderByDescending(p => p.CreateTime)

                        }));

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 根据EPC获取物资出入流水记录
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "id":""
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetInfoByID")]
        //[NoSignAttribute]
        public async Task<HttpResponseMessage> GetInfoByID([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity.id;
                        string epc = entity.epc;




                        var param = new string[] {
                        "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization" ,
                       "PoliceCodeNavigation" ,
                       "SupplierCodeNavigation",
                       "Cabinet",
                       "CommonJpEquipmentStates",
                       "CommonJpEquipmentStates.Police",
                       "CommonJpEquipmentStates.Warehouse",
                       "Shelf",
                    };
                        CommonJpEquipmentInventory equipment;

                        if (string.IsNullOrEmpty(id) && !string.IsNullOrEmpty(epc))
                        {
                            equipment = _unitOfWork.EquipmentInventoryRepository.GetList(p => p.Epc.Equals(epc), null, false, param).FirstOrDefault();
                        }
                        else
                        {
                            equipment = _unitOfWork.EquipmentInventoryRepository.GetList(p => p.Id.Equals(id), null, false, param).FirstOrDefault();
                        }
                        if (equipment == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.EPC_ERROR);
                        }
                        var equ = new
                        {
                            equipment.InventoryState,
                            equipment.Id,
                            equipment.Epc,
                            equipment.UseDirection,
                            equipment.WarrantyCycle,
                            equipment.ProductTime,
                            equipment.UseTime,
                            equipment.CurrentState,

                            shelfId = equipment.ShelfId,
                            shelfCode = equipment.Shelf?.Code,
                            shelfName = equipment.Shelf?.Name,
                            shelf = equipment.Shelf == null ? null : new
                            {
                                equipment.Shelf.Id,
                                equipment.Shelf.CreateTime,
                                equipment.Shelf.Name,
                                equipment.Shelf.Code,
                                equipment.Shelf.Type,
                                equipment.Shelf.WarehouseId,
                                equipment.Shelf.Row,
                                equipment.Shelf.Column,
                                equipment.Shelf.Range,
                            },
                            range = equipment.ShelfRange,
                            row = equipment.ShelfRow,
                            column = equipment.ShelfColumn,
                            safeLevel = equipment.SafeLevel,
                            equipment.Address,
                            sizeName = equipment.EquipmentSizecodeNavigation?.SizeName,
                            equipment.EquipmentSizecode,
                            equipmentName = equipment.EquipmentCodeNavigation?.Name,
                            equipment.EquipmentCode,
                            warehouseName = equipment.WarehouseCodeNavigation?.Name,
                            equipment.WarehouseCode,
                            equipment.WarehouseCodeNavigation?.Orgization?.Name,
                            equipment.WarehouseCodeNavigation?.OrgizationId,
                            policeName = equipment.PoliceCodeNavigation?.Name,
                            equipment.PoliceCode,
                            supplierName = equipment.SupplierCodeNavigation?.Name,
                            equipment.SupplierCode,
                            cabinetID = equipment.Cabinet?.Id,
                            CabinetChildNum = equipment.Cabinet?.CabinetChildNum + "号柜",
                            equipment.Cabinet?.CabinetNum,
                            equipment.Price,
                            equipment.RepairCycle,
                            useCount = equipment.CommonJpEquipmentStates.Where(p => p.OutInState == 0 && p.State != 2).Count(),
                            fixCount = equipment.CommonJpEquipmentStates.Where(p => p.ActionState == 6).Count(),
                            stateList = equipment.CommonJpEquipmentStates?.Select(t => new
                            {
                                t.Id,
                                t.OutInState,
                                t.CreateTime,
                                t.ActionState,
                                t.Warehouse?.Name,
                                policeName = t.Police?.Name,
                                equipmentName = equipment.EquipmentCodeNavigation?.Name,
                                sizeName = equipment.EquipmentSizecodeNavigation?.SizeName,
                                picUrl = t.PicUrl
                            }).OrderByDescending(p => p.CreateTime)
                        };


                        return JsonManager.ReturnSuccessResponse(equ);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 根据装备包实例id获取物资出入流水记录
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "id":""
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetBagInfoByID")]
        public async Task<HttpResponseMessage> GetBagInfoByID([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity.id;
                        string epc = entity.epc;




                        var param = new string[]
                        {
                            "Bag",
                            "Warehouse",
                            "Warehouse.Orgization",
                            "CommonJpEquipmentStates",
                            "CommonJpEquipmentStates.Warehouse",
                            "CommonJpEquipmentStates.Police",
                        };
                        CommonJpBagInventory bagInventory;

                        if (string.IsNullOrEmpty(id) && !string.IsNullOrEmpty(epc))
                        {
                            bagInventory = _unitOfWork.BagInventoryRepository.GetList(p => p.Epc.Equals(epc), null, false, param).FirstOrDefault();
                        }
                        else
                        {
                            bagInventory = _unitOfWork.BagInventoryRepository.GetList(p => p.Id.Equals(id), null, false, param).FirstOrDefault();
                        }
                        if (bagInventory == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.EPC_ERROR);
                        }
                        var equ = new
                        {
                            bagInventory.Id,
                            bagInventory.Epc,
                            //bagInventory.UseDirection,
                            //bagInventory.WarrantyCycle,
                            //bagInventory.ProductTime,
                            //bagInventory.UseTime,
                            bagInventory.State,

                            equipmentName = bagInventory.Bag?.Name,
                            BagId = bagInventory.BagId,
                            warehouseName = bagInventory.Warehouse?.Name,
                            WarehouseCode = bagInventory.WarehouseId,
                            bagInventory.Warehouse?.Orgization?.Name,
                            bagInventory.Warehouse?.OrgizationId,

                            useCount = bagInventory.CommonJpEquipmentStates.Where(p => p.OutInState == 0 && p.State != 2).Count(),
                            fixCount = bagInventory.CommonJpEquipmentStates.Where(p => p.ActionState == 6).Count(),
                            stateList = bagInventory.CommonJpEquipmentStates?.Where(x => string.IsNullOrEmpty(x.ParentId)).Select(t => new
                            {
                                t.Id,
                                t.OutInState,
                                t.CreateTime,
                                t.ActionState,
                                t.Warehouse?.Name,
                                policeName = t.Police?.Name,
                                equipmentName = bagInventory.Bag?.Name,
                                picUrl = t.PicUrl
                            }).OrderByDescending(p => p.CreateTime)
                        };


                        return JsonManager.ReturnSuccessResponse(equ);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取单警柜的出入库列表
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "parentId"：出入库记录父ID
        ///         "page":
        ///         "size":
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetCabinetList")]
        //[NoSignAttribute]
        public async Task<HttpResponseMessage> GetCabinetList([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        if (size == 0)
                        {
                            size = 10;
                        }
                        var param = new string[] {
                        "Eqiupment" ,
                       "Eqiupment.EquipmentCodeNavigation" ,
                       "Eqiupment.EquipmentSizecodeNavigation" ,
                       "Org" ,
                       "Police" ,
                       "CabinetIdIntoNavigation",
                       "CabinetIdOutNavigation"
                    };
                        string parentId = entity.parentId;
                        var equipmentStates = _unitOfWork.EquipmentStateRepository.GetListAsync(p => p.ParentId.Equals(parentId), null, false, param).Result;
                        var data = equipmentStates.AsEnumerable().GroupBy(p => p.Eqiupment.EquipmentSizecode).Select(t => new
                        {
                            count = t.Count(),
                            orgCode = t.FirstOrDefault().Org.FindCode.PadRight(6, '0'),
                            cabinetId = t.FirstOrDefault().CabinetIdInto == null ? t.FirstOrDefault().CabinetIdOut : t.FirstOrDefault().CabinetIdInto,
                            cabinetName = t.FirstOrDefault().CabinetIdIntoNavigation == null ? (t.FirstOrDefault().CabinetIdOutNavigation?.CabinetNum + "-" + t.FirstOrDefault().CabinetIdOutNavigation?.CabinetChildNum.ToString()) : (t.FirstOrDefault().CabinetIdIntoNavigation?.CabinetNum + "-" + t.FirstOrDefault().CabinetIdIntoNavigation?.CabinetChildNum.ToString()),
                            equipmentName = t.FirstOrDefault().Eqiupment?.EquipmentCodeNavigation?.Name,
                            sizeName = string.IsNullOrEmpty(t.FirstOrDefault().EqiupmentId) ? "" : t.FirstOrDefault().Eqiupment?.EquipmentSizecodeNavigation?.SizeName ?? "",
                            t.FirstOrDefault().OutInState,
                            equipmentId = t.FirstOrDefault().EqiupmentId,
                            t.FirstOrDefault().ActionState,
                            t.FirstOrDefault().ParentId,
                            policeName = t.FirstOrDefault().PoliceId == null ? "" : t.FirstOrDefault().Police?.Name,
                            safeLevel = t.FirstOrDefault().HistorySafeLevel ?? t.FirstOrDefault().Eqiupment?.SafeLevel
                        });
                        var content = new
                        {
                            totalElements = data.Count(),
                            content = data.Skip(page * size).Take(size).ToList()
                        };
                        return JsonManager.ReturnSuccessResponse(content);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取全部出入库记录
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "orgId":"当前组织机构Id", 
        ///         "warehouseId":"仓库Id", 
        ///         "equipmentDetail":"装备名", 
        ///         "outInState" : 
        ///         "actionState":1  1采购，2借用，3领用，4调拨,
        ///         "startTime":  起始时间
        ///         "endTime":1  结束时间,
        ///         "epc":  EPC,
        ///         "actionUserName":  "操作用户名",
        ///         "cabinetId":  "单警柜id",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllInOutHistory")]
        //[NoSignAttribute]
        public async Task<HttpResponseMessage> GetAllInOutHistory([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        if (size == 0)
                        {
                            size = 10;
                        }


                        var param = new string[] {
                       "Eqiupment" ,
                       "Eqiupment.EquipmentCodeNavigation" ,
                       "Eqiupment.EquipmentSizecodeNavigation" ,
                       "Warehouse" ,
                       "Org" ,
                       "Police" ,
                       "CabinetIdIntoNavigation",
                       "CabinetIdOutNavigation",
                       "Purchase",
                       "Purchase.Order",
                       "Borrow",
                       "Borrow.Estimate",
                       "Transfer",
                       "Transfer.Estimate",
                       "Fix",
                       "Fix.Order",
                       "BagInventory",
                       "BagInventory.Bag",
                       "BagInventory.CommonJpEquipmentInventories",
                       "BagInventory.CommonJpEquipmentInventories.EquipmentCodeNavigation",
                       "BagInventory.CommonJpEquipmentInventories.EquipmentSizecodeNavigation",
                    };

                        string OrgId = entity.orgId;
                        string actionUserName = entity.actionUserName;
                        string cabinetId = entity.cabinetId;

                        Expression<Func<CommonJpEquipmentState, bool>> expression = t => t.OrgId.Equals(OrgId);
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;

                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseId.Equals(warehouseId) && t.ActionState != 5);

                        }
                        if (!String.IsNullOrEmpty(entity.outInState))
                        {
                            string outInState = entity.outInState;

                            expression = LambdaExtensions.AndAlso(expression, t => t.OutInState.Equals(Convert.ToInt32(outInState)));
                        }
                        if (!String.IsNullOrEmpty(entity.epc))
                        {
                            string epc = entity.epc;
                            epc = epc.ToUpper();
                            expression = LambdaExtensions.AndAlso(expression, t => t.Eqiupment.Epc.Contains(epc));
                        }
                        if (!String.IsNullOrEmpty(entity.actionState))
                        {
                            string actionState = entity.actionState;

                            if (actionState.Equals("5"))
                            {
                                //单警柜
                                expression = LambdaExtensions.AndAlso(expression, t => string.IsNullOrEmpty(t.EqiupmentId));
                                if (!string.IsNullOrEmpty(actionUserName))
                                {
                                    expression = expression.AndAlso(x => x.Police.Name.Contains(actionUserName));
                                }
                                if (!string.IsNullOrEmpty(cabinetId))
                                {
                                    expression = expression.AndAlso(x => x.CabinetIdInto == cabinetId || x.CabinetIdOut == cabinetId);
                                }
                            }
                            else
                            {
                                expression = LambdaExtensions.And(expression, t => t.EqiupmentId != null || t.BagInventoryId != null);
                            }

                            if (actionState.Equals("0"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => !t.ActionState.Equals(5));
                            }
                            else if (actionState.Equals("9"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.ActionState.Equals(9) || t.ActionState.Equals(10));
                            }
                            else if (actionState.Equals("17"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.ActionState.Equals(17) || t.ActionState.Equals(18));
                            }
                            else if (actionState.Equals("15"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.ActionState.Equals(15) || t.ActionState.Equals(16));
                            }
                            else if (actionState.Equals("13"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.ActionState.Equals(13) || t.ActionState.Equals(14));
                            }
                            else if (actionState.Equals("5"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.ActionState.Equals(5));
                            }
                            else
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => !t.ActionState.Equals(5) && t.ActionState.Equals(int.Parse(actionState)));
                            }

                        }
                        else if (String.IsNullOrEmpty(entity.actionState))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => !t.ActionState.Equals(5));
                        }
                        if (!String.IsNullOrEmpty(entity.equipmentDetail))
                        {
                            string equipmentDetail = entity.equipmentDetail;

                            expression = LambdaExtensions.AndAlso(expression, t => t.Eqiupment.EquipmentCodeNavigation.Name.Contains(equipmentDetail));
                        }

                        if (entity.startTime != null && (entity.endTime != null))
                        {
                            DateTime startTime = Convert.ToDateTime(entity.startTime);
                            DateTime endTime = Convert.ToDateTime(entity.endTime);
                            expression = LambdaExtensions.AndAlso(expression, t => t.CreateTime.Value.Date >= startTime && t.CreateTime.Value.Date <= endTime);
                        }
                        expression = LambdaExtensions.AndAlso(expression, t => string.IsNullOrEmpty(t.ErrorEpc) && string.IsNullOrEmpty(t.ParentId));
                        //var b = _unitOfWork.EquipmentStateRepository.GetList(p=>string.IsNullOrEmpty(p.ParentId) , null, false, param);
                        var equipmentStates = _unitOfWork.EquipmentStateRepository.GetPageAsync(expression, "-CreateTime", page, size, false, param).Result;
                        var data = equipmentStates?.content.Select(t => new
                        {
                            result = _unitOfWork.EquipmentStateRepository.GetList(p => p.EqiupmentId.Equals(t.EqiupmentId)).OrderByDescending(p => p.CreateTime).FirstOrDefault().Id.Equals(t.Id) ? true : false,
                            orgCode = t.Org.FindCode.PadRight(6, '0'),
                            order = t.OrderCode,

                            id = t.Id,
                            key = t.Id,
                            t.Note,
                            t.PicUrl,
                            errorEpc = t.ErrorEpc,
                            t.Eqiupment?.Epc,
                            cabinetId = t.CabinetIdInto == null ? t.CabinetIdOut : t.CabinetIdInto,
                            cabinetName = t.CabinetIdIntoNavigation == null ? (t.CabinetIdOutNavigation?.CabinetNum + "-" + t.CabinetIdOutNavigation?.CabinetChildNum.ToString()) : (t.CabinetIdIntoNavigation?.CabinetNum + "-" + t.CabinetIdIntoNavigation?.CabinetChildNum.ToString()),
                            equipmentName = string.IsNullOrEmpty(t.EqiupmentId) ? (string.IsNullOrEmpty(t.BagInventoryId) ? "" : t.BagInventory?.Bag?.Name) : t.Eqiupment?.EquipmentCodeNavigation?.Name,
                            sizeName = string.IsNullOrEmpty(t.EqiupmentId) ? "" : t.Eqiupment?.EquipmentSizecodeNavigation?.SizeName ?? "",
                            t.OutInState,
                            warehouseName = t.Warehouse?.Name ?? "",
                            orgId = t.OrgId,
                            equipmentId = t.EqiupmentId,
                            bagInventoryId = t.BagInventoryId,
                            orgName = t.OrgId == null ? "" : t.Org?.Name,
                            t.ActionState,
                            t.CreateTime,
                            t.PoliceId,
                            parentId = t.Id,
                            policeName = t.PoliceId == null ? "" : t.Police?.Name,
                            safeLevel = t.HistorySafeLevel ?? t.Eqiupment?.SafeLevel,
                            bagIncludeList = t.BagInventory?.CommonJpEquipmentInventories?.Select(x => new
                            {
                                equipmentId = x.Id,
                                key = x.Id + t.Id,
                                equipmentName = x.EquipmentCodeNavigation?.Name,
                                sizeName = x.EquipmentSizecodeNavigation?.SizeName,
                                safeLevel = x.SafeLevel,
                            })
                        });
                        var content = new
                        {
                            equipmentStates?.totalElements,
                            content = data
                        };
                        return JsonManager.ReturnSuccessResponse(content);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取仓库对应的近十条出入库记录
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {   
        ///         "orgId":"",
        ///         "warehouseId":"仓库Id",
        ///         "showNumber":"8",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllInOutTop10History")]
        //[NoSignAttribute]
        public async Task<HttpResponseMessage> GetAllInOutTop10History([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        var param = new string[] {
                       "Eqiupment" ,
                       "Eqiupment.EquipmentCodeNavigation" ,
                       "Eqiupment.EquipmentSizecodeNavigation" ,
                       "Warehouse" ,
                       "Org" ,
                       "Police" ,
                       "CabinetIdIntoNavigation",
                       "CabinetIdOutNavigation",
                       "Purchase",
                       "Purchase.Order",
                       "Borrow",
                       "Borrow.Estimate",
                       "Transfer",
                       "Transfer.Estimate",
                       "Fix",
                       "Fix.Order",
                       "BagInventory",
                       "BagInventory.Bag",
                    };
                        int showNumber = 8;
                        if (!String.IsNullOrEmpty(entity.showNumber))
                        {
                            showNumber = int.Parse(entity.showNumber);
                        }

                        var warehouseId = "";
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            warehouseId = entity.warehouseId;
                        }

                        var orgId = "";
                        if (!String.IsNullOrEmpty(entity.orgId))
                        {
                            orgId = entity.orgId;
                        }

                        Expression<Func<CommonJpEquipmentState, bool>> expression = t => (t.WarehouseId.Equals(warehouseId) || t.OrgId.Equals(orgId)) && t.ActionState != 5;
                        expression = LambdaExtensions.AndAlso(expression, t => string.IsNullOrEmpty(t.ErrorEpc) && string.IsNullOrEmpty(t.ParentId));
                        var equipmentStates = _unitOfWork.EquipmentStateRepository.GetList(expression, t => t.CreateTime, false, param);
                        var data = equipmentStates.Take(showNumber).ToList().Select(t => new
                        {

                            order = t.OrderCode,
                            id = t.Id,
                            t.Eqiupment?.Epc,
                            t.PicUrl,
                            equipmentName = string.IsNullOrEmpty(t.EqiupmentId) ?
                                (string.IsNullOrEmpty(t.BagInventoryId) ? "" : t.BagInventory?.Bag?.Name)
                                : t.Eqiupment?.EquipmentCodeNavigation?.Name,
                            sizeName = string.IsNullOrEmpty(t.EqiupmentId) ? "" : t.Eqiupment?.EquipmentSizecodeNavigation?.SizeName ?? "",
                            t.OutInState,
                            warehouseName = t.Warehouse?.Name ?? "",
                            equipmentId = t.EqiupmentId,
                            orgName = t.OrgId == null ? "" : t.Org?.Name,
                            t.ActionState,
                            t.CreateTime,
                            t.PoliceId,
                            parentId = t.Id,
                            policeName = t.PoliceId == null ? "" : t.Police?.Name,
                            safeLevel = t.HistorySafeLevel ?? t.Eqiupment?.SafeLevel
                        });
                        return JsonManager.ReturnSuccessResponse(data);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取单据中EPC详情
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "realityId":"单据子表ID",  
        ///         "type" : 业务类型，采购，借用，领用，调拨,跨库借用，跨库归还,维修，报废
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetOrderDetailEPC")]
        public async Task<HttpResponseMessage> GetOrderDetailEPC([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        Expression<Func<ViewOrderState, bool>> expression = t => t.Id != null;
                        string realityId = entity.realityId;
                        string type = entity.type;
                        var param = new string[] {
                       "Eqiupment"
                    };
                        if (string.IsNullOrEmpty(type))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }



                        switch (type)
                        {
                            case "采购":
                                var purDetailList = new List<string>();

                                var list = _unitOfWork.PurchaseDetailRepository.GetList(p => p.Id.Equals(realityId)).ToList();

                                foreach (var item in list)
                                {
                                    purDetailList.Add(item.Id);
                                }
                                var pubchaseList = _unitOfWork.EquipmentStateRepository.GetList(p => purDetailList.Contains(p.PurchaseId), null, false, param);
                                var purchaseInList = new
                                {
                                    state = "in",
                                    epcList = pubchaseList.Where(p => p.OutInState.Equals(1)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };
                                var purchaseOutList = new
                                {
                                    state = "out",
                                    epcList = pubchaseList.Where(p => p.OutInState.Equals(0)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };
                                var purchaseTotalResult = new
                                {
                                    purchaseInList,
                                    purchaseOutList
                                };
                                return JsonManager.ReturnSuccessResponse(purchaseTotalResult);

                            case "调拨":
                            case "跨库借用":
                            case "跨库归还":
                                var transferList = _unitOfWork.EquipmentStateRepository.GetList(p => p.TransferId.Equals(realityId), null, false, param);

                                var transferInList = new
                                {
                                    state = "in",
                                    epcList = transferList.Where(p => p.OutInState.Equals(1)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };
                                var transferOutList = new
                                {
                                    state = "out",
                                    epcList = transferList.Where(p => p.OutInState.Equals(0)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };
                                var transferTotalResult = new
                                {
                                    transferInList,
                                    transferOutList
                                };
                                return JsonManager.ReturnSuccessResponse(transferTotalResult);

                            case "维修":
                            case "报废":
                                var fixOrder = _unitOfWork.FixUselessDetailRepository.GetList(p => p.OrderId.Equals(realityId));

                                List<string> orderDetaillist = new List<string>();
                                foreach (var item in fixOrder)
                                {
                                    orderDetaillist.Add(item.Id);
                                }
                                var fixList = _unitOfWork.EquipmentStateRepository.GetList(p => orderDetaillist.Contains(p.FixId), null, false, param);

                                var fixListInList = new
                                {
                                    state = "out",
                                    epcList = fixList.Where(p => p.OutInState.Equals(0)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };

                                var fixListOutList = new
                                {
                                    state = "in",
                                    epcList = fixList.Where(p => p.OutInState.Equals(0)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };
                                var fixTotalResult = new
                                {
                                    fixListInList,
                                    fixListOutList
                                };
                                return JsonManager.ReturnSuccessResponse(fixTotalResult);

                            case "领用":
                            case "借用":
                                var borrowList = _unitOfWork.EquipmentStateRepository.GetList(p => p.BorrowId.Equals(realityId), null, false, param);

                                var borrowInList = new
                                {
                                    state = "in",
                                    epcList = borrowList.Where(p => p.OutInState.Equals(1)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };
                                var borrowOutList = new
                                {
                                    state = "out",
                                    epcList = borrowList.Where(p => p.OutInState.Equals(0)).Select(p => new { p.Eqiupment.Epc, time = p.CreateTime })
                                };
                                var borrowTotalResult = new
                                {
                                    borrowInList,
                                    borrowOutList
                                };
                                return JsonManager.ReturnSuccessResponse(borrowTotalResult);


                            default:
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取组织机构相关的单据
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "orgId":"组织机构Id", 
        ///         "state":"单据状态",已完成，未完成 
        ///         "type" : 业务类型，采购，借用，领用，调拨,跨库借用，跨库归还,维修，报废
        ///         "startTime":  起始时间
        ///         "endTime":1  结束时间
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetCurrentOrder")]
        public async Task<HttpResponseMessage> GetCurrentOrder([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        if (size == 0)
                        {
                            size = 10;
                        }
                        Expression<Func<ViewOrderState, bool>> expression = t => t.Id != null;
                        string state = entity.state;
                        string orgId = entity.orgId;
                        string type = entity.type;
                        if (string.IsNullOrEmpty(orgId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }


                        if (!String.IsNullOrEmpty(state))
                        {


                            expression = LambdaExtensions.AndAlso(expression, t => t.State.Equals(state));

                        }
                        if (!String.IsNullOrEmpty(type))
                        {

                            expression = LambdaExtensions.AndAlso(expression, t => t.Type.Equals(type));
                        }
                        if (entity.startTime != null && (entity.endTime != null))
                        {
                            DateTime startTime = Convert.ToDateTime(entity.startTime);
                            DateTime endTime = Convert.ToDateTime(entity.endTime);
                            expression = LambdaExtensions.AndAlso(expression, t => t.CreateTime.Value.Date >= startTime && t.CreateTime.Value.Date <= endTime);
                        }


                        var outInOrders = _unitOfWork.ViewOrderStateRepository.GetPage(expression, "-CreateTime", page, size, false);
                        var content = new
                        {
                            outInOrders?.totalElements,
                            content = outInOrders.content
                        };
                        return JsonManager.ReturnSuccessResponse(content);

                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 第三方/充电台出入库
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///            "orgId": "0320200512000808",
        ///            "warehouseId": "",
        ///            "orderNo": "",//订单号
        ///            "type": "类型",采购,调拨,借用,领用,维修,跨库借用,跨库归还
        ///            "outInState": "","当前出入库状态",//出库,入库
        ///            "policeId": "",//警员ID
        ///            "picUrl": "出入库图片路径",
        ///            "RFIDList": [{
        ///            "RFID": "A12C621596CDF8A02B18538A"
        ///             },{
        ///             "RFID": "A12C6215BCE9F863C2896E0A"
        ///            },{
        ///           "RFID": "A12C6215BCE9F863C284DF0A"
        ///            }] 
        ///            }
        ///           
        ///     
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("UploadRFIDDevelop")]
        //[NoSign]
        public async Task<HttpResponseMessage> UploadRFIDDevelop([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    _logger.LogDebug("Inventory/UploadRFIDDevelop第三方/充电台出入库:" + jdata.ToString());
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        if (entity == null)
                        {
                            return JsonManager.SimpleCustResponse($"model is null");
                        }

                        //第三方调用接口验证orgId正确性
                        if (!string.IsNullOrEmpty(entity._PUSHORG))
                        {
                            if (entity.orgId?.ToString() != entity._PUSHORG.ToString())
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.ORGANIZATION_ERROR);
                            }
                        }

                        else if (string.IsNullOrEmpty(entity.orgId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"orgId is require");
                        }

                        else if (string.IsNullOrEmpty(entity.warehouseId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"warehouseId is require");
                        }
                        var param = new string[] {




                    };

                        List<string> Rfids = new List<string>();//本次入库的所有EPC
                        foreach (var item in entity.RFIDList)
                        {
                            string rfidItem = item.RFID;

                            if (!string.IsNullOrEmpty(rfidItem))
                            {
                                Rfids.Add(rfidItem);
                            }

                        }
                        string picUrl = entity.picUrl;
                        string orderNo = entity.orderNo;
                        string type = entity.type;
                        string orgId = entity.orgId;
                        string policeId = entity.policeId;
                        string warehouseId = entity.warehouseId;
                        string outInState = entity.outInState;

                        var warehouseInfo = _unitOfWork.WarehouseRepository.Get(p => p.Id == warehouseId);
                        if (warehouseInfo.IsLocked == true)//检查仓库是否被锁
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.WAREHOUSE_LOCK);
                        }

                        List<string> ErrorEq = new List<string>();
                        List<CommonJpEquipmentInventory> UpdateEpc = new List<CommonJpEquipmentInventory>();//更新库存内
                        List<CommonJpEquipmentInventory> InsertEpc = new List<CommonJpEquipmentInventory>();//新增库存
                        if (outInState.Equals("出库"))//出库
                        {
                            List<CommonJpEquipmentInventory> OuterInvens = _unitOfWork.EquipmentInventoryRepository.GetList(p => p.WarehouseCode.Equals(warehouseId) && p.CurrentState != 5 && p.CurrentState != 6 && p.CurrentState.Equals(0) && Rfids.Contains(p.Epc)).ToList();//取出在库存中的库存信息，表示要出库的物资
                            if (type.Equals("库存调拨") || type.Equals("跨库借用") || type.Equals("跨库归还"))//0采购,1调拨,2借用,3领用,4维修,5跨库借用,6跨库归还
                            {
                                List<CommonJpEquipmentInventory> trOutRealEpcList = new List<CommonJpEquipmentInventory>();
                                List<CommonJpTransferDetailReality> tempReality = new List<CommonJpTransferDetailReality>();
                                param = new string[] {
                                "CommonJpTransferDetailRealities",
                                "CommonJpTransferDetailRealities.Estimate"
                                    //"Estimate"
                                };
                                List<CommonJpTransferDetailReality> UpdateTrReal = new List<CommonJpTransferDetailReality>();//需要更新的调拨数据
                                List<CommonJpTransferDetailReality> trOutRealDetailEpcList = new List<CommonJpTransferDetailReality>();
                                var transferOrders = _unitOfWork.TransferRepository.GetList(p => p.Order.Equals(orderNo), null, false, param).FirstOrDefault().CommonJpTransferDetailRealities.ToList();
                                foreach (var item in transferOrders)
                                {
                                    var outList = OuterInvens.Where(x => !trOutRealEpcList.Contains(x)
                                    && x.EquipmentCode.Equals(item.EquipmentDetailId) && x.CurrentState.Equals(0)).ToList();//找出本批物资中调拨出库的物资
                                    trOutRealEpcList.AddRange(outList);
                                    if (!trOutRealDetailEpcList.Contains(item) && item.State.Equals(0))
                                    {
                                        trOutRealDetailEpcList.Add(item);
                                    }

                                }
                                if (trOutRealDetailEpcList.Count > 0)
                                {
                                    if (type.Equals("库存调拨"))
                                    {
                                        tempReality = Extensions.ChannelExtension.TransferOut(trOutRealDetailEpcList, trOutRealEpcList, ref UpdateTrReal, tempReality, ref OuterInvens, ref UpdateEpc, picUrl, policeId, warehouseId, orgId);
                                    }
                                    else if (type.Equals("跨库归还"))
                                    {
                                        tempReality = Extensions.ChannelExtension.ReturnTransferOut(tempReality, trOutRealDetailEpcList, trOutRealEpcList, ref UpdateTrReal, ref OuterInvens, ref UpdateEpc, picUrl, policeId, warehouseId, orgId);
                                    }
                                    else if (type.Equals("跨库借用"))
                                    {
                                        tempReality = Extensions.ChannelExtension.BorrowTransferOut(tempReality, trOutRealDetailEpcList, trOutRealEpcList, ref UpdateTrReal, ref OuterInvens, ref UpdateEpc, picUrl, policeId, warehouseId, orgId);
                                    }
                                }
                                if (UpdateTrReal.Count > 0)
                                {
                                    _unitOfWork.TransferDetailRealityRepository.Update(UpdateTrReal, false);
                                }

                                if (UpdateEpc.Count > 0)//同时有插入和更新的数据
                                {

                                    _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc, true);
                                }


                            }
                            else if (type.Equals("借用") || type.Equals("领用"))
                            {
                                List<CommonJpEquipmentInventory> boOutRealEpcList = new List<CommonJpEquipmentInventory>();
                                List<CommonJpBorrowReturnDetailReality> tempReality = new List<CommonJpBorrowReturnDetailReality>();
                                param = new string[] {
                                "CommonJpBorrowReturnDetailRealities",
                                "CommonJpBorrowReturnDetailRealities.Estimate"

                                    };
                                List<CommonJpBorrowReturnDetailReality> UpdateBoReal = new List<CommonJpBorrowReturnDetailReality>();//需要更新的调拨数据
                                List<CommonJpBorrowReturnDetailReality> boOutRealDetailEpcList = new List<CommonJpBorrowReturnDetailReality>();
                                var borrowOrders = _unitOfWork.BorrowReturnRepository.GetList(p => p.OrderCode.Equals(orderNo), null, false, param).FirstOrDefault().CommonJpBorrowReturnDetailRealities.ToList();
                                if (borrowOrders.Count() > 0)
                                {
                                    if (type.Equals("借用"))
                                    {

                                        UpdateBoReal = Extensions.ChannelExtension.BorrowOut(borrowOrders, boOutRealDetailEpcList, boOutRealEpcList, ref OuterInvens, ref UpdateEpc, UpdateBoReal, picUrl, policeId, warehouseId, orgId);
                                    }
                                    else if (type.Equals("领用"))
                                    {

                                        UpdateBoReal = Extensions.ChannelExtension.Recipients(_unitOfWork, UpdateBoReal, borrowOrders, ref UpdateEpc, ref boOutRealEpcList, boOutRealDetailEpcList, ref OuterInvens, picUrl, policeId, warehouseId, orgId);
                                    }

                                }

                                if (UpdateBoReal.Count > 0)
                                {
                                    _unitOfWork.BorrowReturnDetailRealityRepository.Update(UpdateBoReal, false);
                                }

                                if (UpdateEpc.Count > 0)//同时有插入和更新的数据
                                {

                                    _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc, true);
                                }

                            }

                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else if (outInState.Equals("入库"))
                        {
                            param = new string[] {
                       "CommonJpEquipmentStates" ,
                       "CommonJpEquipmentStates.Borrow",
                       "CommonJpEquipmentStates.Borrow.Estimate",
                       "CommonJpEquipmentStates.Transfer",
                       "CommonJpEquipmentStates.Transfer.Estimate"

                    };

                            List<CommonJpEquipmentInventory> InsideInvens = _unitOfWork.EquipmentInventoryRepository.GetList(p => !p.CurrentState.Equals(0) && p.CurrentState != 5 && p.CurrentState != 6 && !p.CurrentState.Equals(7) && !p.CurrentState.Equals(3) && !p.CurrentState.Equals(10) && Rfids.Contains(p.Epc), null, false, param).ToList();//取出不在库存中的RFID信息,表示要入库的物资

                            #region 采购

                            if (type.Equals("采购"))
                            {
                                var paramPurchase = new string[] {
                       "Detail" ,
                       "Detail.Order",
                       "Detail.Order.CommonJpPurchaseDetails"
                            };
                                List<CommonJpPurchaseEpc> purchaseEpcList = _unitOfWork.PurchaseEpcsRepository.GetList(p => p.Detail.Order.WarehouseId.Equals(warehouseId) && p.CurrentState.Equals(1), null, false, paramPurchase).ToList();//本仓库的全部采购入库EPC

                                if (purchaseEpcList.Count > 0)
                                {

                                    List<CommonJpEquipmentInventory> OuterInvens = new List<CommonJpEquipmentInventory>();
                                    List<CommonJpPurchaseEpc> purchaseEpc = new List<CommonJpPurchaseEpc>();
                                    purchaseEpc = Extensions.ChannelExtension.Purchase(purchaseEpcList, OuterInvens, InsideInvens, Rfids, picUrl, policeId, warehouseId, orgId, ref InsertEpc, ref ErrorEq, _unitOfWork);
                                    if (purchaseEpc.Count > 0)
                                    {
                                        _unitOfWork.PurchaseEpcsRepository.Update(purchaseEpc, false);
                                    }
                                    if (InsertEpc.Count > 0)
                                    {
                                        _unitOfWork.EquipmentInventoryRepository.Insert(InsertEpc, true);
                                    }
                                }

                            }
                            #endregion
                            #region 维修
                            if (type.Equals("维修"))
                            {
                                List<CommonJpFixUselessDetail> UpdateFixReal = new List<CommonJpFixUselessDetail>();//需要更新的维修数据
                                param = new string[] {
                                  "Order" ,
                                 };
                                var fixInRealList = _unitOfWork.FixUselessDetailRepository.GetList(p => p.Order.WarehouseId.Equals(warehouseId) && (p.State.Equals(1)), null, false, param).ToList();//当前仓库的所有维修入库详情表

                                if (fixInRealList.Count > 0)
                                {
                                    UpdateFixReal = Extensions.ChannelExtension.Fix(UpdateFixReal, fixInRealList, ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                                }
                                if (UpdateFixReal.Count > 0)
                                {
                                    _unitOfWork.FixUselessDetailRepository.Update(UpdateFixReal, false);
                                }
                                if (UpdateEpc.Count > 0)
                                {
                                    _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc, true);
                                }
                            }
                            #endregion
                            #region 借用
                            if (type.Equals("借用"))
                            {
                                List<CommonJpBorrowReturnDetailReality> UpdateBoReal = new List<CommonJpBorrowReturnDetailReality>();//需要更新的借用领用数据
                                param = new string[] {
                       "Reality" ,
                       "Estimate",
                       "CommonJpEquipmentStates"
                         };
                                List<CommonJpBorrowReturnDetailReality> boIntoRealDetailEpcList = _unitOfWork.BorrowReturnDetailRealityRepository.GetList(p => p.State < 2 && p.Reality.WarehouseId.Equals(warehouseId), null, false, param).ToList();//找出本仓库的全部借用入库计划表


                                if (boIntoRealDetailEpcList.Count > 0)
                                {
                                    UpdateBoReal = Extensions.ChannelExtension.BorrowIn(ref UpdateEpc, UpdateBoReal, boIntoRealDetailEpcList, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                                }
                                if (UpdateBoReal.Count > 0)
                                {
                                    _unitOfWork.BorrowReturnDetailRealityRepository.Update(UpdateBoReal, false);
                                }
                                if (UpdateEpc.Count > 0)
                                {
                                    _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc, true);
                                }
                            }
                            #endregion
                            #region 调拨入库/借用跨库入库/归还入库
                            if (type.Equals("库存调拨") || type.Equals("跨库借用") || type.Equals("跨库归还"))
                            {
                                List<CommonJpTransferDetailReality> UpdateTrReal = new List<CommonJpTransferDetailReality>();//需要更新的调拨数据
                                param = new string[] {
                        "Reality" ,
                        "Estimate",
                        "CommonJpEquipmentStates"
                         };

                                List<CommonJpTransferDetailReality> trIntoRealDetailEpcList = new List<CommonJpTransferDetailReality>();
                                if (type.Equals("库存调拨"))
                                {
                                    trIntoRealDetailEpcList = _unitOfWork.TransferDetailRealityRepository.GetList(p => (p.State.Equals(1) || p.State.Equals(0)) && p.Reality.ActionType.Equals(0) && (p.Reality.ReceiveWarehouseId.Equals(warehouseId)
                       || p.WarehouseId.Equals(warehouseId)), null, false, param).ToList();//找出本仓库的全部调拨入库计划表
                                }
                                else if (type.Equals("跨库借用"))
                                {
                                    trIntoRealDetailEpcList = _unitOfWork.TransferDetailRealityRepository.GetList(p => (p.State.Equals(1) || p.State.Equals(0)) && p.Reality.ActionType.Equals(1) && (p.Reality.ReceiveWarehouseId.Equals(warehouseId)
                          || p.WarehouseId.Equals(warehouseId)), null, false, param).ToList();//找出本仓库的全部借用跨库入库计划表
                                }
                                else if (type.Equals("跨库归还"))
                                {
                                    trIntoRealDetailEpcList = _unitOfWork.TransferDetailRealityRepository.GetList(p => (p.State.Equals(1) || p.State.Equals(0)) && p.Reality.ActionType.Equals(2) && (p.Reality.ReceiveWarehouseId.Equals(warehouseId)
                         || p.WarehouseId.Equals(warehouseId) || p.ReceivedWarehouseId.Equals(warehouseId)), null, false, param).ToList();//找出本仓库的全部归还跨库入库计划表
                                }

                                if (InsideInvens.Count > 0 && trIntoRealDetailEpcList.Count() > 0)
                                {
                                    if (type.Equals("库存调拨"))
                                    {
                                        UpdateTrReal = Extensions.ChannelExtension.TransferIn(UpdateTrReal, trIntoRealDetailEpcList.ToList(), ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                                    }
                                    else if (type.Equals("跨库借用"))
                                    {
                                        UpdateTrReal = Extensions.ChannelExtension.BorrowTransferIn(UpdateTrReal, trIntoRealDetailEpcList.ToList(), ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                                    }
                                    else if (type.Equals("跨库归还"))
                                    {
                                        UpdateTrReal = Extensions.ChannelExtension.ReturnTransferIn(UpdateTrReal, trIntoRealDetailEpcList.ToList(), ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                                    }


                                }
                                if (UpdateTrReal.Count > 0)
                                {
                                    _unitOfWork.TransferDetailRealityRepository.Update(UpdateTrReal, false);
                                }
                                if (UpdateEpc.Count > 0)
                                {
                                    _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc, true);
                                }
                            }
                            #endregion


                        }




                        //#endregion

                        var resultEpc = UpdateEpc?.Select(p => new
                        {
                            p.Epc,
                            p.CurrentState
                        }).ToList();//更新的库存信息
                        var resultInsertEpc = InsertEpc?.Select(p => new
                        {
                            p.Epc,
                            p.CurrentState
                        }).ToList();//新增的库存信息
                        var resultErrorEpc = ErrorEq?.Select(p => new
                        {
                            Epc = p,
                            CurrentState = 11
                        }).ToList();//不在该仓库的物资信息


                        resultEpc.AddRange(resultInsertEpc);
                        resultEpc.AddRange(resultErrorEpc);

                        //跨仓库出库业务清空存放位置（只有update.currentState=1:调拨中 8:领用中 15跨仓库借用中 17跨仓库归还中）
                        if (outInState.Equals("出库"))
                        {
                            foreach (var inventory in UpdateEpc.Where(x => new[] { 1, 8, 15, 17 }.Contains(x.CurrentState)))
                            {
                                inventory.ShelfId = null;
                                inventory.ShelfRange = null;
                                inventory.ShelfRow = null;
                                inventory.ShelfColumn = null;
                            }
                            _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc);
                        }

                        //计算阈值
                        var sizeList = InsertEpc.Select(x => x.EquipmentSizecode).ToList();
                        sizeList.AddRange(UpdateEpc.Select(x => x.EquipmentSizecode).ToList());
                        _unitOfWork.CalcThreshold(sizeList.Distinct().ToList(), orgId, warehouseId);


                        return JsonManager.ReturnSuccessResponse(resultEpc);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }

                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// Rfid数据上报
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///            "orgId": "0320200512000808",
        ///            "warehouseId": "",
        ///            "policeId": "1",
        ///            "currentState": "当前出入库状态",//0出库,1入库
        ///            "picUrl": "出入库图片路径",
        ///            "RFIDList": [{
        ///            "RFID": "A12C621596CDF8A02B18538A"
        ///             },{
        ///             "RFID": "A12C6215BCE9F863C2896E0A"
        ///            },{
        ///           "RFID": "A12C6215BCE9F863C284DF0A"
        ///            }] ,
        ///            "BagRFIDList": [{
        ///            "RFID": "A12C621596CDF8A02B18538A"
        ///             },{
        ///             "RFID": "A12C6215BCE9F863C2896E0A"
        ///            },{
        ///           "RFID": "A12C6215BCE9F863C284DF0A"
        ///            }] 
        ///            }
        ///           
        ///     
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("UploadRFIDNew")]
        //[NoSignAttribute]

        public async Task<HttpResponseMessage> UploadRFIDNew([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    _logger.LogDebug("UploadRFIDNew/Rfid数据上报:" + jdata.ToString());
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);

                        if (entity == null)
                        {
                            return JsonManager.SimpleCustResponse($"model is null");
                        }
                        else if (string.IsNullOrEmpty(entity.orgId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"orgId is require");
                        }

                        else if (string.IsNullOrEmpty(entity.warehouseId ?? ""))
                        {
                            return JsonManager.SimpleCustResponse($"warehouseId is require");
                        }
                        var bagParam = new string[] {
                       "Bag",
                       "CommonJpEquipmentInventories"
                        };
                        var MissionParam = new string[] {
                            "CommonJpEquipmentStates",
                            "CommonJpEquipmentStates.BagInventory",
                            "CommonJpEquipmentStates.Eqiupment"
                        };
                        var param = new string[] {
                       "EquipmentCodeNavigation" ,
                       "EquipmentSizecodeNavigation" ,
                       "CommonJpEquipmentStates" ,
                       "CommonJpEquipmentStates.Borrow",
                       "CommonJpEquipmentStates.Borrow.Estimate",
                       "CommonJpEquipmentStates.Transfer",
                       "CommonJpEquipmentStates.Transfer.Estimate",
                       "BoxMark",
                       "Shelf"
                        };
                        var paramTr = new string[] {
                       "CommonJpEquipmentStates" ,
                       "Reality",
                       "Reality.CommonJpTransferDetailRealities"

                    };
                        var paramPurchase = new string[] {
                       "Detail" ,
                       "Detail.Order",
                       "Detail.Order.CommonJpPurchaseDetails"
                    };
                        var paramFix = new string[] {
                       "Order" ,
                    };
                        var parambR = new string[] {
                       "Reality" ,
                       "Estimate",
                       "CommonJpEquipmentStates"
                    };
                        var parambMission = new string[] {
                       "CommonJpEquipmentStates",
                       "CommonJpEquipmentStates.BagInventory",
                       "CommonJpEquipmentStates.Eqiupment",
                       "CommonJpMisssionChildren" ,
                       "CommonJpMisssionChildren.Model",
                       "CommonJpMisssionChildren.Model.CommonJpModelBags"
                    };

                        List<string> bagRfids = new List<string>();
                        List<string> Rfids = new List<string>();//本次入库的所有EPC
                        if (entity.RFIDList != null)
                        {
                            foreach (var item in entity.RFIDList)
                            {
                                string rfidItem = item.RFID;

                                if (!string.IsNullOrEmpty(rfidItem))
                                {
                                    Rfids.Add(rfidItem);
                                }

                            }
                        }
                        if (entity.BagRFIDList != null)
                        {
                            foreach (var item in entity.BagRFIDList)
                            {
                                string rfidItem = item.RFID;

                                if (!string.IsNullOrEmpty(rfidItem))
                                {
                                    bagRfids.Add(rfidItem);
                                }

                            }

                        }


                        string picUrl = entity.picUrl;
                        string orgId = entity.orgId;
                        string policeId = string.IsNullOrEmpty(entity.policeId) ? null : entity.policeId;
                        string warehouseId = entity.warehouseId;
                        string currentState = entity.currentState;
                        var warehouseInfo = _unitOfWork.WarehouseRepository.Get(p => p.Id == warehouseId);
                        if (warehouseInfo.IsLocked == true)//检查仓库是否被锁
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.WAREHOUSE_LOCK);
                        }

                        //过滤掉已报废数据
                        var inventoryList = _unitOfWork.EquipmentInventoryRepository.GetList(x => Rfids.Contains(x.Epc)).ToList();
                        var scrapEpc = inventoryList.Where(x => x.CurrentState == 6).ToList();
                        foreach (var epc in scrapEpc)
                        {
                            Rfids.Remove(epc.Epc);
                        }

                        //过滤装备包内散件数据

                        var bagsEpc = inventoryList.Where(x => !string.IsNullOrEmpty(x.BagInventoryId)).ToList();

                        foreach (var epc in bagsEpc)
                        {
                            Rfids.Remove(epc.Epc);
                        }

                        var MissionOut = _unitOfWork.MissionRepository.GetList(p => p.WarehouseId.Equals(warehouseId) && p.IsWork.Equals(true) && p.State.Equals(0), t => t.CreateTime, false, parambMission).FirstOrDefault();//当前仓库的任务出库详情表
                        var brRealOutList = _unitOfWork.BorrowReturnDetailRealityRepository.GetList(p => p.Reality.WarehouseId.Equals(warehouseId) && p.Reality.IsWork.Equals(true) && p.State.Equals(0) && p.Reality.ActionType.Equals(0), t => t.Reality.CreateTime, false, parambR);//当前仓库的所有借用出库详情表
                        var reRealList = _unitOfWork.BorrowReturnDetailRealityRepository.GetList(p => p.Reality.WarehouseId.Equals(warehouseId) && p.State.Equals(0) && p.Reality.ActionType.Equals(1) && p.Reality.IsWork.Equals(true), t => t.Reality.CreateTime, false, parambR);//当前仓库的所有领用出库详情表
                        var fixInRealList = _unitOfWork.FixUselessDetailRepository.GetList(p => p.Order.WarehouseId.Equals(warehouseId) && (p.State.Equals(1)), null, false, paramFix).ToList();//当前仓库的所有维修入库详情表
                        var trOutRealList = _unitOfWork.TransferDetailRealityRepository.GetList(p => p.WarehouseId.Equals(warehouseId) && p.Reality.ActionType.Equals(0) && p.IsWork.Equals(true) && p.State.Equals(0), t => t.Reality.UpdateTime, false, paramTr);//当前仓库的所有调拨出库详情表
                        var trboOutRealList = _unitOfWork.TransferDetailRealityRepository.GetList(p => p.WarehouseId.Equals(warehouseId) && p.Reality.ActionType.Equals(1) && p.IsWork.Equals(true) && p.State.Equals(0), t => t.Reality.UpdateTime, false, paramTr);//当前仓库的所有垮库借用出库详情表
                        var reboOutRealList = _unitOfWork.TransferDetailRealityRepository.GetList(p => p.WarehouseId.Equals(warehouseId) && p.Reality.ActionType.Equals(2) && p.IsWork.Equals(true) && p.State.Equals(0), t => t.Reality.UpdateTime, false, paramTr);//当前仓库的所有垮库归还出库详情表
                        var trIntoRealDetailEpcList = _unitOfWork.TransferDetailRealityRepository.GetList(p => (p.State.Equals(1) || p.State.Equals(0)) && p.Reality.ActionType.Equals(0) && (p.Reality.ReceiveWarehouseId.Equals(warehouseId) || p.WarehouseId.Equals(warehouseId)), null, false, parambR);//找出本仓库的全部调拨入库计划表
                        var boOutIntoRealDetailEpcList = _unitOfWork.TransferDetailRealityRepository.GetList(p => (p.State.Equals(1) || p.State.Equals(0)) && p.Reality.ActionType.Equals(1) && (p.Reality.ReceiveWarehouseId.Equals(warehouseId) || p.WarehouseId.Equals(warehouseId)), null, false, parambR);//找出本仓库的全部借用跨库入库计划表
                        var ReturnIntoRealDetailEpcList = _unitOfWork.TransferDetailRealityRepository.GetList(p => (p.State.Equals(1) || p.State.Equals(0)) && p.Reality.ActionType.Equals(2) && (p.Reality.ReceiveWarehouseId.Equals(warehouseId) || p.WarehouseId.Equals(warehouseId) || p.ReceivedWarehouseId.Equals(warehouseId)), null, false, parambR);//找出本仓库的全部归还跨库入库计划表

                        List<CommonJpEquipmentState> InsertState = new List<CommonJpEquipmentState>();//更新装备出入库流水
                        List<CommonJpBagInventory> UpdateBag = new List<CommonJpBagInventory>();//更新装备包
                        List<CommonJpEquipmentInventory> UpdateEpc = new List<CommonJpEquipmentInventory>();//更新库存内
                        List<CommonJpEquipmentInventory> InsertEpc = new List<CommonJpEquipmentInventory>();//新增库存
                        List<CommonJpBorrowReturnDetailReality> UpdateBoReal = new List<CommonJpBorrowReturnDetailReality>();//需要更新的借用领用数据
                        List<CommonJpTransferDetailReality> UpdateTrReal = new List<CommonJpTransferDetailReality>();//需要更新的调拨数据
                        List<CommonJpTransferDetailReality> UpdateboTrReal = new List<CommonJpTransferDetailReality>();//需要更新的借用垮库数据
                        List<CommonJpTransferDetailReality> UpdateReTrReal = new List<CommonJpTransferDetailReality>();//需要更新的归还垮库数据
                        List<CommonJpFixUselessDetail> UpdateFixReal = new List<CommonJpFixUselessDetail>();//需要更新的维修数据
                        List<string> ErrorEq = new List<string>();


                        List<CommonJpBorrowReturnDetailReality> boIntoRealDetailEpcList = _unitOfWork.BorrowReturnDetailRealityRepository.GetList(p => p.State < 2 && p.Reality.WarehouseId.Equals(warehouseId), null, false, parambR).ToList();//找出本仓库的全部借用入库计划表
                        List<CommonJpEquipmentInventory> MissionEpcList = new List<CommonJpEquipmentInventory>();//属于任务EPC的集合
                        List<CommonJpBagInventory> MissionBagsList = new List<CommonJpBagInventory>();//属于任务包的EPC集合
                        List<CommonJpEquipmentInventory> brOutRealEpcList = new List<CommonJpEquipmentInventory>();//属于借用出库的传过来的EPC集合
                        List<CommonJpEquipmentInventory> reRealEpcList = new List<CommonJpEquipmentInventory>();//属于领用出库的传过来的EPC集合
                        List<CommonJpEquipmentInventory> trOutRealEpcList = new List<CommonJpEquipmentInventory>();//属于调拨出库的传过来的EPC集合
                        List<CommonJpEquipmentInventory> trOutBoEpcList = new List<CommonJpEquipmentInventory>();//属于跨库借用出库的传过来的EPC集合
                        List<CommonJpEquipmentInventory> reOutBoEpcList = new List<CommonJpEquipmentInventory>();//属于跨库归还出库的传过来的EPC集合
                        List<CommonJpMisssionChild> missionChild = new List<CommonJpMisssionChild>();//属于任务表的模块表
                        List<CommonJpBorrowReturnDetailReality> brRealEpcDetailList = new List<CommonJpBorrowReturnDetailReality>();//属于借用的计划表
                        List<CommonJpBorrowReturnDetailReality> reRealEpcDetailList = new List<CommonJpBorrowReturnDetailReality>();//属于领用的计划表
                        List<CommonJpTransferDetailReality> trOutRealDetailEpcList = new List<CommonJpTransferDetailReality>();//属于调拨出库的计划表
                        List<CommonJpTransferDetailReality> trboOutRealDetailEpcList = new List<CommonJpTransferDetailReality>();//属于借用跨库出库的计划表
                        List<CommonJpTransferDetailReality> trreOutRealDetailEpcList = new List<CommonJpTransferDetailReality>();//属于借用跨库归还出库的计划表
                        List<CommonJpMission> MissionListUpdate = new List<CommonJpMission>();
                        List<CommonJpPurchaseEpc> purchaseEpc = new List<CommonJpPurchaseEpc>();
                        List<CommonJpBagInventory> OuterBags = new List<CommonJpBagInventory>();
                        List<CommonJpBagInventory> InsideBags = new List<CommonJpBagInventory>();
                        List<CommonJpEquipmentInventory> OuterInvens = new List<CommonJpEquipmentInventory>();
                        List<CommonJpEquipmentInventory> InsideInvens = new List<CommonJpEquipmentInventory>();
                        List<CommonJpEquipmentState> bagStates = new List<CommonJpEquipmentState>();

                        if (currentState.Equals("1"))//入库业务
                        {

                            InsideInvens = _unitOfWork.EquipmentInventoryRepository.GetList(p => p.InventoryState != "loss" && !p.CurrentState.Equals(0) && p.CurrentState != 5 && p.CurrentState != 6 && !p.CurrentState.Equals(7) && !p.CurrentState.Equals(3) && !p.CurrentState.Equals(10) && Rfids.Contains(p.Epc), null, false, param).ToList();//取出不在库存中的RFID信息,表示要入库的物资
                            InsideBags = _unitOfWork.BagInventoryRepository.GetList(p => p.State.Equals(1) && bagRfids.Contains(p.Epc), null, false, bagParam).ToList();//取出不在库存中的RFID信息,表示要入库的物资
                            List<CommonJpPurchaseEpc> purchaseEpcList = _unitOfWork.PurchaseEpcsRepository.GetList(p => p.Detail.Order.WarehouseId.Equals(warehouseId) && p.CurrentState.Equals(1), null, false, paramPurchase).ToList();//本仓库的全部采购入库EPC

                            #region 采购
                            if (purchaseEpcList.Count > 0)
                            {
                                purchaseEpc = Extensions.ChannelExtension.Purchase(purchaseEpcList, OuterInvens, InsideInvens, Rfids, picUrl, policeId, warehouseId, orgId, ref InsertEpc, ref ErrorEq);
                            }

                            #endregion
                            #region 维修
                            if (fixInRealList.Count > 0)
                            {
                                UpdateFixReal = Extensions.ChannelExtension.Fix(UpdateFixReal, fixInRealList, ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                            }

                            #endregion
                            #region 借用入库
                            if (InsideInvens.Count > 0)
                            {
                                UpdateBoReal = Extensions.ChannelExtension.BorrowIn(ref UpdateEpc, UpdateBoReal, boIntoRealDetailEpcList, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                            }
                            #endregion
                            #region 装备包入库

                            if (InsideInvens.Count > 0 || InsideBags.Count > 0)
                            {
                                List<CommonJpEquipmentState> TempState = new List<CommonJpEquipmentState>();
                                var MissionList = _unitOfWork.MissionRepository.GetList(p => p.State == 1, null, false, MissionParam);
                                if (MissionList.Count() > 0)
                                {
                                    foreach (var item in MissionList)
                                    {
                                        var bagRange = item.CommonJpEquipmentStates.Where(p => p.OutInState == 0 && p.BagInventoryId != null && p.BagInventory.State == 1).ToList();
                                        var InvRange = item.CommonJpEquipmentStates.Where(p => p.OutInState == 0 && p.EqiupmentId != null && p.Eqiupment.CurrentState == 18).ToList();
                                        TempState.AddRange(bagRange);
                                        TempState.AddRange(InvRange);
                                        //foreach (var itemTemp in TempState)
                                        //{
                                        //   item.CommonJpEquipmentStates.Where(p=>p.EqiupmentId.Equals(itemTemp.EqiupmentId)&&itemTemp)

                                        //} 

                                        foreach (var bagitem in InsideBags)
                                        {
                                            if (item.CommonJpEquipmentStates.Where(p => p.BagInventoryId != null && p.BagInventoryId.Equals(bagitem.Id)).Count() > 0)
                                            {
                                                bagitem.State = 0;
                                                string parentId = Guid.NewGuid().ToString();
                                                bagitem.CommonJpEquipmentStates.Add(new CommonJpEquipmentState//新增一条记录表
                                                {
                                                    Id = parentId,
                                                    PicUrl = picUrl,
                                                    IntoTime = DateTime.Now,
                                                    OrgId = orgId,
                                                    WarehouseId = warehouseId,
                                                    State = 7,
                                                    PoliceId = policeId,
                                                    BagInventoryId = bagitem.Id,
                                                    CreateTime = DateTime.Now,
                                                    ActionState = 21,
                                                    OutInState = 1,
                                                    MissionModelId = item.Id
                                                });

                                                foreach (var itemInv in bagitem.CommonJpEquipmentInventories)
                                                {
                                                    if (itemInv.CurrentState.Equals(18))
                                                    {
                                                        itemInv.CurrentState = 0;
                                                        itemInv.CommonJpEquipmentStates.Add(new CommonJpEquipmentState//新增一条记录表
                                                        {
                                                            Id = Guid.NewGuid().ToString(),
                                                            PicUrl = picUrl,
                                                            IntoTime = DateTime.Now,
                                                            OrgId = orgId,
                                                            WarehouseId = warehouseId,
                                                            State = 7,
                                                            PoliceId = policeId,
                                                            BagInventoryId = bagitem.Id,
                                                            CreateTime = DateTime.Now,
                                                            ActionState = 21,
                                                            OutInState = 1,
                                                            ParentId = parentId,
                                                            MissionModelId = item.Id
                                                        });
                                                    }

                                                }
                                                UpdateBag.Add(bagitem);
                                            }
                                            var delete = TempState.Where(p => p.BagInventoryId != null && p.BagInventoryId.Equals(bagitem.Id)).FirstOrDefault();
                                            if (delete != null)
                                            {
                                                TempState.Remove(delete);
                                            }
                                        }
                                        foreach (var Invitem in InsideInvens)
                                        {
                                            if (item.CommonJpEquipmentStates.Where(p => p.EqiupmentId != null && p.EqiupmentId.Equals(Invitem.Id)).Count() > 0)
                                            {
                                                Invitem.CurrentState = 0;
                                                string parentId = Guid.NewGuid().ToString();
                                                Invitem.CommonJpEquipmentStates.Add(new CommonJpEquipmentState//新增一条记录表
                                                {
                                                    Id = parentId,
                                                    PicUrl = picUrl,
                                                    IntoTime = DateTime.Now,
                                                    OrgId = orgId,
                                                    WarehouseId = warehouseId,
                                                    State = 7,
                                                    PoliceId = policeId,
                                                    EqiupmentId = Invitem.Id,
                                                    CreateTime = DateTime.Now,
                                                    ActionState = 21,
                                                    OutInState = 1,
                                                    MissionModelId = item.Id
                                                });
                                                UpdateEpc.Add(Invitem);

                                            }
                                            var delete = TempState.Where(p => p.EqiupmentId != null && p.EqiupmentId.Equals(Invitem.Id)).FirstOrDefault();
                                            if (delete != null)
                                            {
                                                TempState.Remove(delete);
                                            }
                                        }
                                        if (TempState.Count == 0)
                                        {
                                            item.State = 2;
                                            MissionListUpdate.Add(item);
                                        }

                                    }
                                    foreach (var item in UpdateEpc)
                                    {
                                        InsideInvens.Remove(item);
                                    }
                                    foreach (var item in UpdateBag)
                                    {
                                        InsideBags.Remove(item);
                                    }
                                }
                            }
                            #endregion

                            #region 调拨入库
                            if (InsideInvens.Count > 0 && trIntoRealDetailEpcList.Count() > 0)
                            {
                                UpdateTrReal = Extensions.ChannelExtension.TransferIn(UpdateTrReal, trIntoRealDetailEpcList.ToList(), ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);

                            }
                            #endregion
                            #region 借用跨库入库
                            if (InsideInvens.Count > 0 && boOutIntoRealDetailEpcList.Count() > 0)
                            {
                                UpdateboTrReal = Extensions.ChannelExtension.BorrowTransferIn(UpdateboTrReal, boOutIntoRealDetailEpcList.ToList(), ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                            }
                            #endregion
                            #region 归还入库
                            if (InsideInvens.Count > 0)
                            {
                                UpdateReTrReal = Extensions.ChannelExtension.ReturnTransferIn(UpdateboTrReal, ReturnIntoRealDetailEpcList.ToList(), ref UpdateEpc, ref InsideInvens, picUrl, policeId, warehouseId, orgId);
                            }

                            #endregion
                        }
                        else if (currentState.Equals("0"))//出库业务
                        {
                            OuterBags = _unitOfWork.BagInventoryRepository.GetList(p => p.State == 0 && p.WarehouseId.Equals(warehouseId) && p.IsDelete == false && bagRfids.Contains(p.Epc), null, false, bagParam).ToList();//取出在库存中的库存信息，表示要出库的装备包
                            OuterInvens = _unitOfWork.EquipmentInventoryRepository.GetList(p => p.InventoryState != "loss" && p.WarehouseCode.Equals(warehouseId) && p.CurrentState.Equals(0) && Rfids.Contains(p.Epc), null, false, param).ToList();//取出在库存中的库存信息，表示要出库的物资
                            #region 领用出库
                            if (reRealList.Count() > 0)
                            {

                                UpdateBoReal = Extensions.ChannelExtension.Recipients(_unitOfWork, UpdateBoReal, reRealList.ToList(), ref UpdateEpc, ref brOutRealEpcList, reRealEpcDetailList, ref OuterInvens, picUrl, policeId, warehouseId, orgId);

                            }
                            #endregion
                            #region 借用出库 
                            if (brRealOutList.Count() > 0)
                            {
                                UpdateBoReal = Extensions.ChannelExtension.BorrowOut(brRealOutList.ToList(), brRealEpcDetailList, brOutRealEpcList, ref OuterInvens, ref UpdateEpc, UpdateBoReal, picUrl, policeId, warehouseId, orgId);
                            }
                            #endregion

                            #region 调拨出库
                            if (trOutRealList.Count() > 0)
                            {
                                List<CommonJpTransferDetailReality> tempReality = new List<CommonJpTransferDetailReality>();
                                foreach (var item in trOutRealList)
                                {
                                    var outList = OuterInvens.Where(x => !trOutRealEpcList.Contains(x)
                                    && x.EquipmentCode.Equals(item.EquipmentDetailId) && x.CurrentState.Equals(0) && (x.IsBorrowed.Equals(false) || x.IsBorrowed == null)).ToList();//找出本批物资中调拨出库的物资
                                    trOutRealEpcList.AddRange(outList);
                                    if (!trOutRealDetailEpcList.Contains(item) && item.State.Equals(0))
                                    {
                                        trOutRealDetailEpcList.Add(item);
                                    }

                                }
                                if (trOutRealDetailEpcList.Count > 0)
                                {
                                    tempReality = Extensions.ChannelExtension.TransferOut(trOutRealDetailEpcList, trOutRealEpcList, ref UpdateTrReal, tempReality, ref OuterInvens, ref UpdateEpc, picUrl, policeId, warehouseId, orgId);
                                }
                                foreach (var item in tempReality)
                                {
                                    trOutRealDetailEpcList.Remove(item);
                                }
                            }
                            #endregion
                            #region 跨库借用出库
                            if (trboOutRealList.Count() > 0)
                            {
                                List<CommonJpTransferDetailReality> tempboReality = new List<CommonJpTransferDetailReality>();
                                foreach (var item in trboOutRealList)
                                {
                                    var outList = OuterInvens.Where(x => !trOutBoEpcList.Contains(x)
                                    && x.EquipmentCode.Equals(item.EquipmentDetailId) && x.CurrentState.Equals(0) && (x.IsBorrowed.Equals(false) || x.IsBorrowed == null)).ToList();//找出本批物资中跨库借用出库的物资
                                    trOutBoEpcList.AddRange(outList);
                                    if (!trboOutRealDetailEpcList.Contains(item) && item.State.Equals(0))
                                    {
                                        trboOutRealDetailEpcList.Add(item);
                                    }

                                }
                                if (trboOutRealDetailEpcList.Count > 0)
                                {
                                    tempboReality = Extensions.ChannelExtension.BorrowTransferOut(tempboReality, trboOutRealDetailEpcList, trOutBoEpcList, ref UpdateboTrReal, ref OuterInvens, ref UpdateEpc, picUrl, policeId, warehouseId, orgId);
                                }
                                foreach (var item in tempboReality)
                                {
                                    trboOutRealDetailEpcList.Remove(item);
                                }
                            }
                            #endregion
                            #region 跨库归还出库
                            if (reboOutRealList.Count() > 0)
                            {
                                foreach (var item in reboOutRealList)
                                {
                                    var outList = OuterInvens.Where(x => !reOutBoEpcList.Contains(x)
                                    && x.EquipmentCode.Equals(item.EquipmentDetailId) && x.CurrentState.Equals(0)).ToList();//找出本批物资中归还出库的物资
                                    reOutBoEpcList.AddRange(outList);
                                    if (!trreOutRealDetailEpcList.Contains(item) && item.State.Equals(0))
                                    {
                                        trreOutRealDetailEpcList.Add(item);
                                    }

                                }
                                List<CommonJpTransferDetailReality> returnBoReality = new List<CommonJpTransferDetailReality>();
                                if (trreOutRealDetailEpcList.Count > 0)
                                {
                                    returnBoReality = Extensions.ChannelExtension.ReturnTransferOut(returnBoReality, trreOutRealDetailEpcList, reOutBoEpcList, ref UpdateReTrReal, ref OuterInvens, ref UpdateEpc, picUrl, policeId, warehouseId, orgId);
                                }
                                foreach (var item in returnBoReality)
                                {
                                    trreOutRealDetailEpcList.Remove(item);
                                }
                            }

                            #endregion
                            #region 任务出库

                            if (MissionOut != null)
                            {
                                //alltempEq tempEqs = new alltempEq();//所有需要的装备信息
                                List<bags> bagList = new List<bags>();//大致思路为把任务下所有模块的装备包和散件全部取出放到缓存中，将符合标准的物资进行匹配，最后匹配每个类型装备包和物资的数量即可
                                List<invs> eqList = new List<invs>();
                                foreach (var item in MissionOut.CommonJpMisssionChildren)
                                {
                                    foreach (var detail in item.Model.CommonJpModelBags)//将上报的EPC列表去匹配任务下的模块，再将模块下的装备包和装备与上报的EPC进行比较
                                    {
                                        if (!string.IsNullOrEmpty(detail.BagId))
                                        {


                                            if (bagList.Count == 0)
                                            {
                                                bags tempBag = new bags();
                                                tempBag.count = int.Parse(detail.Count.ToString());
                                                tempBag.realCount = 0;
                                                tempBag.bagid = detail.BagId;
                                                bagList.Add(tempBag);
                                            }
                                            else
                                            {
                                                var bags = bagList.FirstOrDefault(p => p.bagid.Equals(detail.BagId));
                                                if (bags != null)
                                                {
                                                    bags.count = bags.count + int.Parse(detail.Count.ToString());

                                                }
                                                else
                                                {
                                                    bags tempBag = new bags();
                                                    tempBag.count = int.Parse(detail.Count.ToString());
                                                    tempBag.realCount = 0;
                                                    tempBag.bagid = detail.BagId;
                                                    bagList.Add(tempBag);
                                                }
                                            }
                                        }
                                        if (!string.IsNullOrEmpty(detail.EquipmentId))
                                        {

                                            if (eqList.Count == 0)
                                            {
                                                invs tempInv = new invs();
                                                tempInv.count = int.Parse(detail.Count.ToString());
                                                tempInv.realCount = 0;
                                                tempInv.detailId = detail.EquipmentId;
                                                eqList.Add(tempInv);
                                            }
                                            else
                                            {
                                                var invs = eqList.FirstOrDefault(p => p.detailId.Equals(detail.EquipmentId));
                                                if (invs != null)
                                                {
                                                    invs.count = invs.count + int.Parse(detail.Count.ToString());

                                                }
                                                else
                                                {
                                                    invs tempInv = new invs();
                                                    tempInv.count = 0;
                                                    tempInv.realCount = 0;
                                                    tempInv.detailId = detail.EquipmentId;
                                                    eqList.Add(tempInv);
                                                }
                                            }
                                        }
                                        var outbagList = OuterBags.Where(x => !MissionBagsList.Contains(x) && x.BagId.Equals(detail.BagId) && x.State.Equals(0)).ToList();//找出本批物资中任务出库的物资
                                        var outList = OuterInvens.Where(x => !MissionEpcList.Contains(x) && x.EquipmentCode.Equals(detail.EquipmentId) && x.CurrentState.Equals(0) && (x.IsBorrowed.Equals(false) || x.IsBorrowed == null)).ToList();//找出本批物资中任务出库的物资
                                        if (outbagList.Count > 0)
                                        {
                                            MissionBagsList.AddRange(outbagList);
                                            foreach (var info in outbagList)
                                            {
                                                OuterBags.Remove(info);
                                            }
                                        }
                                        if (outList.Count > 0)
                                        {
                                            foreach (var info in outList)
                                            {
                                                OuterInvens.Remove(info);
                                            }
                                            MissionEpcList.AddRange(outList);
                                        }
                                    }
                                    if (!missionChild.Contains(item) && (item.State.Equals(0) || item.State == null))
                                    {
                                        missionChild.Add(item);
                                    }
                                }
                                foreach (var item in bagList)
                                {

                                    item.realCount = MissionOut.CommonJpEquipmentStates.Where(p => p.BagInventoryId != null && p.BagInventory.BagId.Equals(item.bagid) && p.OutInState == 0).Count();
                                    if (item.realCount != item.count)
                                    {
                                        foreach (var bagitem in MissionBagsList)
                                        {
                                            if (bagitem.BagId.Equals(item.bagid))
                                            {
                                                string parentId = Guid.NewGuid().ToString();
                                                bagitem.State = 1;
                                                bagitem.CommonJpEquipmentStates.Add(new CommonJpEquipmentState//新增一条记录表
                                                {
                                                    Id = parentId,
                                                    PicUrl = picUrl,
                                                    OutTime = DateTime.Now,
                                                    OrgId = orgId,
                                                    WarehouseId = warehouseId,
                                                    State = 7,
                                                    PoliceId = policeId,
                                                    BagInventoryId = bagitem.Id,
                                                    CreateTime = DateTime.Now,
                                                    ActionState = 20,
                                                    OutInState = 0,
                                                    MissionModelId = MissionOut.Id
                                                });
                                                foreach (var itemInv in bagitem.CommonJpEquipmentInventories)
                                                {
                                                    if (itemInv.CurrentState.Equals(0))
                                                    {
                                                        itemInv.CurrentState = 18;
                                                        itemInv.CommonJpEquipmentStates.Add(new CommonJpEquipmentState//新增一条记录表
                                                        {
                                                            Id = Guid.NewGuid().ToString(),
                                                            PicUrl = picUrl,
                                                            IntoTime = DateTime.Now,
                                                            OrgId = orgId,
                                                            WarehouseId = warehouseId,
                                                            State = 7,
                                                            PoliceId = policeId,
                                                            BagInventoryId = bagitem.Id,
                                                            CreateTime = DateTime.Now,
                                                            ActionState = 20,
                                                            OutInState = 1,
                                                            ParentId = parentId,
                                                            MissionModelId = MissionOut.Id
                                                        });
                                                    }

                                                }
                                                UpdateBag.Add(bagitem);
                                                OuterBags.Remove(bagitem);
                                                item.realCount++;
                                                if (item.realCount == item.count)
                                                {
                                                    break;
                                                }
                                            }
                                        }
                                    }
                                }

                                foreach (var item in eqList)
                                {
                                    item.realCount = MissionOut.CommonJpEquipmentStates.Where(p => p.EqiupmentId != null && p.Eqiupment.EquipmentCode.Equals(item.detailId) && p.OutInState == 0).Count();
                                    if (item.realCount != item.count)
                                    {
                                        foreach (var invitem in MissionEpcList)
                                        {
                                            if (invitem.EquipmentCode.Equals(item.detailId))
                                            {
                                                string parentId = Guid.NewGuid().ToString();
                                                invitem.CurrentState = 18;
                                                invitem.CommonJpEquipmentStates.Add(new CommonJpEquipmentState//新增一条记录表
                                                {
                                                    Id = parentId,
                                                    PicUrl = picUrl,
                                                    OutTime = DateTime.Now,
                                                    OrgId = orgId,
                                                    WarehouseId = warehouseId,
                                                    State = 7,
                                                    PoliceId = policeId,
                                                    EqiupmentId = invitem.Id,
                                                    CreateTime = DateTime.Now,
                                                    ActionState = 20,
                                                    OutInState = 0,
                                                    MissionModelId = MissionOut.Id
                                                });

                                                UpdateEpc.Add(invitem);
                                                OuterInvens.Remove(invitem);
                                                item.realCount++;
                                                if (item.realCount == item.count)
                                                {
                                                    break;
                                                }
                                            }
                                        }
                                    }
                                }

                                var num1 = bagList.Where(p => p.count != p.realCount).Count();
                                var num2 = eqList.Where(p => p.count != p.realCount).Count();
                                if (num1 == 0 && num2 == 0)
                                {
                                    MissionOut.State = 1;
                                }

                            }
                            #endregion
                        }

                        List<CommonJpEquipmentState> ErrorState = new List<CommonJpEquipmentState>();


                        #region 处理全部不属于该仓库的异常数据


                        if (ErrorEq.Count > 0)
                        {
                            foreach (var item in ErrorEq)
                            {
                                if (item.Length.Equals(64))//过滤不属于我们规则的标签
                                {
                                    CommonJpEquipmentState error = new CommonJpEquipmentState();
                                    error.ErrorEpc = item;
                                    error.PicUrl = picUrl;
                                    error.Id = Guid.NewGuid().ToString();
                                    error.ActionState = 11;
                                    error.OrgId = orgId;
                                    error.WarehouseId = warehouseId;
                                    error.State = 6;
                                    error.CreateTime = DateTime.Now;
                                    error.PoliceId = policeId;
                                    ErrorState.Add(error);
                                }
                            }
                        }

                        if (InsideInvens.Count > 0 || OuterInvens.Count > 0)
                        {
                            if (currentState.Equals("1"))
                            {


                                foreach (var InsideItem in InsideInvens)////所有的异常入库
                                {
                                    if (InsideItem.CurrentState.Equals(6))
                                    {
                                        CommonJpEquipmentState error = new CommonJpEquipmentState();
                                        error.ErrorEpc = InsideItem.Epc;
                                        error.PicUrl = picUrl;
                                        error.Id = Guid.NewGuid().ToString();
                                        error.ActionState = 11;
                                        error.OrgId = orgId;
                                        error.WarehouseId = warehouseId;
                                        error.State = 6;
                                        error.CreateTime = DateTime.Now;
                                        error.PoliceId = policeId;
                                        ErrorState.Add(error);
                                    }
                                    else
                                    {
                                        var InsideState = _unitOfWork.EquipmentStateRepository.GetList(p => p.EqiupmentId.Equals(InsideItem.Id)).FirstOrDefault();//取出该物品中除去单警柜的第一条出入库记录

                                        if ((InsideItem.CurrentState.Equals(14) || InsideState.Equals(14)) && InsideItem.WarehouseCode.Equals(warehouseId))//借用入库
                                        {
                                            InsideItem.CurrentState = 0;
                                            InsideItem.EquipmentLocation = 0;
                                            InsideItem.UpdateUser = policeId;
                                            InsideItem.WarehouseCode = warehouseId;
                                            InsideItem.CommonJpEquipmentStates?.Add(new CommonJpEquipmentState//新增一条记录表
                                            {

                                                Id = Guid.NewGuid().ToString(),
                                                IntoTime = DateTime.Now,
                                                PicUrl = picUrl,
                                                OrgId = orgId,
                                                WarehouseId = warehouseId,
                                                State = 6,
                                                PoliceId = policeId,
                                                EqiupmentId = InsideItem.Id,
                                                CreateTime = DateTime.Now,
                                                ActionState = 13,
                                                OutInState = 1,

                                            });
                                            UpdateEpc.Add(InsideItem);
                                        }

                                    }
                                }
                            }
                            if (currentState.Equals("0"))
                            {
                                foreach (var outsideItem in OuterInvens)////所有的异常出库
                                {
                                    if (outsideItem.WarehouseCode.Equals(warehouseId) && outsideItem.EquipmentLocation.Equals(0) && outsideItem.CurrentState.Equals(0))//如果物资在本仓库中，异常出库为借用出库
                                    {
                                        outsideItem.CurrentState = 14;
                                        outsideItem.EquipmentLocation = 3;
                                        outsideItem.UpdateUser = policeId;
                                        outsideItem.WarehouseCode = warehouseId;
                                        outsideItem.CommonJpEquipmentStates?.Add(new CommonJpEquipmentState//新增一条记录表
                                        {

                                            Id = Guid.NewGuid().ToString(),
                                            OutTime = DateTime.Now,
                                            PicUrl = picUrl,
                                            OrgId = orgId,
                                            WarehouseId = warehouseId,
                                            State = 6,
                                            PoliceId = policeId,
                                            EqiupmentId = outsideItem.Id,
                                            CreateTime = DateTime.Now,
                                            ActionState = 14,
                                            OutInState = 0,

                                        });
                                        UpdateEpc.Add(outsideItem);
                                    }

                                }
                            }
                        }
                        #endregion

                        //提取所有上下文跟踪出入库信息
                        var newInvStateList = _unitOfWork.DbContext.ChangeTracker.Entries()
                            .Where(x => x.Entity is CommonJpEquipmentState)
                            .Select(x => x.Entity as CommonJpEquipmentState)
                            .ToList();

                        foreach (var newInvState in newInvStateList)
                        {
                            newInvState.PoliceId = string.IsNullOrEmpty(entity.policeId) ? null : newInvState.PoliceId;
                        }

                        var resultEpc = new List<object>();
                        using (var scope = _unitOfWork.BeginTransaction())
                        {
                            try
                            {
                                bool errorResult = false;
                                if (ErrorState.Count > 0)
                                {
                                    errorResult = _unitOfWork.EquipmentStateRepository.Insert(ErrorState, true);
                                }
                                bool result = false;
                                if (purchaseEpc.Count > 0)
                                {
                                    result = _unitOfWork.PurchaseEpcsRepository.Update(purchaseEpc, false);
                                }
                                //if (bagStates.Count > 0)
                                //{
                                //    result = _unitOfWork.EquipmentStateRepository.Insert(bagStates, false);
                                //}

                                if (UpdateBoReal.Count > 0)
                                {
                                    //_unitOfWork.DbContext.BaseJpCabinets.Update
                                    result = _unitOfWork.BorrowReturnDetailRealityRepository.Update(UpdateBoReal, false);
                                }
                                if (MissionOut != null)
                                {
                                    result = _unitOfWork.MissionRepository.Update(MissionOut, false);
                                }
                                if (MissionListUpdate.Count > 0)
                                {
                                    result = _unitOfWork.MissionRepository.Update(MissionListUpdate, false);
                                }
                                if (UpdateFixReal.Count > 0)
                                {
                                    result = _unitOfWork.FixUselessDetailRepository.Update(UpdateFixReal, false);
                                }
                                if (UpdateTrReal.Count > 0)
                                {
                                    result = _unitOfWork.TransferDetailRealityRepository.Update(UpdateTrReal, false);
                                }
                                if (UpdateboTrReal.Count > 0)
                                {
                                    result = _unitOfWork.TransferDetailRealityRepository.Update(UpdateboTrReal, false);
                                }
                                if (UpdateReTrReal.Count > 0)
                                {
                                    result = _unitOfWork.TransferDetailRealityRepository.Update(UpdateReTrReal, false);
                                }
                                if (UpdateBag.Count > 0)//同时有插入和更新的数据
                                {

                                    result = _unitOfWork.BagInventoryRepository.Update(UpdateBag, true);
                                }
                                if (UpdateEpc.Count > 0 && InsertEpc.Count > 0)//同时有插入和更新的数据
                                {
                                    result = _unitOfWork.EquipmentInventoryRepository.Insert(InsertEpc, false);
                                    result = _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc, true);
                                }
                                else if (UpdateEpc.Count > 0)
                                {
                                    result = _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc, true);
                                }
                                else if (InsertEpc.Count > 0)
                                {
                                    result = _unitOfWork.EquipmentInventoryRepository.Insert(InsertEpc, true);
                                }

                                //#endregion



                                var resultUpdateEpc = UpdateEpc?.Select(p => new
                                {
                                    p.Epc,
                                    p.CurrentState,
                                    p.SafeLevel,
                                    Name = p.EquipmentCodeNavigation?.Name,
                                    Size = p.EquipmentSizecodeNavigation?.SizeName,
                                    type = "single",
                                    ShelfAddress = string.Join(',',
                                        _unitOfWork.EquipmentInventoryRepository
                                            .GetList(x => x.EquipmentCode == p.EquipmentCode && x.WarehouseCode == warehouseId && x.EquipmentSizecode == p.EquipmentSizecode && x.CurrentState != 6, includes: new string[] { "Shelf" })
                                            .ToList()
                                            .Select(x => x.Shelf?.Url)
                                            .Where(x => !string.IsNullOrEmpty(x))
                                            .Distinct())
                                }).ToList();//更新的库存信息
                                var resultInsertEpc = InsertEpc?.Select(p => new
                                {
                                    p.Epc,
                                    p.CurrentState,
                                    p.SafeLevel,
                                    Name = p.EquipmentCodeNavigation?.Name,
                                    Size = p.EquipmentSizecodeNavigation?.SizeName,
                                    type = "single",
                                    ShelfAddress = string.Join(',',
                                        _unitOfWork.EquipmentInventoryRepository
                                            .GetList(x => x.EquipmentCode == p.EquipmentCode && x.WarehouseCode == warehouseId && x.EquipmentSizecode == p.EquipmentSizecode && x.CurrentState != 6, includes: new string[] { "Shelf" })
                                            .ToList()
                                            .Select(x => x.Shelf?.Url)
                                            .Where(x => !string.IsNullOrEmpty(x))
                                            .Distinct())
                                }).ToList();//新增的库存信息
                                var resultErrorEpc = ErrorEq?.Select(p => new
                                {
                                    Epc = p,
                                    CurrentState = 11,
                                    SafeLevel = (int?)0,
                                    Name = "",
                                    Size = "",
                                    type = "single",
                                    ShelfAddress = ""

                                }).ToList();//不在该仓库的物资信息
                                var resultScrapEpc = scrapEpc?.Select(p => new
                                {
                                    Epc = p.Epc,
                                    CurrentState = 6,
                                    SafeLevel = p.SafeLevel,
                                    Name = p.EquipmentCodeNavigation?.Name,
                                    Size = p.EquipmentSizecodeNavigation?.SizeName,
                                    type = "single",
                                    ShelfAddress = ""

                                }).ToList();//已报废物资信息
                                var bagEpc = UpdateBag?.Select(p => new
                                {
                                    Epc = p.Epc,
                                    CurrentState = Convert.ToInt32(p.State),
                                    SafeLevel = (int?)0,
                                    Name = p.Bag?.Name,
                                    Size = "",
                                    type = "bag",
                                    ShelfAddress = ""

                                }).ToList();//不在该仓库的物资信息
                                resultEpc.AddRange(resultUpdateEpc);
                                resultEpc.AddRange(resultInsertEpc);
                                resultEpc.AddRange(resultErrorEpc);
                                resultEpc.AddRange(resultScrapEpc);
                                resultEpc.AddRange(bagEpc);
                                //跨仓库出库业务清空存放位置（只有update.currentState=1:调拨中 8:领用中 15跨仓库借用中 17跨仓库归还中）
                                if (currentState.Equals("0"))
                                {
                                    foreach (var inventory in UpdateEpc.Where(x => new[] { 1, 8, 15, 17 }.Contains(x.CurrentState)))
                                    {
                                        inventory.ShelfId = null;
                                        inventory.ShelfRange = null;
                                        inventory.ShelfRow = null;
                                        inventory.ShelfColumn = null;
                                    }
                                    _unitOfWork.EquipmentInventoryRepository.Update(UpdateEpc);
                                }

                                //上报数据
                                var pushData = new List<object>();
                                pushData.AddRange(InsertEpc.Select(x => new
                                {
                                    editType = "ADD",
                                    id = x.Id,
                                    supplierCode = x.SupplierCode,
                                    epc = x.Epc,
                                    policeCode = x.PoliceCode,
                                    equipmentCode = x.EquipmentCode,
                                    warehouseCode = x.WarehouseCode,
                                    cabinetId = x.CabinetId,
                                    equipmentSizecode = x.EquipmentSizecode,
                                    equipmentLocation = x.EquipmentLocation,
                                    currentState = x.CurrentState,
                                    createTime = x.CreateTime,
                                    warrantyCycle = x.WarrantyCycle,
                                    repairCycle = x.RepairCycle,
                                    productTime = x.ProductTime,
                                    useTime = x.UseTime,
                                    price = x.Price,
                                    isBorrowed = x.IsBorrowed,
                                    lostFlag = x.LostFlag,
                                    inventoryState = x.InventoryState,
                                    isInBox = x.IsInBox,
                                    boxMarkId = x.BoxMarkId,
                                    instantiationState = x.InstantiationState,
                                }));
                                pushData.AddRange(UpdateEpc.Select(x => new
                                {
                                    editType = "UPDATE",
                                    id = x.Id,
                                    supplierCode = x.SupplierCode,
                                    epc = x.Epc,
                                    policeCode = x.PoliceCode,
                                    equipmentCode = x.EquipmentCode,
                                    warehouseCode = x.WarehouseCode,
                                    cabinetId = x.CabinetId,
                                    equipmentSizecode = x.EquipmentSizecode,
                                    equipmentLocation = x.EquipmentLocation,
                                    currentState = x.CurrentState,
                                    createTime = x.CreateTime,
                                    warrantyCycle = x.WarrantyCycle,
                                    repairCycle = x.RepairCycle,
                                    productTime = x.ProductTime,
                                    useTime = x.UseTime,
                                    price = x.Price,
                                    isBorrowed = x.IsBorrowed,
                                    lostFlag = x.LostFlag,
                                    inventoryState = x.InventoryState,
                                    isInBox = x.IsInBox,
                                    boxMarkId = x.BoxMarkId,
                                    instantiationState = x.InstantiationState,
                                }));

                                var httpResult = _httpHelper.GetHtml(new HttpItem()
                                {
                                    URL = _httpHelper._centerServerAddress + "/api/Print/EditInventory",
                                    Encoding = Encoding.UTF8,
                                    Method = "POST",
                                    ContentType = "application/json",
                                    Timeout = 5000,
                                    Postdata = _httpHelper.CreatePostData(JsonConvert.SerializeObject(new
                                    {
                                        data = pushData
                                    }))
                                });
                                if (httpResult.StatusCode != HttpStatusCode.OK || string.IsNullOrEmpty(httpResult.Html))
                                {
                                    scope.Rollback();
                                    return JsonManager.SimpleCustResponse("远端上报数据失败");
                                }
                                var obj = JsonConvert.DeserializeObject<JmpBaseResponse<object>>(httpResult.Html);
                                if (obj.code != "10000")
                                {
                                    scope.Rollback();
                                    return JsonManager.SimpleCustResponse(obj.msg);
                                }
                                scope.Commit();

                            }
                            catch (Exception e)
                            {
                                scope.Rollback();
                                throw;
                            }

                            //计算阈值
                            var sizeList = InsertEpc.Select(x => x.EquipmentSizecode).ToList();
                            sizeList.AddRange(UpdateEpc.Select(x => x.EquipmentSizecode).ToList());
                            _unitOfWork.CalcThreshold(sizeList.Distinct().ToList(), orgId, warehouseId);

                            return JsonManager.ReturnSuccessResponse(resultEpc);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }
        private class bags
        {
            public string bagid { get; set; }
            public int count { get; set; }
            public int realCount { get; set; }
        }
        private class invs
        {
            public string detailId { get; set; }
            public int count { get; set; }
            public int realCount { get; set; }
        }

        /// <summary>
        /// 获取物资的详细信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "orgId":"当前组织机构代码",
        ///         "warehouseId":"选择的仓库代码",
        ///         "state":"当前状态"//0:库存内,1:调拨中,2:借出中,3:单警柜中,4:维修中,5:报废中,6:已报废，7:单警柜出库中8:领用中 空为全选
        ///         "equipmentSizeId":"装备型号",
        ///         "isCurrentLevel":"是否本级"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetAllDetailLocation")]
        public async Task<HttpResponseMessage> GetAllDetailLocation([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        string isCurrentLevel = entity.isCurrentLevel;
                        if (string.IsNullOrEmpty(isCurrentLevel))
                        {
                            isCurrentLevel = "false";
                        }
                        Expression<Func<BaseJpOrganization, bool>> expressionOrg = t => t.State == 1;
                        var orgInfo = _unitOfWork.OrganizationRepository.Get(p => p.Id.Equals(orgCode));

                        var param = new string[] {
                    };


                        ListEq = _unitOfWork.OrganizationRepository.GetList(expressionOrg, t => t.Code).ToList();

                        string OrgId = Guid.Empty.ToString();
                        List<BaseJpOrganization> dataList = null;

                        if (!String.IsNullOrEmpty(entity.orgId))
                        {
                            OrgId = entity.orgId;
                        }

                        dataList = GetClassID(OrgId);

                        dataList = dataList.ToList().Concat(ListEq.Where(c => c.Id.Equals(OrgId))).ToList();
                        List<string> orgList = new List<string>();
                        //Expression<Func<ViewEquipmentLocation, bool>> expression = null;
                        Expression<Func<ViewEquipmentLocation, bool>> expression = t => t.CurrentState != null;
                        if (isCurrentLevel.Equals("true"))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => t.CabinetOrgId.Equals(OrgId) || t.WarehouseOrgId.Equals(OrgId));
                        }
                        else if (isCurrentLevel.Equals("false"))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => dataList.Select(c => c.Id).Contains(t.CabinetOrgId) || dataList.Select(c => c.Id).Contains(t.WarehouseOrgId));
                        }

                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseCode.Equals(warehouseId));
                        }
                        if (!String.IsNullOrEmpty(entity.equipmentSizeId))
                        {
                            string equipmentSizeId = entity.equipmentSizeId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentSizecode.Equals(equipmentSizeId));
                        }
                        if (!String.IsNullOrEmpty(entity.equipmentId))
                        {
                            string equipmentId = entity.equipmentId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentCode.Equals(equipmentId));
                        }
                        if (!String.IsNullOrEmpty(entity.state))
                        {
                            string state = entity.state;
                            if (state.Equals("13") || state.Equals("14"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(13) || t.CurrentState.Equals(14));
                            }
                            else if (state.Equals("15") || state.Equals("16"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(15) || t.CurrentState.Equals(16));
                            }
                            else if (state.Equals("17") || state.Equals("18"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(17) || t.CurrentState.Equals(18));
                            }
                            else
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(int.Parse(state)));
                            }


                        }


                        var EquipmentList = _unitOfWork.ViewEquipmentLocationRepository.GetList(expression).ToList();
                        var datas = EquipmentList.GroupBy(p => new { p.WarehouseOrgName, p.CabinetOrgName, p.CabinetName, p.WarehouseName, p.CurrentState }).Select(x => new
                        {

                            WarehouseOrgName = string.IsNullOrEmpty(x.Key.CabinetOrgName) ? x.Key.WarehouseOrgName : x.Key.CabinetOrgName,
                            CabinetName = x.Key.CurrentState.Equals(3) || x.Key.CurrentState.Equals(7) ? x.Key.CabinetName : "",
                            WarehouseName = x.Key.CurrentState.Equals(3) || x.Key.CurrentState.Equals(7) ? "" : x.Key.WarehouseName,
                            x.Key.CurrentState,
                            counts = x.Sum(c => c.Counts)
                        }).OrderBy(c => c.WarehouseOrgName);

                        var content = new
                        {
                            totalElements = datas.Count(),
                            content = datas.Skip(page * size).Take(size).ToList()
                        };




                        return JsonManager.ReturnSuccessResponse(content);
                        //return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取组织机构及下级的所有库存
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "orgId":"当前组织机构代码",
        ///         "warehouseId":"选择的仓库代码",
        ///         "equipmentId":"装备名称",
        ///         "equipmentSizeId":"装备型号"
        ///         "equipmentInfoId":"装备类型"
        ///         "state":"状态"
        ///         "isCurrentLevel":"true"本级，"false"本级及下级
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquipmentDetail")]
        public async Task<HttpResponseMessage> GetListEquipmentDetail([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        string isCurrentLevel = entity.isCurrentLevel;
                        if (string.IsNullOrEmpty(isCurrentLevel))
                        {
                            isCurrentLevel = "false";
                        }
                        var orgInfo = _unitOfWork.OrganizationRepository.Get(p => p.Id.Equals(orgCode));

                        var param = new string[] {


                    };
                        Expression<Func<BaseJpOrganization, bool>> expressionOrg = t => t.State == 1;

                        ListEq = _unitOfWork.OrganizationRepository.GetList(expressionOrg, t => t.Code).ToList();

                        string OrgId = Guid.Empty.ToString();


                        if (!String.IsNullOrEmpty(entity.orgId))
                        {
                            OrgId = entity.orgId;
                        }


                        Expression<Func<ViewEquipmentLocation, bool>> expression = t => t.CurrentState != null;
                        if (isCurrentLevel.Equals("true"))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => t.CabinetOrgId.Equals(OrgId) || t.WarehouseOrgId.Equals(OrgId));
                        }
                        else if (isCurrentLevel.Equals("false"))
                        {
                            List<BaseJpOrganization> dataList = null;
                            dataList = GetClassID(OrgId);

                            dataList = dataList.ToList().Concat(ListEq.Where(c => c.Id.Equals(OrgId))).ToList();

                            var orgNewList = _unitOfWork.OrganizationRepository.GetList(expressionOrg).Select(p => p.Id);
                            expression = LambdaExtensions.AndAlso(expression, t => dataList.Select(c => c.Id).Contains(t.CabinetOrgId) || dataList.Select(c => c.Id).Contains(t.WarehouseOrgId));
                        }
                        //Expression<Func<ViewEquipmentLocation, bool>> expression = t => dataList.Select(c => c.Id).Contains(t.CabinetOrgId)|| dataList.Select(c => c.Id).Contains(t.WarehouseOrgId);



                        //orgList.Add(orgInfo.Id);//查询所有组织机构仓库库存信息
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseCode.Equals(warehouseId));

                        }
                        if (!String.IsNullOrEmpty(entity.equipmentId))
                        {
                            string equipmentId = entity.equipmentId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentCode.Contains(equipmentId));
                        }
                        if (!String.IsNullOrEmpty(entity.equipmentInfoId))
                        {
                            string equipmentInfoId = entity.equipmentInfoId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentId.Contains(equipmentInfoId));

                        }
                        if (!String.IsNullOrEmpty(entity.equipmentSizeId))
                        {
                            string equipmentSizeId = entity.equipmentSizeId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentSizecode.Contains(equipmentSizeId));

                        }
                        if (!String.IsNullOrEmpty(entity.state))
                        {
                            string state = entity.state;
                            if (state.Equals("13") || state.Equals("14"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(13) || t.CurrentState.Equals(14));
                            }
                            else if (state.Equals("15") || state.Equals("16"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(15) || t.CurrentState.Equals(16));
                            }
                            else if (state.Equals("17") || state.Equals("18"))
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(17) || t.CurrentState.Equals(18));
                            }
                            else
                            {
                                expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState.Equals(int.Parse(state)));
                            }


                        }


                        var EquipmentList = _unitOfWork.ViewEquipmentLocationRepository.GetList(expression, null, false, param).ToList();
                        //var sss= _unitOfWork.EquipmentInventoryRepository.GetGroupPage(expression,t => new { t.WarehouseCode, t.CabinetId, t.CurrentState, t.EquipmentSizecode },"", page,size, false, param);

                        var datas = EquipmentList.GroupBy(p => p.EquipmentSizecode).Select(x => new
                        {
                            sizeId = x.Key,
                            count = x.Sum(c => c.Counts),
                            sizeName = x.FirstOrDefault().SizeName,
                            detailName = x.FirstOrDefault().EquipmentName
                        }).OrderBy(c => c.sizeId);

                        var content = new
                        {
                            totalElements = datas.Count(),
                            content = datas.Skip(page * size).Take(size).ToList()
                        };

                        return JsonManager.ReturnSuccessResponse(content);
                        // return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取组织机构及下级的某装备名称所有库存
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "findCode":"当前组织机构代码",
        ///         "equipmentId":"装备名称",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquByFindCode")]
        public async Task<HttpResponseMessage> GetListEquByFindCode([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string findcode = entity.findCode;
                        string equipmentId = entity.equipmentId;
                        string orgCode = entity.orgCode;

                        Expression<Func<ViewEquipment, bool>> expressionPoliceman = null;

                        expressionPoliceman = t => t.FindCode.Contains(findcode) && t.EquipmentCode.Equals(equipmentId);


                        if (!string.IsNullOrEmpty(orgCode))
                        {
                            expressionPoliceman = LambdaExtensions.AndAlso(expressionPoliceman, t => t.OrgCode.Equals(orgCode));
                        }

                        List<ViewEquipment> equipmentList = _unitOfWork.ViewEquipmentRepository.GetList(expressionPoliceman).OrderBy(c => c.FindCode.Length).ToList();

                        var data = equipmentList.GroupBy(t => new { t.WarehouseCode, t.EquipmentSizecode }).Select(c => new
                        {
                            c.FirstOrDefault().OrgId,
                            c.FirstOrDefault().WarehouseCode,
                            c.FirstOrDefault().WarehouseName,
                            c.FirstOrDefault().EquipmentCode,
                            c.FirstOrDefault().EquipmentName,
                            c.FirstOrDefault().EquipmentSizecode,
                            c.FirstOrDefault().SizeName,
                            zsCount = c.Count(f => f.CurrentState != null),
                            zkCount = c.Where(f => f.CurrentState.Equals(0) || f.CurrentState.Equals(3)).Count(),
                            ckCount = c.Where(f => !(f.CurrentState == null || f.CurrentState.Equals(0) || f.CurrentState.Equals(3) || f.CurrentState.Equals(6))).Count(),
                            bfCount = c.Where(f => f.CurrentState.Equals(6)).Count()
                        }).Where(c => c.WarehouseCode != null).ToList();
                        return JsonManager.ReturnSuccessResponse(data);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取本组织机构物资信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "orgId":"当前组织机构代码",
        ///         "warehouseId":"选择的仓库代码",
        ///         "equipmentId":"装备名称",
        ///         "equipmentSizeId":"装备型号"
        ///         "shelfId":"货架id",
        ///         "range":"货架排位置"
        ///         "row":"货架行位置",
        ///         "column":"货架列位置",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquipmentByOrg")]
        public async Task<HttpResponseMessage> GetListEquipmentByOrg([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }

                        string OrgId = Guid.Empty.ToString();


                        if (String.IsNullOrEmpty(entity.orgId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            OrgId = entity.orgId;
                        }

                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization"
                    };

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => OrgId.Equals(t.WarehouseCodeNavigation.OrgizationId);

                        expression = LambdaExtensions.AndAlso(expression, t => t.InventoryState != "loss");
                        //orgList.Add(orgInfo.Id);//查询所有组织机构仓库库存信息
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseCode.Equals(warehouseId));
                        }

                        //装备名称
                        if (!String.IsNullOrEmpty(entity.equipmentId))
                        {
                            string equipmentId = entity.equipmentId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentCode.Contains(equipmentId));
                        }

                        //装备型号
                        if (!String.IsNullOrEmpty(entity.equipmentSizeId))
                        {
                            string equipmentSizeId = entity.equipmentSizeId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentSizecode.Contains(equipmentSizeId));
                        }

                        if (!string.IsNullOrEmpty(entity.shelfId))
                        {
                            string shelfId = entity.shelfId;
                            expression = expression.AndAlso(t => t.ShelfId == shelfId);

                            if (int.TryParse(entity.range, out int range) && range > 0)
                            {
                                expression = expression.AndAlso(t => t.ShelfRange == range);
                            }
                            if (int.TryParse(entity.row, out int row) && row > 0)
                            {
                                expression = expression.AndAlso(t => t.ShelfRow == row);
                            }
                            if (int.TryParse(entity.column, out int column) && column > 0)
                            {
                                expression = expression.AndAlso(t => t.ShelfColumn == column);
                            }
                        }


                        var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, null, false, param).ToList();
                        //var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetGroupPage(expression,t => new { t.WarehouseCode,t.EquipmentCode,t.EquipmentSizecode },"", page,size, false, param);

                        var datas = EquipmentList.GroupBy(p => new { p.WarehouseCode, p.EquipmentCode, p.EquipmentSizecode }).Select(x => new
                        {
                            id = x.Key.WarehouseCode + x.Key.EquipmentSizecode,
                            warehouseName = x.FirstOrDefault().WarehouseCodeNavigation?.Name,
                            equipmentName = x.FirstOrDefault().EquipmentCodeNavigation?.Name,
                            warehouseId = x.FirstOrDefault().WarehouseCodeNavigation?.Id,
                            equipmentSizeName = x.FirstOrDefault().EquipmentSizecodeNavigation?.SizeName,
                            equipmentSizeId = x.FirstOrDefault().EquipmentSizecodeNavigation?.Id,
                            sumCount = x.Count(),
                            zkCount = x.Count(c => c.CurrentState.Equals(3) || c.CurrentState.Equals(0)),
                            ckCount = x.Count(c => !(c.CurrentState.Equals(3) || c.CurrentState.Equals(0) || c.CurrentState.Equals(6))),
                            bfCount = x.Count(c => c.CurrentState.Equals(6))
                        })
                            .OrderBy(c => c.equipmentName)
                            .ThenBy(c => c.equipmentSizeName);

                        var content = new
                        {
                            totalElements = datas.Count(),
                            content = datas.Skip(page * size).Take(size).ToList()
                        };

                        return JsonManager.ReturnSuccessResponse(content);
                        // return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取织机构物资信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "orgId":"当前组织机构代码"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquipmentByOrgZk")]
        public async Task<HttpResponseMessage> GetListEquipmentByOrgZk([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }

                        string OrgId = Guid.Empty.ToString();


                        if (String.IsNullOrEmpty(entity.orgId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            OrgId = entity.orgId;
                        }

                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization"
                    };

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => OrgId.Equals(t.WarehouseCodeNavigation.OrgizationId);
                        expression = LambdaExtensions.AndAlso(expression, t => t.InventoryState != "loss");
                        ////装备名称
                        //if (!String.IsNullOrEmpty(entity.equipmentId))
                        //{
                        //    string equipmentId = entity.equipmentId;
                        //    expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentCode.Contains(equipmentId));
                        //}

                        var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, null, false, param).ToList();
                        //var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetGroupPage(expression,t => new { t.WarehouseCode,t.EquipmentCode,t.EquipmentSizecode },"", page,size, false, param);

                        var datas = EquipmentList.GroupBy(p => new { p.EquipmentCode }).Select(x => new
                        {
                            equipmentId = x.Key.EquipmentCode,
                            equipmentName = x.FirstOrDefault().EquipmentCodeNavigation?.Name,
                            zkCount = x.Count(c => c.CurrentState.Equals(0))
                        }).Where(c => c.zkCount > 0).OrderBy(c => c.equipmentName);

                        return JsonManager.ReturnSuccessResponse(datas.Where(c => c.zkCount > 0).ToList());
                        // return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取上级组织机构物资信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "orgId":"当前组织机构代码"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquipmentByOrgParentZk")]
        public async Task<HttpResponseMessage> GetListEquipmentByOrgParentZk([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }

                        string OrgId = Guid.Empty.ToString();


                        if (String.IsNullOrEmpty(entity.orgId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            OrgId = entity.orgId;
                        }

                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization"
                    };

                        Expression<Func<BaseJpOrganization, bool>> expressionOrg = t => t.State == 1;

                        var parentId = _unitOfWork.OrganizationRepository.GetList(expressionOrg, t => t.Code).Where(c => c.Id.Equals(OrgId))?.FirstOrDefault().ParentId;

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => parentId.Equals(t.WarehouseCodeNavigation.OrgizationId);
                        expression = LambdaExtensions.AndAlso(expression, t => t.InventoryState != "loss");
                        ////装备名称
                        //if (!String.IsNullOrEmpty(entity.equipmentId))
                        //{
                        //    string equipmentId = entity.equipmentId;
                        //    expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentCode.Contains(equipmentId));
                        //}

                        var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, null, false, param).ToList();
                        //var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetGroupPage(expression,t => new { t.WarehouseCode,t.EquipmentCode,t.EquipmentSizecode },"", page,size, false, param);

                        var datas = EquipmentList.GroupBy(p => new { p.EquipmentCode }).Select(x => new
                        {
                            equipmentId = x.Key.EquipmentCode,
                            equipmentName = x.FirstOrDefault().EquipmentCodeNavigation?.Name,
                            zkCount = x.Count(c => c.CurrentState.Equals(0))
                        }).Where(c => c.zkCount > 0).OrderBy(c => c.equipmentName);

                        return JsonManager.ReturnSuccessResponse(datas.Where(c => c.zkCount > 0).ToList());
                        // return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取本组织机构仓库下型号对应物资信息详情
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "orgId":"当前组织机构代码",
        ///         "warehouseId":"选择的仓库代码",
        ///         "equipmentSizeId":"装备型号",
        ///         "shelfId":"货架id",
        ///         "range":"货架排位置",
        ///         "row":"货架行位置",
        ///         "column":"货架列位置",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquipmentByOrgSize")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetListEquipmentByOrgSize([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }

                        string OrgId = Guid.Empty.ToString();

                        if (String.IsNullOrEmpty(entity.warehouseId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        if (String.IsNullOrEmpty(entity.equipmentSizeId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        if (String.IsNullOrEmpty(entity.orgId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            OrgId = entity.orgId;
                        }



                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization",
                       "CommonJpEquipmentStates",
                       "SupplierCodeNavigation",
                       "Cabinet",
                       "Shelf"
                    };

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => OrgId.Equals(t.WarehouseCodeNavigation.OrgizationId);
                        expression = LambdaExtensions.AndAlso(expression, t => t.InventoryState != "loss");

                        //orgList.Add(orgInfo.Id);//查询所有组织机构仓库库存信息
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseCode.Equals(warehouseId));
                        }

                        //装备型号
                        if (!String.IsNullOrEmpty(entity.equipmentSizeId))
                        {
                            string equipmentSizeId = entity.equipmentSizeId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentSizecode.Contains(equipmentSizeId));
                        }



                        //位置
                        if (!string.IsNullOrEmpty(entity.shelfId))
                        {
                            string shelfId = entity.shelfId;
                            expression = expression.AndAlso(t => t.ShelfId == shelfId);

                            if (int.TryParse(entity.range, out int range) && range > 0)
                            {
                                expression = expression.AndAlso(t => t.ShelfRange == range);
                            }
                            if (int.TryParse(entity.row, out int row) && row > 0)
                            {
                                expression = expression.AndAlso(t => t.ShelfRow == row);
                            }
                            if (int.TryParse(entity.column, out int column) && column > 0)
                            {
                                expression = expression.AndAlso(t => t.ShelfColumn == column);
                            }
                        }

                        var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, c => c.CreateTime, false, param).ToList();

                        var datas = EquipmentList.Select(equipment => new
                        {
                            equipment.Id,
                            equipment.Epc,
                            equipment.UseDirection,
                            equipment.WarrantyCycle,
                            equipment.ProductTime,
                            equipment.UseTime,
                            equipment.CurrentState,

                            shelfId = equipment.ShelfId,
                            shelfCode = equipment.Shelf?.Code,
                            shelfName = equipment.Shelf?.Name,
                            shelf = equipment.Shelf == null ? null : new
                            {
                                equipment.Shelf.Id,
                                equipment.Shelf.CreateTime,
                                equipment.Shelf.Name,
                                equipment.Shelf.Code,
                                equipment.Shelf.Type,
                                equipment.Shelf.WarehouseId,
                                equipment.Shelf.Row,
                                equipment.Shelf.Column,
                                equipment.Shelf.Range,
                            },
                            range = equipment.ShelfRange,
                            row = equipment.ShelfRow,
                            column = equipment.ShelfColumn,
                            safeLevel = equipment.SafeLevel,
                            equipment.Address,
                            sizeName = equipment.EquipmentSizecodeNavigation?.SizeName,
                            equipment.EquipmentSizecode,
                            equipmentName = equipment.EquipmentCodeNavigation?.Name,
                            equipment.EquipmentCode,
                            warehouseName = equipment.WarehouseCodeNavigation?.Name,
                            equipment.WarehouseCode,
                            equipment.WarehouseCodeNavigation?.Orgization?.Name,
                            equipment.WarehouseCodeNavigation?.OrgizationId,
                            supplierName = equipment.SupplierCodeNavigation?.Name,
                            equipment.SupplierCode,
                            cabinetID = equipment.Cabinet?.Id,
                            CabinetChildNum = equipment.Cabinet?.CabinetChildNum + "号柜",
                            equipment.Cabinet?.CabinetNum,
                            equipment.Price,
                            equipment.RepairCycle
                        }
                        );

                        var content = new
                        {
                            totalElements = datas.Count(),
                            content = datas.Skip(page * size).Take(size).ToList()
                        };

                        return JsonManager.ReturnSuccessResponse(content);
                        // return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 批量修改本组织机构仓库下型号对应物资信息详情
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "orgId":"当前组织机构代码",
        ///         "warehouseId":"选择的仓库代码",
        ///         "equipmentSizeId":"装备型号",
        ///         "warrantyCycle":"质保周期(月)",
        ///         "repairCycle":"维保周期(月)",
        ///         "price":"购入单价(元)",
        ///         "productTime":"生产日期",
        ///         "shelfId":"货架id",
        ///         "range":"货架排位置",
        ///         "row":"货架行位置",
        ///         "column":"货架列位置",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("BatchUpdateEquipmentByOrgSize")]
        public async Task<HttpResponseMessage> BatchUpdateEquipmentByOrgSize([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        string OrgId = Guid.Empty.ToString();

                        if (String.IsNullOrEmpty(entity.warehouseId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        if (String.IsNullOrEmpty(entity.equipmentSizeId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        if (String.IsNullOrEmpty(entity.orgId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            OrgId = entity.orgId;
                        }

                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization",
                       "CommonJpEquipmentStates",
                       "SupplierCodeNavigation",
                       "Cabinet",
                       "Shelf"
                    };

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => OrgId.Equals(t.WarehouseCodeNavigation.OrgizationId);


                        //orgList.Add(orgInfo.Id);//查询所有组织机构仓库库存信息
                        if (!String.IsNullOrEmpty(entity.warehouseId))
                        {
                            string warehouseId = entity.warehouseId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.WarehouseCode.Equals(warehouseId));
                        }

                        //装备型号
                        if (!String.IsNullOrEmpty(entity.equipmentSizeId))
                        {
                            string equipmentSizeId = entity.equipmentSizeId;
                            expression = LambdaExtensions.AndAlso(expression, t => t.EquipmentSizecode.Contains(equipmentSizeId));
                        }


                        var equipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, c => c.CreateTime, false, param).ToList();


                        foreach (var item in equipmentList)
                        {
                            if (item.CurrentState == 6)
                            {
                                //已报废不能修改物品信息，存放位置置空
                                item.ShelfRow = null;
                                item.ShelfRange = null;
                                item.ShelfColumn = null;
                                item.ShelfId = null;
                                continue;
                            }

                            item.WarrantyCycle = entity.warrantyCycle == null ? item.WarrantyCycle : Convert.ToInt32(entity.warrantyCycle);
                            item.RepairCycle = entity.repairCycle == null ? item.RepairCycle : Convert.ToInt32(entity.repairCycle);
                            item.Price = entity.price == null ? item.Price : Convert.ToDecimal(entity.price);
                            item.ProductTime = entity.productTime == null ? item.ProductTime : Convert.ToDateTime(entity.productTime);

                            //存放位置
                            //item.Address = entity.address == null ? item.Address : entity.address;
                            if (item.CurrentState != 0 && item.CurrentState != 3 && item.CurrentState != 6)
                            {
                                //出库状态不能保存位置信息
                                continue;
                            }
                            else if (item.CurrentState == 3)
                            {
                                //单警柜不需要保存位置信息
                                continue;
                            }
                            else
                            {
                                if (!string.IsNullOrEmpty(entity.shelfId))
                                {
                                    string shelfId = entity.shelfId;
                                    var shelf = _unitOfWork.BaseJpShelfInfoRepository.Get(shelfId);
                                    if (shelf == null)
                                    {
                                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                    }
                                    if (int.TryParse(entity.range, out int range))
                                    {
                                        if (range > shelf.Range)
                                        {
                                            return JsonManager.SimpleCustResponse("排超过货架限制");
                                        }
                                        item.ShelfRange = range;
                                    }
                                    else
                                    {
                                        item.ShelfRange = null;
                                    }
                                    if (int.TryParse(entity.row, out int row))
                                    {
                                        if (row > shelf.Row)
                                        {
                                            return JsonManager.SimpleCustResponse("行超过货架限制");
                                        }
                                        item.ShelfRow = row;
                                    }
                                    else
                                    {
                                        item.ShelfRow = null;
                                    }
                                    if (int.TryParse(entity.column, out int column))
                                    {
                                        if (column > shelf.Column)
                                        {
                                            return JsonManager.SimpleCustResponse("列超过货架限制");
                                        }
                                        item.ShelfColumn = column;
                                    }
                                    else
                                    {
                                        item.ShelfColumn = null;
                                    }

                                    item.ShelfId = shelf.Id;
                                }
                            }

                        }

                        using (var scope = _unitOfWork.BeginTransaction())
                        {
                            try
                            {
                                if (_unitOfWork.EquipmentInventoryRepository.Update(equipmentList))
                                {
                                    //上报数据
                                    var httpResult = _httpHelper.GetHtml(new HttpItem()
                                    {
                                        URL = _httpHelper._centerServerAddress + "/api/Print/EditInventory",
                                        Encoding = Encoding.UTF8,
                                        Method = "POST",
                                        ContentType = "application/json",
                                        Timeout = 5000,
                                        Postdata = _httpHelper.CreatePostData(JsonConvert.SerializeObject(new
                                        {
                                            data = equipmentList.Select(x => new
                                            {
                                                editType = "UPDATE",
                                                id = x.Id,
                                                supplierCode = x.SupplierCode,
                                                epc = x.Epc,
                                                policeCode = x.PoliceCode,
                                                equipmentCode = x.EquipmentCode,
                                                warehouseCode = x.WarehouseCode,
                                                cabinetId = x.CabinetId,
                                                equipmentSizecode = x.EquipmentSizecode,
                                                equipmentLocation = x.EquipmentLocation,
                                                currentState = x.CurrentState,
                                                createTime = x.CreateTime,
                                                warrantyCycle = x.WarrantyCycle,
                                                repairCycle = x.RepairCycle,
                                                productTime = x.ProductTime,
                                                useTime = x.UseTime,
                                                price = x.Price,
                                                isBorrowed = x.IsBorrowed,
                                                lostFlag = x.LostFlag,
                                                inventoryState = x.InventoryState,
                                                isInBox = x.IsInBox,
                                                boxMarkId = x.BoxMarkId,
                                                instantiationState = x.InstantiationState,
                                            })
                                        }))
                                    });
                                    if (httpResult.StatusCode != HttpStatusCode.OK || string.IsNullOrEmpty(httpResult.Html))
                                    {
                                        scope.Rollback();
                                        return JsonManager.SimpleCustResponse("远端上报数据失败");
                                    }
                                    var obj = JsonConvert.DeserializeObject<JmpBaseResponse<object>>(httpResult.Html);
                                    if (obj.code != "10000")
                                    {
                                        scope.Rollback();
                                        return JsonManager.SimpleCustResponse(obj.msg);
                                    }
                                    scope.Commit();
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    scope.Rollback();
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                                }
                            }
                            catch (Exception e)
                            {
                                scope.Rollback();
                                throw;
                            }
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 批量修改epc对应物资位置信息
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "epcList":["epc1","epc2"],
        ///         "shelfId":"货架id",
        ///         "range":"货架排位置",
        ///         "row":"货架行位置",
        ///         "column":"货架列位置",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("BatchUpdateEquipmentAddressByEpc")]
        public async Task<HttpResponseMessage> BatchUpdateEquipmentAddressByEpc([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        if (String.IsNullOrEmpty(entity.shelfId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        if (entity.epcList == null || entity.epcList.Count == 0)
                        {
                            return JsonManager.SimpleCustResponse("无epc信息", 10009);
                        }

                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization",
                       "CommonJpEquipmentStates",
                       "SupplierCodeNavigation",
                       "Cabinet",
                       "Shelf"
                    };
                        List<string> epcList = entity.epcList;

                        var equipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(t => epcList.Contains(t.Epc), c => c.CreateTime, false, param).ToList();

                        string shelfId = entity.shelfId;
                        var shelf = _unitOfWork.BaseJpShelfInfoRepository.Get(shelfId);
                        if (shelf == null)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        int.TryParse(entity.range, out int range);
                        int.TryParse(entity.row, out int row);
                        int.TryParse(entity.column, out int column);
                        foreach (var item in equipmentList)
                        {
                            item.ShelfId = shelf.Id;
                            item.ShelfRange = range == 0 ? (int?)null : range;
                            item.ShelfRow = row == 0 ? (int?)null : row;
                            item.ShelfColumn = column == 0 ? (int?)null : column;
                        }

                        _unitOfWork.EquipmentInventoryRepository.Update(equipmentList);

                        return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {
                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取本组织机构物资状态对应物资信息详情
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "orgId":"当前组织机构代码",
        ///         "type":"物资状态,中文：装备总数、在库数量、调拨数量、领用数量、维修数量、报废数量、单警柜中",
        ///         "level":"本级、本级及下级",
        ///         "name":"装备名称",
        ///         "topTypeName":"九大类名称",
        ///         "safeLevel":"0：保质期内,1：即将过期,2：已过期"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListEquipmentByOrgAndType")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetListEquipmentByOrgAndType([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        string orgCode = entity.orgId;
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);

                        if (size == 0)
                        {
                            size = 10;
                        }

                        string OrgId = Guid.Empty.ToString();

                        if (String.IsNullOrEmpty(entity.orgId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                        else
                        {
                            OrgId = entity.orgId;
                        }



                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentSizecodeNavigation.Detail" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization",
                       "CommonJpEquipmentStates",
                       "SupplierCodeNavigation",
                       "Cabinet",
                       "Shelf",
                       "CommonJpEquipmentStates",
                       "CommonJpEquipmentStates.Transfer",
                       "CommonJpEquipmentStates.Transfer.Reality",
                       "CommonJpEquipmentStates.Transfer.Reality.SendWarehouse",
                       "CommonJpEquipmentStates.Transfer.Reality.ReceiveWarehouse",
                        };
                        ListEq = _unitOfWork.OrganizationRepository.GetList(t => t.State == 1, x => x.Code).ToList();
                        List<BaseJpOrganization> dataList = GetClassID(OrgId);
                        dataList = dataList.ToList().Concat(ListEq.Where(c => c.Id.Equals(OrgId))).ToList();

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression;

                        if (!string.IsNullOrEmpty(entity.level))
                        {
                            if (!String.IsNullOrEmpty(entity.level))
                            {
                                if (entity.level == "本级")
                                {
                                    expression = t => t.WarehouseCodeNavigation.OrgizationId.Equals(OrgId);
                                }
                                else
                                {
                                    expression = t => dataList.Select(c => c.Id).Contains(t.WarehouseCodeNavigation.OrgizationId);
                                }
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);

                        }


                        expression = LambdaExtensions.AndAlso(expression, t => t.InventoryState != "loss");
                        if (!String.IsNullOrEmpty(entity.type))
                        {
                            string type = entity.type;
                            switch (type)
                            {
                                case "装备总数":
                                    {
                                        //无过滤
                                        break;
                                    }
                                case "在库数量":
                                    {
                                        //0:库存内+3:单警柜
                                        expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState == 0 || t.CurrentState == 3);
                                        break;
                                    }
                                case "调拨数量":
                                    {
                                        //1:调拨中
                                        expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState == 1);
                                        break;
                                    }
                                case "领用数量":
                                    {
                                        //使用中=装备总数-（0:+3：在库数量）-1:调拨数量-4:维修数量-6:报废数量
                                        expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState != 0 && t.CurrentState != 3 && t.CurrentState != 1 && t.CurrentState != 4 && t.CurrentState != 6);
                                        break;
                                    }
                                case "维修数量":
                                    {
                                        //4:维修中
                                        expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState == 4);
                                        break;
                                    }
                                case "报废数量":
                                    {
                                        //6:已报废
                                        expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState == 6);
                                        break;
                                    }
                                case "单警柜中":
                                    {
                                        expression = LambdaExtensions.AndAlso(expression, t => t.CurrentState == 3 || t.CurrentState == 7);
                                        break;
                                    }
                                default:
                                    break;
                            }
                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        if (!string.IsNullOrEmpty(entity.safeLevel))
                        {
                            if (int.TryParse(entity.safeLevel, out int safeLevel))
                            {
                                expression = expression.AndAlso(t => t.CurrentState != 6 && t.SafeLevel == safeLevel);
                            }
                        }

                        if (!string.IsNullOrEmpty(entity.name))
                        {
                            string name = entity.name;
                            expression = expression.AndAlso(x => x.EquipmentSizecodeNavigation != null && x.EquipmentSizecodeNavigation.Detail != null && x.EquipmentSizecodeNavigation.Detail.Name.Contains(name));
                        }

                        if (!string.IsNullOrEmpty(entity.topTypeName))
                        {
                            string topTypeName = entity.topTypeName;
                            var topTypeInfo = _unitOfWork.EquipmentInfoRepository.Get(x => x.Name == topTypeName);
                            if (topTypeInfo == null)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }

                            //遍历下级（3级）
                            var secondInfoList =
                                _unitOfWork.EquipmentInfoRepository.GetList(x => x.ParentId == topTypeInfo.Id).ToList();
                            var threedInfoList = new List<BaseJpEquipmentInfo>();
                            foreach (var info in secondInfoList)
                            {
                                threedInfoList.AddRange(_unitOfWork.EquipmentInfoRepository.GetList(x => x.ParentId == info.Id).ToList());
                            }

                            var equIdList = threedInfoList.Select(x => x.Id);

                            expression = expression.AndAlso(x => x.EquipmentSizecodeNavigation != null && x.EquipmentSizecodeNavigation.Detail != null && equIdList.Contains(x.EquipmentSizecodeNavigation.Detail.EquipmentId));
                        }

                        var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, c => c.CreateTime, false, param)
                            .OrderBy(x => x.EquipmentCode)
                            .ThenBy(x => x.EquipmentSizecode)
                            .ThenBy(x => x.WarehouseCodeNavigation.Orgization.Code)
                            .ThenBy(x => x.WarehouseCode)
                            .ToList();

                        var datas = EquipmentList.Select(equipment => new
                        {
                            equipment.Id,
                            equipment.Epc,
                            equipment.UseDirection,
                            equipment.WarrantyCycle,
                            equipment.ProductTime,
                            equipment.UseTime,
                            equipment.CurrentState,

                            //TransferSendWarehouseName = equipment.CurrentState == 1 ?
                            //    equipment.CommonJpEquipmentStates?.OrderByDescending(x => x.CreateTime).FirstOrDefault()?.Transfer?.Reality?.SendWarehouse?.Name
                            //    : "",
                            TransferReceiveWarehouseName = equipment.CurrentState == 1 ?
                                equipment.CommonJpEquipmentStates?.OrderByDescending(x => x.CreateTime).FirstOrDefault()?.Transfer?.Reality?.ReceiveWarehouse?.Name
                                : "",

                            shelfId = equipment.ShelfId,
                            shelfCode = equipment.Shelf?.Code,
                            shelfName = equipment.Shelf?.Name,
                            shelf = equipment.Shelf == null ? null : new
                            {
                                equipment.Shelf.Id,
                                equipment.Shelf.CreateTime,
                                equipment.Shelf.Name,
                                equipment.Shelf.Code,
                                equipment.Shelf.Type,
                                equipment.Shelf.WarehouseId,
                                equipment.Shelf.Row,
                                equipment.Shelf.Column,
                                equipment.Shelf.Range,
                            },
                            range = equipment.ShelfRange,
                            row = equipment.ShelfRow,
                            column = equipment.ShelfColumn,
                            safeLevel = equipment.SafeLevel,
                            equipment.Address,
                            sizeName = equipment.EquipmentSizecodeNavigation?.SizeName,
                            equipment.EquipmentSizecode,
                            equipmentName = equipment.EquipmentCodeNavigation?.Name,
                            equipment.EquipmentCode,
                            warehouseName = equipment.WarehouseCodeNavigation?.Name,
                            equipment.WarehouseCode,
                            equipment.WarehouseCodeNavigation?.Orgization?.Name,
                            equipment.WarehouseCodeNavigation?.OrgizationId,
                            supplierName = equipment.SupplierCodeNavigation?.Name,
                            equipment.SupplierCode,
                            cabinetID = equipment.Cabinet?.Id,
                            CabinetChildNum = equipment.Cabinet?.CabinetChildNum + "号柜",
                            equipment.Cabinet?.CabinetNum,
                            equipment.Price,
                            equipment.RepairCycle
                        }
                        );

                        var content = new
                        {
                            totalElements = datas.Count(),
                            content = datas.Skip(page * size).Take(size).ToList()
                        };

                        return JsonManager.ReturnSuccessResponse(content);
                        // return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 查看盘点记录
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "size":10,
        ///         "page":0,
        ///         "warehouseId":"当前仓库代码",
        ///         "roles":"manager,user",
        ///         "orderNo":"单据号",,查询条件,非必传
        ///         "userName":"盘点人",查询条件,非必传
        ///         "remarks":"备注信息",查询条件,非必传
        ///         "actionState":"操作类型",查询条件,非必传
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetInvHistory")]
        //[NoSign]
        public async Task<HttpResponseMessage> GetInvHistory([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);
                        int.TryParse(entity.page, out int page);
                        int.TryParse(entity.size, out int size);
                        string orderNo = entity.orderNo;
                        string userName = entity.userName;
                        string warehouseId = entity.warehouseId;
                        string remarks = entity.remarks;
                        string actionState = entity.actionState;
                        string roles = entity.roles;
                        var param = new string[] {
                       "Org" ,
                       "User" ,
                       "Detail",
                       "Size",
                       "CommonJpInventoryDetails" ,
                       "CommonJpInventoryDetails.Inv",
                       "CommonJpInventoryDetails.Inv.EquipmentCodeNavigation",
                       "CommonJpInventoryDetails.Inv.EquipmentSizecodeNavigation",
                       "CommonJpInventoryDetails.Warehouse",
                       "CommonJpInventoryDetails.Inv.WarehouseCodeNavigation",
                       "CommonJpInventoryDetails.Inv.WarehouseCodeNavigation.Orgization",
                        };
                        Expression<Func<CommonJpInventoryTable, bool>> expression;

                        expression = t => t.WarehouseId.Equals(warehouseId) && t.ActionState != "waitAction";
                        if (!string.IsNullOrEmpty(orderNo))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => t.OrderNo.Contains(orderNo));
                        }
                        if (!string.IsNullOrEmpty(actionState))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => t.ActionState.Contains(actionState));
                        }
                        if (!string.IsNullOrEmpty(userName))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => t.User.Username.Contains(userName));
                        }
                        if (!string.IsNullOrEmpty(remarks))
                        {
                            expression = LambdaExtensions.AndAlso(expression, t => t.Remarks.Contains(remarks));
                        }
                        //if (roles.Equals("user"))//普通成员
                        //{
                        //    expression = LambdaExtensions.AndAlso(expression, t => t.ProcessCurrentId != null && t.ProcessCurrent.FlowState == "true");
                        //}
                        var EquipmentList = _unitOfWork.CommonJpInventoryTableRepository.GetPage(expression, "-ActionState,UserId", page, size, true, param);



                        var data = EquipmentList.content.Select(c => new
                        {
                            c.DetailId,
                            detailName = c.Detail?.Name,
                            c.SizeId,
                            sizeName = c.Size?.SizeName,
                            tableType = c.Type,
                            c.ShelfId,
                            c.Column,
                            c.Side,
                            c.Line,
                            c.Id,
                            c.Org?.Name,
                            c.OrgId,
                            c.OrderNo,
                            c.Remarks,
                            c.User?.Username,
                            c.User?.NickName,
                            c.UpdateTime,
                            c.UserId,
                            c.ActionState,
                            //type = c.ProcessCurrentId == null ? "manager" : "user",
                            detailList = c.CommonJpInventoryDetails.Select(t => new
                            {
                                t.Id,
                                state = t.State.Trim(),
                                t.Inv?.Epc,
                                t.Inv?.CurrentState,
                                t.Inv?.EquipmentCodeNavigation?.Name,
                                t.Inv?.EquipmentSizecodeNavigation?.SizeName,
                                warehouseName = t.Warehouse?.Name,
                                orgName = t.Inv?.WarehouseCodeNavigation?.Orgization?.Name,
                                invId = t.Inv?.Id
                            }).OrderBy(p => p.state).ThenBy(p => p.Name).ThenBy(p => p.SizeName).ToList()
                        }
                        ).ToList();

                        var dataPage = new
                        {
                            totalElements = EquipmentList.totalElements,
                            content = data
                        };
                        return JsonManager.ReturnSuccessResponse(dataPage);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    _logger.LogError("Inventory/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 上报盘点记录
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "orderNo":"单据号",
        ///         "userId":"用户ID",
        ///         "orgId":"当前组织机构代码",
        ///         "warehouseId":"当前仓库代码",        
        ///         "remarks":"备注信息",
        ///         "type":"",//All为全盘，location位置盘点,variety物资类型盘点
        ///         "shelfId":"",//货架ID
        ///         "side":"",//面
        ///         "line":"",//行
        ///         "column":"",//列
        ///         "detailList": [{
        ///            "state": "盈亏状态:addition 盈余，deficiency缺失",
        ///            "invID":"物资ID"
        ///            "warehouseID":"原属仓库"
        ///             }
        ///             ] 
        ///            }
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("UploadInv")]
        //[NoSign]
        public async Task<HttpResponseMessage> UploadInv([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        _logger.LogDebug("UploadInv/上报盘点记录:" + jdata.ToString());
                        var entity = JsonManager.GetJsonEntity(jdata);

                        string orderNo = entity.orderNo;
                        string userId = entity.userId;
                        string orgId = entity.orgId;
                        string remarks = entity.remarks;
                        string warehouseId = entity.warehouseId;
                        var InvOrder = _unitOfWork.CommonJpInventoryTableRepository.Get(p => p.OrderNo.Equals(orderNo));

                        if (InvOrder != null)
                        {
                            if (InvOrder.CommonJpInventoryDetails.Count > 0)
                            {
                                _unitOfWork.CommonJpInventoryDetailRepository.Delete(InvOrder.CommonJpInventoryDetails.ToList(), false);
                            }
                            InvOrder.CommonJpInventoryDetails.Clear();

                            foreach (var item in entity.detailList)
                            {
                                if (string.IsNullOrEmpty(item.invID ?? ""))
                                {
                                    return JsonManager.SimpleCustResponse("invID is require");
                                }
                                if (string.IsNullOrEmpty(item.state ?? ""))
                                {
                                    return JsonManager.SimpleCustResponse("state is require");
                                }
                                InvOrder.CommonJpInventoryDetails.Add(new CommonJpInventoryDetail
                                {
                                    Id = Guid.NewGuid().ToString(),
                                    State = item.state,
                                    InvId = item.invID,
                                    TableId = InvOrder.Id,
                                    WarehouseId = item.warehouseID
                                });
                            }
                            InvOrder.ActionState = "noAction";

                            bool result = _unitOfWork.CommonJpInventoryTableRepository.Update(InvOrder, true);
                            if (result)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                        else
                        {
                            string shelfId = entity.shelfId;

                            var tableAction = _unitOfWork.CommonJpInventoryTableRepository.Get(p => p.WarehouseId.Equals(warehouseId) && p.ActionState.Equals("noAction"));
                            //if (tableAction != null)//有未完成的盘点单据，先处理盘点单据才可上报
                            //{
                            //    return JsonManager.SimpleStatusResponse(ResultCode.WAREHOUSE_INV_EXIST);
                            //}

                            CommonJpInventoryTable table = new CommonJpInventoryTable()
                            {
                                Id = Guid.NewGuid().ToString(),
                                Remarks = remarks,
                                WarehouseId = warehouseId,
                                OrgId = orgId,
                                OrderNo = orderNo,
                                UserId = userId,
                                UpdateTime = DateTime.Now,
                                ActionState = "noAction",
                                Type = entity.type,
                                Column = entity.column,
                                Side = entity.side,
                                Line = entity.line
                            };
                            foreach (var item in entity.detailList)
                            {
                                if (string.IsNullOrEmpty(item.invID ?? ""))
                                {
                                    return JsonManager.SimpleCustResponse("invID is require");
                                }
                                if (string.IsNullOrEmpty(item.state ?? ""))
                                {
                                    return JsonManager.SimpleCustResponse("state is require");
                                }

                                table.CommonJpInventoryDetails.Add(new CommonJpInventoryDetail
                                {
                                    Id = Guid.NewGuid().ToString(),
                                    State = item.state,
                                    InvId = item.invID,
                                    TableId = table.Id,
                                    WarehouseId = item.warehouseID
                                });
                            }

                            bool result = _unitOfWork.CommonJpInventoryTableRepository.Insert(table, true);
                            if (result)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 设置库存盘盈盘亏状态
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseID":"仓库",
        ///         "inventoryOrderID": "盘点订单ID",
        ///         "action": "cancel 作废,create 设置订单",
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("SetInvState")]
        public async Task<HttpResponseMessage> SetInvState([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        string warehouseId = entity.warehouseID;
                        string inventoryOrder = entity.inventoryOrderID;
                        string action = entity.action;
                        List<string> AddinvIds = new List<string>();
                        List<string> RemoveinvIds = new List<string>();
                        var InvTabInfo = _unitOfWork.DbContext.CommonJpInventoryTables
                                  .Include(t => t.CommonJpInventoryDetails)
                                  .Include(t => t.Warehouse)
                                  .FirstOrDefault(p => p.Id.Equals(inventoryOrder));
                        if (action.Equals("create"))//生成订单
                        {

                            if (InvTabInfo != null)
                            {
                                foreach (var item in InvTabInfo.CommonJpInventoryDetails)
                                {
                                    if (item.State.Equals("deficiency"))//亏损物资
                                    {
                                        RemoveinvIds.Add(item.InvId);
                                    }
                                    else if (item.State.Contains("addition"))
                                    {
                                        AddinvIds.Add(item.InvId);
                                    }
                                }

                                var addList = _unitOfWork.EquipmentInventoryRepository.GetList(p => AddinvIds.Contains(p.Id)).ToList();
                                var RemoveList = _unitOfWork.EquipmentInventoryRepository.GetList(p => RemoveinvIds.Contains(p.Id)).ToList();
                                var stateList = new List<CommonJpEquipmentState>();
                                foreach (var item in addList)//盈余的物资
                                {
                                    item.EquipmentLocation = 0;
                                    item.CurrentState = 0;
                                    item.InventoryState = "normal";
                                    item.WarehouseCode = warehouseId;
                                    //添加盘盈入库出入库记录
                                    stateList.Add(new CommonJpEquipmentState
                                    {
                                        Id = Guid.NewGuid().ToString(),
                                        EqiupmentId = item.Id,
                                        OutInState = 1,   //1 入库
                                        State = 8,        //8 盘库
                                        WarehouseId = InvTabInfo.WarehouseId,
                                        OrgId = InvTabInfo.Warehouse.OrgizationId,
                                        IntoTime = DateTime.Now,
                                        CreateTime = DateTime.Now,
                                        ActionState = 22, //22 盘盈入库

                                    });
                                }
                                foreach (var item in RemoveList)
                                {
                                    if (item.WarehouseCode.Equals(warehouseId))
                                    {
                                        item.InventoryState = "loss";
                                        //添加盘亏出库出入库记录
                                        stateList.Add(new CommonJpEquipmentState
                                        {
                                            Id = Guid.NewGuid().ToString(),
                                            EqiupmentId = item.Id,
                                            OutInState = 0,   //0 出库
                                            State = 8,        //8 盘库
                                            WarehouseId = InvTabInfo.WarehouseId,
                                            OrgId = InvTabInfo.Warehouse.OrgizationId,
                                            IntoTime = DateTime.Now,
                                            CreateTime = DateTime.Now,
                                            ActionState = 23, //23 盘亏出库

                                        });
                                    }

                                }
                                InvTabInfo.ActionState = "create";

                                using (var scope = _unitOfWork.BeginTransaction())
                                {
                                    try
                                    {
                                        bool result = _unitOfWork.EquipmentStateRepository.Insert(stateList, false);
                                        result = _unitOfWork.EquipmentInventoryRepository.Update(addList, false);
                                        result = _unitOfWork.EquipmentInventoryRepository.Update(RemoveList, false);
                                        result = _unitOfWork.CommonJpInventoryTableRepository.Update(InvTabInfo, true);
                                        if (result)
                                        {
                                            addList.AddRange(RemoveList);
                                            //上报数据
                                            var httpResult = _httpHelper.GetHtml(new HttpItem()
                                            {
                                                URL = _httpHelper._centerServerAddress + "/api/Print/EditInventory",
                                                Encoding = Encoding.UTF8,
                                                Method = "POST",
                                                ContentType = "application/json",
                                                Timeout = 5000,
                                                Postdata = _httpHelper.CreatePostData(JsonConvert.SerializeObject(new
                                                {
                                                    data = addList.Select(x => new
                                                    {
                                                        editType = "UPDATE",
                                                        id = x.Id,
                                                        supplierCode = x.SupplierCode,
                                                        epc = x.Epc,
                                                        policeCode = x.PoliceCode,
                                                        equipmentCode = x.EquipmentCode,
                                                        warehouseCode = x.WarehouseCode,
                                                        cabinetId = x.CabinetId,
                                                        equipmentSizecode = x.EquipmentSizecode,
                                                        equipmentLocation = x.EquipmentLocation,
                                                        currentState = x.CurrentState,
                                                        createTime = x.CreateTime,
                                                        warrantyCycle = x.WarrantyCycle,
                                                        repairCycle = x.RepairCycle,
                                                        productTime = x.ProductTime,
                                                        useTime = x.UseTime,
                                                        price = x.Price,
                                                        isBorrowed = x.IsBorrowed,
                                                        lostFlag = x.LostFlag,
                                                        inventoryState = x.InventoryState,
                                                        isInBox = x.IsInBox,
                                                        boxMarkId = x.BoxMarkId,
                                                        instantiationState = x.InstantiationState,
                                                    })
                                                }))
                                            });
                                            if (httpResult.StatusCode != HttpStatusCode.OK || string.IsNullOrEmpty(httpResult.Html))
                                            {
                                                scope.Rollback();
                                                return JsonManager.SimpleCustResponse("远端上报数据失败");
                                            }
                                            var obj = JsonConvert.DeserializeObject<JmpBaseResponse<object>>(httpResult.Html);
                                            if (obj.code != "10000")
                                            {
                                                scope.Rollback();
                                                return JsonManager.SimpleCustResponse(obj.msg);
                                            }
                                            scope.Commit();
                                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                        }
                                        else
                                        {
                                            scope.Rollback();
                                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                        }
                                    }
                                    catch (Exception e)
                                    {
                                        scope.Rollback();
                                        throw;
                                    }
                                }
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }

                        }
                        else if (action.Equals("cancel"))//作废订单
                        {
                            if (InvTabInfo != null)
                            {
                                InvTabInfo.ActionState = "cancel";
                                var result = _unitOfWork.CommonJpInventoryTableRepository.Update(InvTabInfo, true);
                                if (result)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                                }
                            }
                            else
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                            }

                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }


                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 设置库存盘盈盘亏状态
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseId":"仓库",
        ///         "isLocked": "0:解除锁库，1:锁库"
        ///          
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("SetWarehouseState")]
        public async Task<HttpResponseMessage> SetWarehouseState([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        string warehouseId = entity.warehouseId;
                        string isLocked = entity.isLocked;
                        var warehouse = _unitOfWork.WarehouseRepository.Get(p => p.Id.Equals(warehouseId));
                        if (isLocked == "0")
                        {
                            warehouse.IsLocked = false;
                        }
                        else
                        {
                            warehouse.IsLocked = true;
                        }
                        bool result = _unitOfWork.WarehouseRepository.Update(warehouse, true);


                        if (result)
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                        }
                        else
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取本组织机构仓库下型号对应物资信息详情
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseId":"选择的仓库代码",
        ///         "currentState":"物资状态"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetListWarehouseEqu")]
        public async Task<HttpResponseMessage> GetListWarehouseEqu([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {

                        var entity = JsonManager.GetJsonEntity(jdata);

                        if (String.IsNullOrEmpty(entity.warehouseId))
                        {
                            return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                        }

                        var param = new string[] {
                       "EquipmentSizecodeNavigation" ,
                       "EquipmentCodeNavigation" ,
                       "WarehouseCodeNavigation" ,
                       "WarehouseCodeNavigation.Orgization",
                       "CommonJpEquipmentStates",
                       "SupplierCodeNavigation",
                       "Cabinet",
                       "Shelf",
                       "BoxMark",
                    };
                        string warehouseId = entity.warehouseId;

                        Expression<Func<CommonJpEquipmentInventory, bool>> expression = t => t.WarehouseCode.Equals(warehouseId) && t.InventoryState != "loss";

                        if (int.TryParse(entity.currentState, out int currentState))
                        {
                            expression = expression.AndAlso(t => t.CurrentState == currentState);
                        }

                        var EquipmentList = _unitOfWork.EquipmentInventoryRepository.GetList(expression, c => c.CreateTime, false, param).ToList();

                        var datas = EquipmentList.Select(equipment => new
                        {

                            equipment.Id,
                            equipment.IsInBox,
                            equipment.Epc,
                            equipment.UseDirection,
                            equipment.WarrantyCycle,
                            equipment.ProductTime,
                            equipment.UseTime,
                            equipment.CurrentState,

                            shelfId = equipment.ShelfId,
                            shelfCode = equipment.Shelf?.Code,
                            shelfName = equipment.Shelf?.Name,
                            shelf = equipment.Shelf == null ? null : new
                            {
                                equipment.Shelf.Id,
                                equipment.Shelf.CreateTime,
                                equipment.Shelf.Name,
                                equipment.Shelf.Code,
                                equipment.Shelf.Type,
                                equipment.Shelf.WarehouseId,
                                equipment.Shelf.Row,
                                equipment.Shelf.Column,
                                equipment.Shelf.Range,
                            },
                            range = equipment.ShelfRange,
                            row = equipment.ShelfRow,
                            column = equipment.ShelfColumn,
                            safeLevel = equipment.SafeLevel,
                            equipment.Address,
                            sizeName = equipment.EquipmentSizecodeNavigation?.SizeName,
                            equipment.EquipmentSizecode,
                            equipmentName = equipment.EquipmentCodeNavigation?.Name,
                            equipment.EquipmentCode,
                            warehouseName = equipment.WarehouseCodeNavigation?.Name,
                            equipment.WarehouseCode,
                            equipment.WarehouseCodeNavigation?.Orgization?.Name,
                            equipment.WarehouseCodeNavigation?.OrgizationId,
                            supplierName = equipment.SupplierCodeNavigation?.Name,
                            equipment.SupplierCode,
                            cabinetID = equipment.Cabinet?.Id,
                            CabinetChildNum = equipment.Cabinet?.CabinetChildNum + "号柜",
                            equipment.Cabinet?.CabinetNum,
                            equipment.Price,
                            equipment.RepairCycle,

                            boxMarkEpc = equipment.BoxMark?.Epc
                        }
                        );

                        return JsonManager.ReturnSuccessResponse(datas);
                        // return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });

        }

        /// <summary>
        /// 获取所有未完成的出入库单据
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetOrderInfo")]

        public async Task<HttpResponseMessage> GetOrderInfo([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    var entity = JsonManager.GetJsonEntity(jdata);
                    string warehouseId = entity.warehouseId ?? "";
                    int.TryParse(entity.page, out int page);
                    int.TryParse(entity.size, out int size);
                    if (size == 0)
                    {
                        size = 10;
                    }
                    if (string.IsNullOrEmpty(warehouseId))
                    {
                        return JsonManager.SimpleCustResponse($"warehouseId is require");
                    }
                    else
                    {
                        var b = _unitOfWork.VOrderInfoRepository.GetList(t => t.WarehouseId == warehouseId).OrderByDescending(p => p.CreateTime).ToList();
                        var content = new
                        {
                            totalElements = b.Count(),
                            content = b.Skip(page * size).Take(size).ToList()
                        };
                        //var resList = _unitOfWork.VOrderInfoRepository.GetPage(t => t.WarehouseId == warehouseId, "-CreateTime", page, size);
                        return JsonManager.ReturnSuccessResponse(content);
                    }

                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取当前仓库开启的单据（包括物资详情）
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "warehouseId":"选择的仓库代码",
        ///         "orderType":"单据类型"
        ///     }
        /// 
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetOpenOrderInfo")]
        public async Task<HttpResponseMessage> GetOpenOrderInfo([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    var entity = JsonManager.GetJsonEntity(jdata);
                    string warehouseId = entity.warehouseId ?? "";
                    string orderType = entity.orderType ?? "";

                    Expression<Func<VOrderInfo, bool>> expression = x => x.IsWork == true;

                    if (string.IsNullOrEmpty(warehouseId))
                    {
                        return JsonManager.SimpleCustResponse($"warehouseId is require");
                    }
                    expression = expression.AndAlso(x => x.WarehouseId == warehouseId);

                    if (!string.IsNullOrEmpty(orderType))
                    {
                        expression = expression.AndAlso(x => x.OrderType == orderType);
                    }

                    var res = _unitOfWork.VOrderInfoRepository.Get(expression);
                    if (res == null)
                    {
                        return JsonManager.SimpleCustResponse("无开启单据", "10000");
                    }

                    switch (res.OrderType)
                    {
                        case "借用":
                        case "领用":
                            {
                                var t = _unitOfWork.DbContext.CommonJpBorrowReturnApplies
                                    .Include("Warehouse")
                                    .Include("CommonJpBorrowReturnDetailEstimates")
                                    .Include("CommonJpBorrowReturnDetailEstimates.Equipment")
                                    .Include("CommonJpBorrowReturnDetailEstimates.Size")
                                    .Include("CommonJpBorrowReturnDetailEstimates.Supplier")
                                    .Include("CommonJpBorrowReturns")
                                    .Include("CommonJpBorrowReturns.CommonJpBorrowReturnDetailRealities")
                                    .Include("CommonJpBorrowReturns.CommonJpBorrowReturnDetailRealities.EquipmentDetail")
                                    .Include("CommonJpBorrowReturns.CommonJpBorrowReturnDetailRealities.Size")
                                    .Include("CommonJpBorrowReturns.CommonJpBorrowReturnDetailRealities.Supplier")
                                    .First(x => x.Id == res.ApplyId);

                                return JsonManager.ReturnSuccessResponse(new
                                {
                                    type = t.ActionType.Equals(0) ? 2 : 3,//2借用，3领用
                                    t.ActionType,
                                    t.ChangeFlag,
                                    id = t.Id,
                                    warehouseId = t.WarehouseId,
                                    applyId = t.ApplyUser,
                                    applyName = t.ApplyUserName,
                                    applyTime = t.ApplyTime,
                                    note = t.Note,
                                    t.PoliceNote,
                                    createTime = t.CreateTime,
                                    warehouseName = t.Warehouse?.Name,
                                    t.OrderCode,
                                    expectedTime = t.ExpectedReturnTime,

                                    detailList = t.CommonJpBorrowReturnDetailEstimates?.Select(x => new
                                    {
                                        id = x.Id,
                                        equipmentId = x.Equipment?.Id,
                                        equipmentName = x.Equipment?.Name,
                                        equipmentSizeId = x.Size?.Id,
                                        equipmentSizeName = x.Size?.SizeName,
                                        quantity = x.EquipmentCount,
                                        supplierId = x.Supplier?.Id,
                                        supplierName = x.Supplier?.Name,
                                        sizeCount = _unitOfWork.EquipmentInventoryRepository.GetList(p => p.EquipmentCode.Equals(x.EquipmentId) && p.WarehouseCode.Equals(t.Warehouse.Id)).Count()

                                    }),
                                    applyResult = t.CommonJpBorrowReturns?.Select(p => new
                                    {
                                        p.CurrentState,
                                        resultList = p.CommonJpBorrowReturnDetailRealities?.Select(q => new
                                        {
                                            realityId = q.Id,
                                            quantity = q.EquipmentCount,
                                            equipmentId = q.EquipmentDetail?.Id,
                                            equipmentName = q.EquipmentDetail?.Name,
                                            equipmentCode = q.EquipmentDetail?.Code,
                                            equipmentSizeId = q.Size?.Id,
                                            equipmentSizeName = q.Size?.SizeName,
                                            //applyQuantity = t.CommonJpBorrowReturnDetailEstimates?.Where(x => q.SizeId.Equals(x.SizeId) && q.EquipmentDetailId.Equals(x.EquipmentId))?.FirstOrDefault().EquipmentCount,
                                            supplierId = q.Supplier?.Id,
                                            supplierName = q.Supplier?.Name,
                                            q.RealCount,
                                            q.InsideRealCount,
                                            q.State
                                        })
                                    }).FirstOrDefault()
                                });
                            }
                        case "跨库借用":
                        case "跨库归还":
                        case "库存调拨":
                            {
                                Expression<Func<CommonJpProcessCurrent, bool>> expressionTrans = p => p.OrderId != null;
                                if (res.OrderType.Equals("库存调拨") || res.OrderType.Equals("跨库借用"))
                                {
                                    expressionTrans = expressionTrans.And(p => p.OrderId.Equals(res.ApplyId) || p.CommonJpTransferApplies.Any(c => c.Order.Equals(res.ApplyId)));
                                }
                                else if (res.OrderType.Equals("跨库归还"))
                                {
                                    expressionTrans = expressionTrans.And(p => p.OrderId.Equals(res.ApplyId) || p.CommonJpTransferApplies.FirstOrDefault().CommonJpTransfers.Any(q => q.Order.Equals(res.ApplyId)));
                                }

                                var t = _unitOfWork.DbContext.CommonJpProcessCurrents
                                    .Include("CommonJpTransferApplies")
                                    .Include("CommonJpTransferApplies.TransferWarehouseNavigation")
                                    .Include("CommonJpTransferApplies.Transfer")
                                    .Include("CommonJpTransferApplies.Target")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailEstimates")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailEstimates.EquipmentSize")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailEstimates.EquipmentDetail")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailEstimates.Supplier")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailRealities")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailRealities.Size")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailRealities.EquipmentDetail")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailRealities.Warehouse")
                                    .Include("CommonJpTransferApplies.CommonJpTransferDetailRealities.Supplier")
                                    .FirstOrDefault(expressionTrans);

                                string value = RedisCacheHelper.GetStringValue(t.CommonJpTransferApplies.FirstOrDefault()?.Order);//获取数据

                                if (!string.IsNullOrEmpty(value))
                                {
                                    entity = JObject.Parse(value.ToString());
                                }

                                return JsonManager.ReturnSuccessResponse(new
                                {
                                    type = t.CommonJpTransferApplies.First().ActionType.ToString().Equals("0") ? 0 :
                                        t.CommonJpTransferApplies.First().ActionType.ToString().Equals("1") ? 7 : 9,

                                    t.UserInfo,
                                    t.UserType,
                                    t.DataScope,
                                    t.OrgId,
                                    t.CommonJpTransferApplies?.FirstOrDefault()?.UpdateTime,
                                    t.CurrentStepName,
                                    expectedReturnTime =
                                        t.CommonJpTransferApplies?.FirstOrDefault()?.ExpectedReturnTime,
                                    CurrentOrg = t.CommonJpTransferApplies?.FirstOrDefault()?.TransferId,
                                    id = t.CommonJpTransferApplies?.FirstOrDefault()?.Id,
                                    applyId = t.CommonJpTransferApplies?.FirstOrDefault()?.ApplyUser,
                                    applyName = t.CommonJpTransferApplies?.FirstOrDefault()?.ApplyUserName,
                                    applyTime = t.CommonJpTransferApplies?.FirstOrDefault()?.ApplyTime,
                                    note = t.CommonJpTransferApplies?.FirstOrDefault()?.Reason,
                                    createTime = t.CommonJpTransferApplies?.FirstOrDefault()?.CreateTime,
                                    warehouseName = t.CommonJpTransferApplies?.FirstOrDefault()
                                        ?.TransferWarehouseNavigation?.Name,
                                    warehouseId = t.CommonJpTransferApplies?.FirstOrDefault()?.TransferWarehouse,
                                    order =
                                        t.CommonJpTransferApplies?.FirstOrDefault()?.CommonJpTransfers?.FirstOrDefault()
                                            ?.Order == null
                                            ? t.CommonJpTransferApplies?.FirstOrDefault()?.Order
                                            : t.CommonJpTransferApplies?.FirstOrDefault()?.CommonJpTransfers
                                                ?.FirstOrDefault()?.Order,
                                    transferName = t.CommonJpTransferApplies?.FirstOrDefault()?.Transfer?.Name,
                                    transferId = t.CommonJpTransferApplies?.FirstOrDefault()?.Transfer?.Id,
                                    targetName = t.CommonJpTransferApplies?.FirstOrDefault()?.Target?.Name,
                                    targetId = t.CommonJpTransferApplies?.FirstOrDefault()?.Target?.Id,

                                    detailList = t.CommonJpTransferApplies?.FirstOrDefault()
                                        ?.CommonJpTransferDetailEstimates?.Select(x => new
                                        {

                                            equipmentSizeId = x.EquipmentSize?.Id,
                                            equipmentSizeName = x.EquipmentSize?.SizeName,
                                            equipmentDetailId = x.EquipmentDetail?.Id,
                                            equipmentDetailName = x.EquipmentDetail?.Name,
                                            quantity = x.EquipmentCount,

                                            supplierId = x.Supplier?.Id,
                                            supplierName = x.Supplier?.Name,
                                        }).OrderBy(p => p.equipmentDetailName),
                                    realityDetailList =
                                        t.CommonJpTransferApplies?.FirstOrDefault()?.CommonJpTransferDetailRealities
                                            .Count() == 0
                                            ? entity.realityDetailList
                                            : t.CommonJpTransferApplies?.FirstOrDefault()
                                                ?.CommonJpTransferDetailRealities?.Select(x => new
                                                {
                                                    realityId = x.Id,
                                                    equipmentSizeId = x.Size?.Id,
                                                    equipmentSizeName = x.Size?.SizeName,
                                                    equipmentDetailId = x.EquipmentDetail?.Id,
                                                    equipmentDetailCode = x.EquipmentDetail?.Code,
                                                    warehouseId = x.WarehouseId,
                                                    warehouseName = x.Warehouse?.Name,
                                                    equipmentDetailName = x.EquipmentDetail?.Name,
                                                    quantity = x.EquipmentCount,
                                                    x.RealCount,
                                                    InsideRealCount = x.InsideRealCount ?? 0,
                                                    x.State,
                                                    supplierId = x.Supplier?.Id,
                                                    supplierName = x.Supplier?.Name,
                                                }).OrderBy(p => p.equipmentDetailName)
                                });
                            }
                        default:
                            {
                                return JsonManager.SimpleCustResponse("异常OrderType");
                            }
                    }
                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 获取出入库单据明细，用于充电台出入库查看明细
        /// </summary>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("GetOrderDetail")]
        public async Task<HttpResponseMessage> GetOrderDetail([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    var entity = JsonManager.GetJsonEntity(jdata);
                    string id = entity.id ?? "";
                    string type = entity.type ?? "";
                    string warehouseId = entity?.warehouseId ?? "";
                    if (string.IsNullOrEmpty(id))
                    {
                        return JsonManager.SimpleCustResponse($"id is require");
                    }
                    else if (string.IsNullOrEmpty(type))
                    {
                        return JsonManager.SimpleCustResponse($"type is require");
                    }
                    else if (string.IsNullOrEmpty(warehouseId))
                    {
                        return JsonManager.SimpleCustResponse($"warehouseId is require");
                    }
                    else
                    {
                        if (type == "借用" || type == "领用")
                        {
                            var _borrInfo = _unitOfWork.DbContext.CommonJpBorrowReturns
                                .Include(t => t.CommonJpBorrowReturnDetailRealities)
                                .Include("CommonJpBorrowReturnDetailRealities.Size")
                                .Include("CommonJpBorrowReturnDetailRealities.EquipmentDetail")
                                .FirstOrDefault(t => t.Id == id);
                            var res = _borrInfo.CommonJpBorrowReturnDetailRealities.Select(s => new
                            {
                                s.Id,
                                equId = s.EquipmentDetailId,
                                equName = s.EquipmentDetail?.Name,
                                equCode = s.EquipmentDetail?.Code,
                                equCount = s.EquipmentCount,
                                equUsed = s.RealCount
                            });
                            return JsonManager.ReturnSuccessResponse(res);
                        }
                        else if (type == "库存调拨" || type == "跨库借用" || type == "跨库归还")
                        {
                            var _transfer = _unitOfWork.DbContext.CommonJpTransfers
                                                       .Include(t => t.CommonJpTransferDetailRealities)
                                                       .Include("CommonJpTransferDetailRealities.Size")
                                                       .Include("CommonJpTransferDetailRealities.EquipmentDetail")
                                                       .FirstOrDefault(t => t.Id == id);
                            var res = _transfer.CommonJpTransferDetailRealities
                                .Where(s => s.WarehouseId == warehouseId)
                                .Select(s => new
                                {
                                    s.Id,
                                    equId = s.EquipmentDetailId,
                                    equName = s.EquipmentDetail?.Name,
                                    equCode = s.EquipmentDetail?.Code,
                                    equCount = s.EquipmentCount,
                                    equUsed = s.RealCount
                                });
                            return JsonManager.ReturnSuccessResponse(res);
                        }
                        else
                        {
                            return JsonManager.SimpleCustResponse($"invalid type");
                        }
                    }

                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 设置单据开启关闭
        /// </summary>
        /// <returns></returns>
        [HttpPost("ChangeOrderIsWork")]
        public async Task<HttpResponseMessage> ChangeOrderIsWork([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    var entity = JsonManager.GetJsonEntity(jdata);
                    string id = entity.id ?? "";
                    string type = entity.type ?? "";
                    string warehouseId = entity.warehouseId ?? "";

                    bool bWork = bool.TryParse(entity.isWork ?? "false", out bool isWork);
                    if (string.IsNullOrEmpty(warehouseId))
                    {
                        return JsonManager.SimpleCustResponse($"warehouseId is require");
                    }
                    else if (!bWork)
                    {
                        return JsonManager.SimpleCustResponse($"isWork is require");
                    }
                    else if (string.IsNullOrEmpty(id))
                    {
                        return JsonManager.SimpleCustResponse($"id is require");
                    }
                    else if (string.IsNullOrEmpty(type))
                    {
                        return JsonManager.SimpleCustResponse($"type is require");
                    }
                    else
                    {
                        if (isWork && (_unitOfWork.VOrderInfoRepository.Any(t => t.IsWork == true && t.WarehouseId == warehouseId) || _unitOfWork.MissionRepository.Any(p => p.IsWork == true && p.State == 0 && p.WarehouseId == warehouseId)))
                        {
                            return JsonManager.SimpleCustResponse("已有开启的单据，请先关闭");
                        }
                        else if (type == "借用" || type == "领用")
                        {
                            var _borrInfo = _unitOfWork.DbContext.CommonJpBorrowReturns
                                .Include(t => t.CommonJpBorrowReturnDetailRealities)
                                .Include("CommonJpBorrowReturnDetailRealities.Size")
                                .Include("CommonJpBorrowReturnDetailRealities.EquipmentDetail")
                                .FirstOrDefault(t => t.Id == id);
                            if (_borrInfo == null)
                            {
                                return JsonManager.SimpleCustResponse("无效参数，未找到数据");
                            }
                            else
                            {
                                _borrInfo.IsWork = isWork;
                                var res = _unitOfWork.BorrowReturnRepository.Update(_borrInfo);
                                if (res)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleCustResponse("状态位保存失败");
                                }
                            }
                        }
                        else if (type == "库存调拨" || type == "跨库借用" || type == "跨库归还")
                        {
                            var _transfers = _unitOfWork.DbContext.CommonJpTransferDetailRealities
                                                       .Include(t => t.Reality)
                                                       .Where(t => t.RealityId == id && t.WarehouseId == warehouseId);
                            if (_transfers == null || _transfers.Count() == 0)
                            {
                                return JsonManager.SimpleCustResponse("无效参数，未找到数据");
                            }
                            else
                            {
                                foreach (var item in _transfers)
                                {
                                    item.IsWork = isWork;
                                    _unitOfWork.TransferDetailRealityRepository.Update(item, false);
                                }

                                var res = _unitOfWork.TransferDetailRealityRepository.SaveChanges() > 0;
                                if (res)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleCustResponse("状态位保存失败");
                                }
                            }
                        }
                        else
                        {
                            return JsonManager.SimpleCustResponse($"invalid type");
                        }
                    }

                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 开启本单据，关闭其他单据
        /// </summary>
        /// <remarks>
        /// 
        /// ## 例子
        /// 
        ///     {
        ///         "id":"单据id",
        ///         "type":"单据类型",
        ///         "warehouseId":"选择的仓库代码"
        ///     }
        /// 
        /// </remarks>
        /// <returns></returns>
        [HttpPost("OpenThisOrderAndCloseOther")]
        //[NoSign]
        public async Task<HttpResponseMessage> OpenThisOrderAndCloseOther([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    var entity = JsonManager.GetJsonEntity(jdata);
                    string id = entity.id ?? "";
                    string type = entity.type ?? "";
                    string warehouseId = entity.warehouseId ?? "";
                    if (string.IsNullOrEmpty(warehouseId))
                    {
                        return JsonManager.SimpleCustResponse($"warehouseId is require");
                    }
                    else if (string.IsNullOrEmpty(id))
                    {
                        return JsonManager.SimpleCustResponse($"id is require");
                    }
                    else if (string.IsNullOrEmpty(type))
                    {
                        return JsonManager.SimpleCustResponse($"type is require");
                    }
                    else
                    {
                        //关闭所有单据
                        var openedList = _unitOfWork.VOrderInfoRepository.GetList(t => t.IsWork == true && t.WarehouseId == warehouseId).ToList();
                        foreach (var vOrder in openedList)
                        {
                            if (vOrder.OrderType == "借用" || vOrder.OrderType == "领用")
                            {
                                var borrowInfo = _unitOfWork.DbContext.CommonJpBorrowReturns
                                    .Include(t => t.CommonJpBorrowReturnDetailRealities)
                                    .Include("CommonJpBorrowReturnDetailRealities.Size")
                                    .Include("CommonJpBorrowReturnDetailRealities.EquipmentDetail")
                                    .FirstOrDefault(t => t.Id == vOrder.Id);
                                borrowInfo.IsWork = false;
                                if (!_unitOfWork.BorrowReturnRepository.Update(borrowInfo))
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_OPERATE_FAILED);
                                }
                            }

                            if (vOrder.OrderType == "库存调拨" || vOrder.OrderType == "跨库借用" || vOrder.OrderType == "跨库归还")
                            {
                                var transfers = _unitOfWork.DbContext.CommonJpTransferDetailRealities
                                    .Include(t => t.Reality)
                                    .Where(t => t.RealityId == vOrder.Id && t.WarehouseId == warehouseId)
                                    .ToList();
                                foreach (var item in transfers)
                                {
                                    item.IsWork = false;
                                }

                                if (!_unitOfWork.TransferDetailRealityRepository.Update(transfers))
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_OPERATE_FAILED);
                                }
                            }

                        }

                        if (id == "空的单据信息")
                        {
                            //只关闭全部单据不开启新单据
                            return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                        }

                        if (type == "借用" || type == "领用")
                        {
                            var _borrInfo = _unitOfWork.DbContext.CommonJpBorrowReturns
                                .Include(t => t.CommonJpBorrowReturnDetailRealities)
                                .Include("CommonJpBorrowReturnDetailRealities.Size")
                                .Include("CommonJpBorrowReturnDetailRealities.EquipmentDetail")
                                .FirstOrDefault(t => t.Id == id);
                            if (_borrInfo == null)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.ORDER_NOT_EXIST);
                            }
                            else
                            {
                                _borrInfo.IsWork = true;
                                var res = _unitOfWork.BorrowReturnRepository.Update(_borrInfo);
                                if (res)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_OPERATE_FAILED);
                                }
                            }
                        }
                        else if (type == "库存调拨" || type == "跨库借用" || type == "跨库归还")
                        {
                            var _transfers = _unitOfWork.DbContext.CommonJpTransferDetailRealities
                                                       .Include(t => t.Reality)
                                                       .Where(t => t.RealityId == id && t.WarehouseId == warehouseId)
                                                       .ToList();
                            if (_transfers == null || _transfers.Count() == 0)
                            {
                                return JsonManager.SimpleStatusResponse(ResultCode.ORDER_NOT_EXIST);
                            }
                            else
                            {
                                foreach (var item in _transfers)
                                {
                                    item.IsWork = true;
                                }

                                var res = _unitOfWork.TransferDetailRealityRepository.Update(_transfers);
                                if (res)
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS);
                                }
                                else
                                {
                                    return JsonManager.SimpleStatusResponse(ResultCode.ORDER_OPERATE_FAILED);
                                }
                            }
                        }
                        else
                        {
                            return JsonManager.SimpleCustResponse($"invalid type");
                        }
                    }

                }
                catch (Exception ex)
                {
                    _logger.LogError(ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }

        /// <summary>
        /// 强制结束单据（借用 调拨）
        /// </summary>
        /// <remarks>
        /// ## 例子
        /// 
        ///     {
        ///         "id":"",                //订单id
        ///         "organizationId":"",    //组织机构id
        ///         "userId":""             //操作人id
        ///         "type":""               //单据类型      Borrow：借用  Consume：领用   Transfer：调拨
        ///     }
        /// /
        /// </remarks>
        /// <param name="jdata"></param>
        /// <returns></returns>
        [HttpPost("CloseOrder")]
        public async Task<HttpResponseMessage> CloseOrder([FromBody] JObject jdata)
        {
            return await Task.Run(() =>
            {
                try
                {
                    if (jdata != null)
                    {
                        var entity = JsonManager.GetJsonEntity(jdata);
                        string id = entity?.id ?? "";
                        string organizationId = entity?.organizationId ?? "";
                        string userId = entity?.userId ?? "";
                        string type = entity?.type ?? "";

                        if (entity == null)
                        {
                            return JsonManager.SimpleCustResponse($"model is null");
                        }

                        if (string.IsNullOrEmpty(id))
                        {
                            return JsonManager.SimpleCustResponse($"id is require");
                        }
                        if (string.IsNullOrEmpty(organizationId))
                        {
                            return JsonManager.SimpleCustResponse($"organizationId is require");
                        }
                        if (string.IsNullOrEmpty(userId))
                        {
                            return JsonManager.SimpleCustResponse($"userId is require");
                        }
                        if (string.IsNullOrEmpty(type))
                        {
                            return JsonManager.SimpleCustResponse($"type is require");
                        }

                        var user = _unitOfWork.CommonJpUserRepository.Get(userId);
                        if (user == null)
                        {
                            return JsonManager.SimpleCustResponse($"userId is require");
                        }

                        switch (type)
                        {
                            case "Borrow":
                                {
                                    var borrowReturn = _unitOfWork.DbContext.CommonJpBorrowReturns
                                        .Include(x => x.Apply)
                                        .Include(x => x.Apply.CommonJpBorrowReturnDetailRealities)
                                        .FirstOrDefault(x => x.Id == id);

                                    if (borrowReturn == null)
                                    {
                                        return JsonManager.SimpleCustResponse($"id is require");
                                    }
                                    if (borrowReturn.ActionType != 0)
                                    {
                                        return JsonManager.SimpleCustResponse($"只有借用单能关闭单据");
                                    }
                                    if (borrowReturn.CurrentState == 2)
                                    {
                                        return JsonManager.SimpleCustResponse($"currentState is require");
                                    }
                                    if (borrowReturn.OrgId != organizationId)
                                    {
                                        return JsonManager.SimpleCustResponse($"只有发货仓库才能关闭订单");
                                    }

                                    foreach (var reality in borrowReturn.Apply.CommonJpBorrowReturnDetailRealities)
                                    {
                                        //state -》1已出库
                                        if (reality.State != 2)
                                        {
                                            if (reality.RealCount == 0)
                                            {
                                                //直接结束
                                                reality.State = 2;
                                            }
                                            else
                                            {
                                                reality.State = 1;
                                            }
                                        }
                                    }
                                    if (borrowReturn.Apply.CommonJpBorrowReturnDetailRealities.All(x => x.State == 2))
                                    {
                                        //current_state-》2已完成
                                        borrowReturn.CurrentState = 2;
                                    }
                                    else
                                    {
                                        //current_state-》1已出库
                                        borrowReturn.CurrentState = 1;
                                    }
                                    borrowReturn.UpdateUser = user.NickName;
                                    _unitOfWork.DbContext.CommonJpBorrowReturns.Update(borrowReturn);
                                    break;
                                }
                            case "Consume":
                                {
                                    var borrowReturn = _unitOfWork.DbContext.CommonJpBorrowReturns
                                        .Include(x => x.Apply)
                                        .Include(x => x.Apply.CommonJpBorrowReturnDetailRealities)
                                        .FirstOrDefault(x => x.Id == id);

                                    if (borrowReturn == null)
                                    {
                                        return JsonManager.SimpleCustResponse($"id is require");
                                    }
                                    if (borrowReturn.ActionType != 1)
                                    {
                                        return JsonManager.SimpleCustResponse($"只有领用单能关闭单据");
                                    }
                                    if (borrowReturn.CurrentState == 2)
                                    {
                                        return JsonManager.SimpleCustResponse($"currentState is require");
                                    }
                                    if (borrowReturn.OrgId != organizationId)
                                    {
                                        return JsonManager.SimpleCustResponse($"只有发货仓库才能关闭订单");
                                    }

                                    foreach (var reality in borrowReturn.Apply.CommonJpBorrowReturnDetailRealities)
                                    {
                                        //state -》1已完成
                                        if (reality.State != 1)
                                        {
                                            reality.State = 1;
                                        }
                                    }
                                    if (borrowReturn.Apply.CommonJpBorrowReturnDetailRealities.All(x => x.State == 1))
                                    {
                                        //current_state-》1已出库
                                        borrowReturn.CurrentState = 1;
                                    }
                                    borrowReturn.UpdateUser = user.NickName;
                                    _unitOfWork.DbContext.CommonJpBorrowReturns.Update(borrowReturn);
                                    break;
                                }
                            case "Transfer":
                                {
                                    var transfer = _unitOfWork.DbContext.CommonJpTransfers
                                        .Include(x => x.Apply)
                                        .Include(x => x.Apply.CommonJpTransferDetailRealities)
                                        .FirstOrDefault(x => x.Id == id);

                                    if (transfer == null)
                                    {
                                        return JsonManager.SimpleCustResponse($"id is require");
                                    }

                                    if (transfer.CurrentState == 2)
                                    {
                                        return JsonManager.SimpleCustResponse($"currentState is require");
                                    }
                                    if (transfer.SendOrgizationId != organizationId)
                                    {
                                        return JsonManager.SimpleCustResponse($"只有发货仓库才能关闭订单");
                                    }

                                    foreach (var reality in transfer.Apply.CommonJpTransferDetailRealities)
                                    {
                                        //state -》1已出库
                                        if (reality.State != 2)
                                        {
                                            if (reality.RealCount == 0)
                                            {
                                                //直接结束
                                                reality.State = 2;
                                            }
                                            else
                                            {
                                                reality.State = 1;
                                            }
                                        }
                                    }

                                    if (transfer.Apply.CommonJpTransferDetailRealities.All(x => x.State == 2))
                                    {
                                        //current_state-》2已完成
                                        transfer.CurrentState = 2;
                                    }
                                    else
                                    {
                                        //current_state-》1已出库
                                        transfer.CurrentState = 1;
                                    }
                                    transfer.UpdateUser = userId;
                                    _unitOfWork.DbContext.CommonJpTransfers.Update(transfer);
                                    break;
                                }
                            default:
                                {
                                    return JsonManager.SimpleCustResponse($"type is require");
                                }
                        }

                        return _unitOfWork.SaveChanges() > 0 ?
                            JsonManager.SimpleStatusResponse(ResultCode.OPERATE_SUCCESS) :
                            JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                    }
                    else
                    {
                        return JsonManager.SimpleStatusResponse(ResultCode.REQUEST_DATA_ERROR);
                    }
                }
                catch (Exception ex)
                {

                    //_logger.LogError("Login/SignIn 错误:" + ex.ToString());
                    return JsonManager.SimpleStatusResponse(ResultCode.OPERATE_FAILED);
                }
            });
        }
    }
}
