'use strict';

import {EventEmitter} from 'events';
import hexify from 'hexify';
import CmdApdu from './CmdApdu';
import ResApdu from './ResApdu';


//INS编码
const ins = {
    //追加记录
    APPEND_RECORD: 0xE2,
    //外部认证
    EXTERNAL_AUTHENTICATE: 0x82,
    //产生随机数
    GET_CHALLENGE: 0x84,
    //取响应
    GET_RESPONSE: 0xC0,
    //内部认证
    INTERNAL_AUTHENTICATE: 0x88,
    //读二进制
    READ_BINARY: 0xB0,
    //读记录
    READ_RECORD: 0xB2,
    //选择文件
    SELECT_FILE: 0xA4,
    //修改二进制
    UPDATE_BINARY: 0xD6,
    //修改记录
    UPDATE_RECORD: 0xDC,
    //校验PIN
    VERIFY: 0x20,
    //导出母卡密钥
    OUT_KEY: 0xF6
};


class Application extends EventEmitter {

    constructor(card) {
        super();
        this.card = card;
    }

    send(cmdApdu) {
        return this.card
            .sendCmd(cmdApdu)
            .then(resp => {
                var response = new ResApdu(resp);
                //console.log(`状态码： '${response.statusCode()}'`);
                if (response.hasMoreBytesAvailable()) {
                    //console.log(`长度： '${response.data.length}' `);
                    return this.getResponse(response.numberOfBytesAvailable()).then((resp) => {
                        var resp = new ResApdu(resp);
                        return new ResApdu(response.getDataOnly() + resp.data);
                    });
                } else if (response.isWrongLength()) {
                    cmdApdu.setLe(response.correctLength());
                    return this.send(cmdApdu).then((resp) => {
                        var resp = new ResApdu(resp);
                        return new ResApdu(response.getDataOnly() + resp.data);
                    });
                }
                return response;
            });
    };

    /**
     * 选择文件
     */
    selectFile(bytes, p1, p2) {
        var cmd = new CmdApdu({
            cla: 0x00,
            ins: ins.SELECT_FILE,
            p1: p1 || 0x04,
            p2: p2 || 0x00,
            data: bytes
        });
        return this.send(cmd).then((response) => {
            if (response.isOk()) {
                this.emit('application-selected', {
                    application: hexify.toHexString(bytes)
                });
            }
            return response;
        });

    };

    /**
     * 读取响应信息
     *
     */
    getResponse(length) {
        return this.send(new CmdApdu({
            cla: 0x00,
            ins: ins.GET_RESPONSE,
            p1: 0x00,
            p2: 0x00,
            le: length
        }));
    };


    /**
     * 获取母卡导出密钥
     */
    getOutKey(data) {
        if (typeof data === 'string') {
            data = hexify.toByteArray(data);
        }
        var cmd = {
            cla: 0x80,
            ins: 0xF6,
            p1: 0x07,
            p2: 0x10,
            lc: 0x08,
            data: data
        };
        return this.send(new CmdApdu(cmd));
    }

    /**
     * 获取随机数
     */
    getChallenge() {
        var cmd = {
            cla: 0x00,
            ins: 0x84,
            p1: 0x00,
            p2: 0x00,
            le: 0x04
        };
        return this.send(new CmdApdu(cmd));
    }


    /**
     * 读取记录
     */
    readRecord(sfi, record) {
        //console.log(`Application.readRecord, sfi='${sfi}', record=${record}`);
        return this.send(new CmdApdu({
            cla: 0x00,
            ins: ins.READ_RECORD,
            p1: record,
            p2: (sfi << 3) + 4,
            le: 0
        }));
    };

    getData(p1, p2) {
        return this.send(new CmdApdu({
            cla: 0x00,
            ins: ins.GET_DATA,
            p1: p1,
            p2: p2,
            le: 0
        }));
    };
}


module.exports = Application;