package com.junmp.jyzb.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.junmp.jyzb.api.bean.dto.CabinetDto;
import com.junmp.jyzb.api.bean.dto.CabinetBoxDto;
import com.junmp.jyzb.api.bean.dto.CabinetLogDto;
import com.junmp.jyzb.api.bean.dto.CabinetLogFileDto;
import com.junmp.jyzb.api.bean.query.CabinetReq;
import com.junmp.jyzb.api.bean.req.CabinetLogFileReq;
import com.junmp.jyzb.api.bean.req.CabinetLogReq;
import com.junmp.jyzb.api.bean.req.UpdateCabinetBoxReq;
import com.junmp.jyzb.api.bean.req.UpdateCabinetReq;
import com.junmp.jyzb.api.cmd.CabinetCmd;
import com.junmp.jyzb.api.enums.CabinetStateEnum;
import com.junmp.jyzb.api.exception.enums.CabinetBoxExceptionEnum;
import com.junmp.jyzb.cache.MsgRedisCache;
import com.junmp.jyzb.config.ThreadPoolTaskConfig;
import com.junmp.jyzb.entity.*;
import com.junmp.jyzb.api.exception.enums.CabinetExceptionEnum;
import com.junmp.jyzb.mapper.*;
import com.junmp.jyzb.service.*;
import com.junmp.jyzb.utils.*;
import com.junmp.jyzb.websocket.WsCabinetAppServer;
import com.junmp.v2.common.bean.response.WsCmdRes;
import com.junmp.v2.common.exception.base.ServiceException;
import com.junmp.v2.common.util.BeanPlusUtil;
import com.junmp.v2.db.api.factory.PageFactory;
import com.junmp.v2.db.api.factory.PageResultFactory;
import com.junmp.v2.db.api.page.PageResult;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.websocket.Session;

import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;


@Service
public class CabinetServiceImpl extends ServiceImpl<CabinetMapper, Cabinet> implements CabinetService {


    @Resource
    private CabinetMapper cabinetMapper;
    @Resource
    private PubOrgService pubOrgService;
    @Resource
    private CabinetBoxService cabinetBoxService;

    @Resource
    private MsgRedisCache redisCache;

    @Autowired
    private RabbitMQSendMsg MQ;
    @Autowired
    private WsCabinetAppServer wsCabinetAppServer;
    @Autowired
    private ThreadPoolTaskConfig threadPoolTaskConfig;

    @Resource
    private CabinetBoxPoliceService cabinetBoxPoliceService;

    /**
     * 添加单警柜
     *
     * @param req
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public String addCabinetInfo(UpdateCabinetReq req) {
        PubOrg pubOrg;
        // 查询传入的组织机构id是否存在
        if (ObjectUtil.isNotEmpty(req.getOrgId())){
            pubOrg=pubOrgService.PubOrgExist(req.getOrgId());
        }else {
            pubOrg=pubOrgService.PubOrgCodeExist(req.getOrgCode());
        }
        Cabinet cabinet;
        if (StrUtil.isBlank(req.getCabinetId())) {
            // 查询是否有重复的单警柜编号
            long isExists = this.count(new LambdaQueryWrapper<Cabinet>()
                    .eq(ObjectUtil.isNotEmpty(req.getDevSn()), Cabinet::getDevSn, req.getDevSn()));
            if (isExists > 0) {
                // 抛出单警柜重复异常
                throw new ServiceException(CabinetExceptionEnum.CABINETID_IS_EXISTS);
            }
            // 添加单警柜
            cabinet = BeanPlusUtil.toBean(req, Cabinet.class);
            cabinet.setCreateTime(new Date());
            cabinet.setOrgIdInt(pubOrg.getOrgId());
            cabinet.setOrgCode(pubOrg.getOrgCode());
            //新增后设置离线
            cabinet.setOnlineState(CabinetStateEnum.OFFLINE.getValue());
            cabinet.setDevSn(req.getDevSn());
            cabinet.setNum(ObjectUtil.isNotNull(req.getCabinetBoxList()) ? req.getCabinetBoxList().size() : 0);
            // 添加或更新单警柜
            save(cabinet);
        } else {
            // 更新单警柜
            cabinet = this.getById(req.getCabinetId());
            if (ObjectUtil.isNull(cabinet)) {
                throw new ServiceException(CabinetExceptionEnum.CABINET_ISNOT_EXISTS);
            }
            BeanPlusUtil.copyProperties(req, cabinet);
            cabinet.setId(req.getCabinetId());
            cabinet.setDevSn(req.getDevSn());
            cabinet.setOrgIdInt(pubOrg.getOrgId());
            cabinet.setOrgCode(pubOrg.getOrgCode());
            cabinet.setNum(CollUtil.isNotEmpty(req.getCabinetBoxList()) ? req.getCabinetBoxList().size() : 0);
            cabinet.setUpdateTime(new Date());
            updateById(cabinet);
        }
        // 处理箱门信息
        if (CollUtil.isNotEmpty(req.getCabinetBoxList())) {
            List<CabinetBox> boxList = req.getCabinetBoxList().stream()
                    .map(boxReq -> {
                        CabinetBox box = new CabinetBox();
                        box.setCabinetId(cabinet.getId());
                        box.setPower(boxReq.getPower());
                        box.setEquipmentConfig(boxReq.getEquipmentConfig());
                        box.setContainType(boxReq.getContainType());
                        box.setState(0);
                        box.setBoxName(boxReq.getBoxName());
                        box.setBoxNo(boxReq.getBoxNo());
                        return box;
                    })
                    .collect(Collectors.toList());
            cabinetBoxService.saveOrUpdateBatch(boxList);
        }
        return cabinet.getId();
    }


    /**
     * 删除单警柜(批量删除)
     *
     * @param req
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deleteCabinetInfo(UpdateCabinetReq req) {
        //获取传递的id列表
        if (ObjectUtil.isNull(req.getCabinetList()) || req.getCabinetList().size() == 0) {
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        List<String> cabinetIdList = req.getCabinetList();
        for (String s : cabinetIdList) {
            if (s.equals("")) {
                throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
            }
        }
        List<Cabinet> cabinetList = new ArrayList<>();
        //遍历单警柜id列表
        for (String id : cabinetIdList) {
            //通过单警柜id查询出他的箱门数
            Cabinet cabinet = CabinetExist(id);
            //查询出单警柜之后添加到列表中
            cabinetList.add(cabinet);
            //通过单警柜id循环遍历出并删除所有箱门信息
            List<CabinetBox> list = cabinetBoxService.list(new LambdaQueryWrapper<CabinetBox>()
                    .eq(ObjectUtil.isNotEmpty(id), CabinetBox::getCabinetId, id));
            //创建cabinetBox列表方便批量删除操作
            List<CabinetBox> cabinetBoxList = new ArrayList<>();
            for (CabinetBox cabinetBox : list) {
                CabinetBox cabinetBox1 = new CabinetBox();
                BeanPlusUtil.copyProperties(cabinetBox, cabinetBox1);
                cabinetBoxList.add(cabinetBox1);
//
            }
            //批量修改绑定箱门的警员信息，将其解绑
//            cabinetBoxPoliceService.remove(new LambdaQueryWrapper<CabinetBoxPolice>().eq(CabinetBoxPolice::getCabinetId,cabinetIdList));
            //批量删除箱门数
            cabinetBoxService.removeBatchByIds(cabinetBoxList);
            //批量删除单警柜id
            removeBatchByIds(cabinetList);
        }
        return true;
    }

    /**
     * 获取单警柜列表
     * 通过机构id进行查询
     *
     * @param req 单警柜信息
     * @param req
     * @return
     */
    @Override
    public List<CabinetDto> getAllCabinetList(CabinetReq req) {
        List<CabinetDto> cabinetList = new ArrayList<>();
        List<Cabinet> list = cabinetMapper.getAllCabinetsWithSingleCabinet(String.valueOf(req.getOrgId()), null);

        if (list.size() == 0) {
            return new ArrayList<>();
        }

        // 使用 Stream API 对数据进行分组和排序
        Map<String, List<Cabinet>> groupedCabinets = list.stream()
                .collect(Collectors.groupingBy(Cabinet::getId));

        groupedCabinets.forEach((id, cabinets) -> {
            // 对每个分组内的数据按照 num 字段升序排序
            List<Cabinet> sortedCabinets = cabinets.stream()
                    .sorted(Comparator.comparingInt(Cabinet::getBoxNo))
                    .collect(Collectors.toList());

            // 创建 CabinetDto 对象并设置属性
            CabinetDto cabinetDto = new CabinetDto();
            BeanPlusUtil.copyProperties(sortedCabinets.get(0), cabinetDto);
            cabinetDto.setOrgId(sortedCabinets.get(0).getOrgIdInt());
            cabinetDto.setOrgName(sortedCabinets.get(0).getOrgName());
            List<CabinetBoxDto> cabinetBoxList = new ArrayList<>();
            for (Cabinet cabinet : sortedCabinets) {
                // 在 cabinetBoxList 中查找是否已存在相同 id 的 CabinetBoxDto
                Optional<CabinetBoxDto> existingCabinetBox = cabinetBoxList.stream()
                        .filter(box -> box.getId().equals(cabinet.getBoxId()))
                        .findFirst();

                if (existingCabinetBox.isPresent()) {
                    // 如果已存在，则在现有对象的 policeName 和 policeId 字段分别继续拼接
                    CabinetBoxDto cabinetBoxDto = existingCabinetBox.get();
                    String existingPoliceName = cabinetBoxDto.getPoliceName();
                    String existingPoliceId = cabinetBoxDto.getPoliceId();
                    String currentPoliceName = cabinet.getPoliceName();
                    String currentPoliceId = cabinet.getPoliceId();

                    // 拼接 policeName，使用逗号分隔
                    String updatedPoliceName = existingPoliceName + "," + currentPoliceName;
                    cabinetBoxDto.setPoliceName(updatedPoliceName);

                    // 拼接 policeId，使用逗号分隔
                    String updatedPoliceId = existingPoliceId + "," + currentPoliceId;
                    cabinetBoxDto.setPoliceId(updatedPoliceId);

                    // 更新其他属性...
                } else {
                    // 如果不存在，则创建新的对象并添加到 cabinetBoxList 中
                    CabinetBoxDto cabinetBoxDto = new CabinetBoxDto();
                    BeanPlusUtil.copyProperties(cabinet, cabinetBoxDto);
                    cabinetBoxDto.setBoxNo(cabinet.getBoxNo());
                    cabinetBoxDto.setPoliceId(cabinet.getPoliceId());
                    cabinetBoxDto.setId(cabinet.getBoxId());

                    // 直接拼接 policeName 和 policeId，使用逗号分隔
                    cabinetBoxDto.setPoliceName(cabinet.getPoliceName());
                    cabinetBoxDto.setPoliceId(cabinet.getPoliceId());

                    // 设置其他属性...

                    cabinetBoxList.add(cabinetBoxDto);
                }
            }


            cabinetDto.setCabinetBoxList(cabinetBoxList);
            cabinetList.add(cabinetDto);
        });

        return cabinetList;
    }

    //根据组织机构id查询出单警柜的详细信息(page)
    @Override
    public PageResult<CabinetDto> ShowCabinetPage(CabinetReq req) {
        // 判断组织机构是否存在
        PubOrg pubOrg = pubOrgService.PubOrgExist(req.getOrgId());

        LambdaQueryWrapper<Cabinet> eq = new LambdaQueryWrapper<Cabinet>()
                .eq(Cabinet::getOrgIdInt, req.getOrgId())
                .like(ObjectUtil.isNotNull(req.getName()), Cabinet::getName, req.getName())
                .like(ObjectUtil.isNotNull(req.getDevSn()), Cabinet::getDevSn, req.getDevSn());

        // 使用 PageHelper 开始分页
        Page<Cabinet> page = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());

        // 执行分页查询
        IPage<Cabinet> cabinetPage = page(page, eq);

        // 获取查询结果列表
        List<Cabinet> records = cabinetPage.getRecords();

        // 将 Cabinet 转换为 CabinetDto
        List<CabinetDto> cabinetList = records.stream()
                .map(cabinet -> {
                    CabinetDto cabinetDto = new CabinetDto();
                    BeanPlusUtil.copyProperties(cabinet, cabinetDto);
                    cabinetDto.setOrgName(pubOrg.getOrgName());
                    return cabinetDto;
                })
                .collect(Collectors.toList());

        // 构造一个新的 Page 对象，将 cabinetPage 的数据拷贝到新的对象中
        Page<CabinetDto> newPage = new Page<>(cabinetPage.getCurrent(), cabinetPage.getSize(), cabinetPage.getTotal());
        newPage.setRecords(cabinetList);

        // 使用新的 Page 对象构造返回的 PageResult 对象
        PageResult<CabinetDto> result = PageResultFactory.createPageResult(newPage);

        return result;
    }


    /**
     * 通过单警柜id查询单个单警柜及其箱门信息
     *
     * @param req
     * @return
     */
    @Override
    public CabinetDto showOneCabinet(CabinetReq req) {
        // 查询单警柜信息
        Cabinet cabinet = CabinetExistByNum(req.getDevSn());

        // 查询单警柜的箱门信息
        List<CabinetBox> list = cabinetBoxService.list(new LambdaQueryWrapper<CabinetBox>()
                .eq(ObjectUtil.isNotEmpty(cabinet.getId()), CabinetBox::getCabinetId, cabinet.getId())
                .eq(ObjectUtil.isNotEmpty(req.getContainType()), CabinetBox::getContainType, req.getContainType())
                .orderByAsc(CabinetBox::getBoxNo));

        List<CabinetBoxDto> boxList = list.stream()
                .map(cabinetBox -> {
                    CabinetBoxDto cabinetBoxDto = new CabinetBoxDto();
                    BeanPlusUtil.copyProperties(cabinetBox, cabinetBoxDto);
                    return cabinetBoxDto;
                })
                .collect(Collectors.toList());

        // 将单警柜的信息和它的箱门信息返回
        CabinetDto cabinetDto = new CabinetDto();
        BeanPlusUtil.copyProperties(cabinet, cabinetDto);
        cabinetDto.setCabinetBoxList(boxList);

        return cabinetDto;
    }

    /**
     * 修改单警柜信息
     * 新增箱数
     *
     * @param req
     * @return
     */
    @Override
    public boolean updateCabinetInfo(UpdateCabinetReq req) {
        //查询传入的组织机构id是否存在
        pubOrgService.PubOrgExist(req.getOrgId());



        Cabinet one = CabinetExist(req.getId());
        String oldDevSn="";
        if (!one.getDevSn().equals(req.getDevSn())){
            oldDevSn=one.getDevSn();
        }
        BeanPlusUtil.copyProperties(req, one);

        //查询是否有重复的单警柜编号
        List<Cabinet> list = this.list(new LambdaQueryWrapper<Cabinet>()
                .eq(Cabinet::getDevSn, req.getDevSn()));
        if (list.size() > 0 && !list.get(0).getDevSn().equals(one.getDevSn())) {
            //抛出单警柜重复异常
            throw new ServiceException(CabinetExceptionEnum.CABINETID_IS_EXISTS);
        }
        if (ObjectUtil.isNotEmpty(oldDevSn)|| !oldDevSn.trim().isEmpty()){
            one.setOldSevSn(oldDevSn);
        }
        //箱门信息
        List<CabinetBox> collect = cabinetBoxService.list(new LambdaQueryWrapper<CabinetBox>()
                .eq(CabinetBox::getCabinetId, one.getId())
                .orderByDesc(CabinetBox::getBoxNo));

        List<CabinetBox> addList=new ArrayList<>();
        List<CabinetBox> delList=new ArrayList<>();

        if (ObjectUtil.isNotNull(req.getNum()) && ObjectUtil.isNotNull(req.getOldNum())){
            //如果新的num比旧的num大，那么新增 num-oldNum个箱门信息
            if (req.getNum()>req.getOldNum()){
                for (int i = 1; i < req.getNum()-req.getOldNum()+1; i++) {
                    CabinetBox box = new CabinetBox();
                    box.setCabinetId(one.getId());
                    box.setContainType(1);
                    box.setState(0);
                    box.setBoxName(req.getOldNum()+i+"号箱门");
                    box.setBoxNo(req.getOldNum()+i);
                    addList.add(box);
                }
            } else if (req.getNum()<req.getOldNum()) {
                if (collect.size()>0){
                    //如果现传的num比旧的小，那么删除 oldNum-num个箱门信息
                    for (int i = 0; i < req.getOldNum()-req.getNum(); i++) {
                        CabinetBox cabinetBox = collect.get(i);
                        delList.add(cabinetBox);
                    }
                }

            }
        }

        if (addList.size()>0){
            cabinetBoxService.saveBatch(addList);
        }

        if (delList.size()>0){
            cabinetBoxService.removeBatchByIds(delList);
        }
        //后面还要进行消息推送
        MQ.SendMsg("cabinetMsg", one.getDevSn(), "cabinetChange");
        return this.updateById(one);
    }
    /**
     * 修改单警柜信息，警柜使用
     * 新增箱数
     *
     * @param req
     * @return
     */
    @Override
    public boolean updateCabinetClient(UpdateCabinetReq req) {
        //查询传入的组织机构id是否存在
        pubOrgService.PubOrgExist(req.getOrgId());
        Cabinet one = getOne(new LambdaQueryWrapper<Cabinet>()
                .eq(Cabinet::getDevSn, req.getDevSn()));

        if (one == null) {
            throw new ServiceException(CabinetExceptionEnum.CABINET_ISNOT_EXISTS);
        }
        //查询是否有重复的单警柜编号
        List<Cabinet> list = this.list(new LambdaQueryWrapper<Cabinet>()
                .eq(Cabinet::getDevSn, req.getDevSn()));
        if (list.size() > 0 && !list.get(0).getDevSn().equals(one.getDevSn())) {
            //抛出单警柜重复异常
            throw new ServiceException(CabinetExceptionEnum.CABINETID_IS_EXISTS);
        }
        Cabinet cabinet = new Cabinet();
        BeanPlusUtil.copyProperties(req, cabinet);
        String oldDevSn=one.getDevSn();
        cabinet.setOldSevSn(oldDevSn);
        //后面还要进行消息推送
        return this.updateById(cabinet);
    }

    /**
     * 箱门绑定人员
     * 箱门id-->人员
     * @param req
     * @return
     */
//    @Override
//    public boolean boxBindPolice(CabinetBoxReq req) {
//        //通过箱门id查询出箱门信息
//        CabinetBox one = cabinetBoxService.getOne(new LambdaQueryWrapper<CabinetBox>()
//                .eq(ObjectUtil.isNotEmpty(req.getId()), CabinetBox::getId, req.getId()));
//        if (ObjectUtil.isEmpty(one)){
//            //箱门id不存在
//            throw new ServiceException(CabinetExceptionEnum.CABINETBOX_ISNOT_EXISTS);
//        }
//        //获取箱门的主柜id
//        String cabinetId = one.getCabinetId();
//        //通过主柜id获取机构id
//        Cabinet one1 = getOne(new LambdaQueryWrapper<Cabinet>()
//                .eq(ObjectUtil.isNotEmpty(cabinetId), Cabinet::getId,cabinetId));
//        if (ObjectUtil.isEmpty(one1)){
//            //单警柜不存在异常
//            throw new ServiceException(CabinetExceptionEnum.CABINET_ISNOT_EXISTS);
//        }
//        long orgId = one1.getOrgIdInt();
//        //通过警员id获取警员所在的机构id
//        Policeman one2 = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
//                .eq(ObjectUtil.isNotEmpty(req.getPoliceId()), Policeman::getId, req.getPoliceId()));
//        if (ObjectUtil.isEmpty(one2)){
//            //警员不存在异常
//            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
//        }
//        Long orgId1 = one2.getOrgId();
//        //判断两个机构id是否相同，相同则可以绑定，不相同则绑定失败
//        if (!(orgId1==orgId)){
//            return false;
//        }
//        one2.setCabinetBoxId(one.getId());
//        return policemanService.updateById(one2);
//    }


    /**
     * 修改单警柜箱门状态
     *
     * @param req
     * @return
     */
    @Override
    public boolean ChangeCabinetState(UpdateCabinetReq req) {
        //查询单警柜是否存在
        CabinetExistByNum(req.getDevSn());
        if (ObjectUtil.isNull(req.getCabinetBoxList()) || req.getCabinetBoxList().size() == 0) {
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        List<UpdateCabinetBoxReq> cabinetBoxList = req.getCabinetBoxList();
        List<CabinetBox> list = new ArrayList<>();
        for (UpdateCabinetBoxReq updateCabinet : cabinetBoxList) {
            CabinetBox cabinetBox = new CabinetBox();
            BeanPlusUtil.copyProperties(updateCabinet, cabinetBox);
            list.add(cabinetBox);
        }
        return cabinetBoxService.updateBatchById(list);
    }

    //根据单警柜id查询出组织机构id
    @Override
    public Long SearchOrgId(CabinetReq req) {
        Cabinet cabinet = CabinetExistByNum(req.getDevSn());
        return cabinet.getOrgIdInt();
    }

    //根据单警柜id判断单警柜是否存在
    @Override
    public Cabinet CabinetExist(String id) {
        Cabinet one = getById(id);
        if (ObjectUtil.isNull(one)) {
            throw new ServiceException(CabinetExceptionEnum.CABINET_ISNOT_EXISTS);
        }
        return one;
    }

    //根据单警柜id判断单警柜是否存在
    public Cabinet CabinetExistByNum(String serverNum) {
        Cabinet one = getOne(new LambdaQueryWrapper<Cabinet>()
                .eq(Cabinet::getDevSn, serverNum));
        if (ObjectUtil.isNull(one)) {
            throw new ServiceException(CabinetExceptionEnum.CABINET_ISNOT_EXISTS);
        }
        return one;
    }

    //设置单警柜库存数量信息
    @Override
    public boolean SetInventoryInfo(String id) {
        //统计单警柜总数，在库总数，出库总数，装备总价格
        cabinetMapper.SetInventoryInfo(id);
        return true;
    }


    @Override
    public Cabinet getOneCabinetByBoxId(String boxId) {

        return cabinetMapper.getOneCabinetByBoxId(boxId);
    }

    //查询单警柜日志文件列表
    @Override
    public List<CabinetLogDto> ShowCabinetLogList(CabinetLogReq req)
    {

        if(StringUtils.isEmpty(req.getDevSn()) || StringUtils.isEmpty(req.getOrgCode()))
        {
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }

        String infoKey = req.getOrgCode() + "_" + req.getDevSn();

        List<CabinetLogDto> result = new ArrayList<>();

        if(redisCache.contains(infoKey))//从redis获取
        {
            result = JSON.parseArray(redisCache.get(infoKey),CabinetLogDto.class);
        }
        else //从ws请求获取
        {
            //获取单警柜连接
            String mapKey = "ws_" + infoKey;
            if(!wsCabinetAppServer.sessionMap.containsKey(mapKey))
            {
                throw new ServiceException(CabinetExceptionEnum.CABINET_DISCONNECT);
            }
            WsCabinetAppServer wsServer = wsCabinetAppServer.sessionMap.get(mapKey);
            //发送请求
            wsServer.sendMessage(wsServer.cabinetSession, WsCmdRes.success(CabinetCmd.GET_LOGLIST, req.getDevSn()));

            //等待返回结果 8s
            long targetTime = System.currentTimeMillis() + 8000;

            while (targetTime - System.currentTimeMillis() > 0)
            {
                try
                {
                    if(redisCache.contains(infoKey))
                    {
                        result = JSON.parseArray(redisCache.get(infoKey),CabinetLogDto.class);
                        targetTime = 0;
                    }
                    else
                    {
                        Thread.sleep(300);
                    }
                }
                catch (InterruptedException e)
                {
                    throw new RuntimeException(e);
                }
            }
        }
        if(ObjectUtil.isEmpty(result))
        {
            throw new ServiceException(CabinetExceptionEnum.CABINET_GET_LOG_TIMEOUT);
        }
        return result;
    }

    //获取单警柜单个日志文件
    @Override
    public CabinetLogFileDto ShowCabinetLogFile(CabinetLogFileReq req)
    {
        if(StringUtils.isEmpty(req.getDevSn()) || StringUtils.isEmpty(req.getOrgCode()) || StringUtils.isEmpty(req.getFileName()))
        {
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }

        String infoKey = req.getOrgCode() + "_" + req.getDevSn() + "_" + req.getFileName();

        CabinetLogFileDto result = new CabinetLogFileDto();
        result.setOrgCode(req.getOrgCode());
        result.setDevSn(req.getDevSn());
        result.setLogName(req.getFileName());

        Boolean isSuccess = false;

        if(redisCache.contains(infoKey))//从redis获取
        {
            String info = redisCache.get(infoKey).replace("\n","");
            byte[] decode = Base64.getDecoder().decode(info);
            String s = new String(decode, StandardCharsets.UTF_8);
            result.setFile(s);
            isSuccess = true;
        }
        else //从ws请求获取
        {
            //获取单警柜连接
            String mapKey = "ws_" + req.getOrgCode() + "_" + req.getDevSn();
            if(!wsCabinetAppServer.sessionMap.containsKey(mapKey))
            {
                throw new ServiceException(CabinetExceptionEnum.CABINET_DISCONNECT);
            }
            WsCabinetAppServer wsServer = wsCabinetAppServer.sessionMap.get(mapKey);

            //发送请求
            wsServer.sendMessage(wsServer.cabinetSession, WsCmdRes.success(CabinetCmd.GET_LOGFILE, req.getFileName()));

            //等待返回结果 8s
            long targetTime = System.currentTimeMillis() + 8000;

            while (targetTime - System.currentTimeMillis() > 0)
            {
                try
                {
                    if(redisCache.contains(infoKey))
                    {
                        String info = redisCache.get(infoKey).replace("\n","");
                        byte[] decode = Base64.getDecoder().decode(info);
                        String s = new String(decode, StandardCharsets.UTF_8);
                        result.setFile(s);
                        targetTime = 0;
                        isSuccess = true;
                    }
                    else
                    {
                        Thread.sleep(300);
                    }
                }
                catch (InterruptedException e)
                {
                    throw new RuntimeException(e);
                }
            }
        }
        if(!isSuccess)
        {
            throw new ServiceException(CabinetExceptionEnum.CABINET_GET_LOG_TIMEOUT);
        }
        return result;
    }

}
