﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text.RegularExpressions;
using System.Threading;

namespace JmpDehumidifierLib
{
    public class MachineStatus
    {
        public byte Temp = 0;               // 室内温度
        public byte Humid = 0;              // 室内湿度
        public byte SetHumid = 80;          // 控制湿度
        public bool IsWorking = false;      // 是否正在工作
        public byte ErrorCode = 0;          // 1 设置失败
        public string ip = "";
        public int port = 0;
    }

    public enum CurrentCmd
    {
        Idle = 0,
        GetWSD = 0x01,          // 获取温湿度
        GetWorkState = 0x02,    // 获取工作状态
        GetSetHumid = 0x03,     // 获取设置的湿度值
        OpenMachine = 0x11,     // 打开或关闭设备
        SetHumid = 0x12         // 设置湿度
    }

    public class Dehumidifier
    {
        private Controller controller;

        #region DevDef
        private bool bSending = false;
        private bool isOpen = false;
        private readonly byte DevId = 0;
        private CurrentCmd currentCmd = CurrentCmd.Idle;
        private MachineStatus Status = new MachineStatus();
        private MachineStatus lastStatusHashcode = new MachineStatus();
        #endregion

        #region Base
        public Dehumidifier(string comPort = "COM3", byte devId = 1)
        {
            DevId = devId;
            controller = new Controller(new Serial(comPort.ToUpper()));
        }

        public Dehumidifier(string ip, int port, byte devId = 1)
        {
            if (IsIp(ip))
            {
                DevId = devId;
                ip = ip.Trim();
                Status.ip = ip;
                Status.port = port;
                controller = new Controller(new JpSocket(ip, port));
            }
        }

        private bool IsIp(string ip)
        {
            if (string.IsNullOrEmpty(ip))
            {
                return false;
            }

            string pattern = @"^(1\d{2}|2[0-4]\d|25[0-5]|[1-9]\d|[1-9])(\.(1\d{2}|2[0-4]\d|25[0-5]|[1-9]\d|\d)){3}$";
            return Regex.IsMatch(ip, pattern);
        }

        private bool isClosed = true;
        /// <summary>
        /// 打开设备
        /// </summary>
        public bool Open()
        {
            //controller
            isOpen = controller.Open();
            if (isOpen)
            {
                if (controller.GetType().GetProperty("MsgRcvEvent", BindingFlags.Instance | BindingFlags.NonPublic) == null)
                {
                    controller.MsgRcvEvent += new MessageReceivedDeleg(Controller_MsgRcvEvent);

                }
                if(isClosed)
                {
                    Thread thr_Status = new Thread(CheckStatus)
                    {
                        IsBackground = true
                    };
                    thr_Status.Start();
                }
                isClosed = false;
                return true;
            }
            else
            {
                isClosed = true;
                return false;
            }
        }

        /// <summary>
        /// 关闭
        /// </summary>
        /// <returns></returns>
        public bool Close()
        {
            if(controller.Close())
            {
                controller.MsgRcvEvent -= new MessageReceivedDeleg(Controller_MsgRcvEvent);
                isOpen = false;
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// 判断设备是否打开
        /// </summary>
        /// <returns></returns>
        public bool IsOpen()
        {
            return isOpen;
        }

        private bool Send(byte[] data)
        {
            if (bSending)
            {
                Console.WriteLine("命令正在处理中，不接收新命令!");
                return false;
            }            
            bSending = true;
            Status.ErrorCode = 1;
            // 计算CRC

            var crc = Crc16_ModBus(data, data.Length - 2);
            data[data.Length - 1] = (byte)((crc >> 8) & 0xff);
            data[data.Length - 2] = (byte)(crc & 0xff);
            var rtn = controller.Send(data);
            return rtn;
        }

        /// <summary>
        /// 接收消息并处理
        /// </summary>
        /// <param name="msg"></param>
        /// <param name="len"></param>
        private void Controller_MsgRcvEvent(byte[] msg)
        {
            // 如果消息为空，则不处理
            if (msg == null || msg.Length == 0)
            {
                return;
            }
            DealData(msg);
        }

        #endregion

        #region App
        private void CheckStatus()
        {
            while (isOpen)
            {
                GetWSD();
                IsMachineWorking();
                GetSetHumid();
                CheckStatusChanged();
                Thread.Sleep(10000);
            }
        }
        /// <summary>
        /// 获取温湿度
        /// </summary>
        /// <param name="args">参数</param>
        /// <returns></returns>
        private bool GetWSD()
        {
            currentCmd = CurrentCmd.GetWSD;
            byte[] cmd = new byte[8];
            cmd[0] = DevId;
            cmd[1] = 0x04;      // 读数据
            cmd[2] = 0;
            cmd[3] = 0x21;
            cmd[4] = 0;
            cmd[5] = 2;         // 0x21 温度， 0X22 湿度

            Send(cmd);
            ReqSuccessed();
            return Status.ErrorCode == 0;
        }

        /// <summary>
        /// 设备是否开机
        /// </summary>
        /// <returns></returns>
        private bool IsMachineWorking()
        {
            currentCmd = CurrentCmd.GetWorkState;
            byte[] cmd = new byte[8];
            cmd[0] = DevId;
            cmd[1] = 0x04;      // 读数据
            cmd[2] = 0;
            cmd[3] = 0x13;
            cmd[4] = 0;
            cmd[5] = 1;
            Send(cmd);
            ReqSuccessed();
            return Status.ErrorCode == 0;
        }

        /// <summary>
        /// 设置抽湿启动湿度
        /// </summary>
        /// <returns></returns>
        public bool SetHumid(byte humid)
        {
            ReqSuccessed();
            currentCmd = CurrentCmd.SetHumid;
            byte[] cmd = new byte[8];
            cmd[0] = DevId;
            cmd[1] = 0x06;      // 读数据
            cmd[2] = 0;
            cmd[3] = 0x14;
            cmd[4] = 0;
            cmd[5] = humid;
            Send(cmd);
            ReqSuccessed();
            return Status.ErrorCode == 0;
        }

        public bool GetSetHumid()
        {
            ReqSuccessed();
            currentCmd = CurrentCmd.GetSetHumid;
            byte[] cmd = new byte[8];
            cmd[0] = DevId;
            cmd[1] = 0x04;      // 读数据
            cmd[2] = 0;
            cmd[3] = 0x14;
            cmd[4] = 0;
            cmd[5] = 1;
            Send(cmd);
            ReqSuccessed();
            return Status.ErrorCode == 0;
        }

        /// <summary>
        /// 是否打开除湿机
        /// </summary>
        /// <param name="IsOpen"></param>
        /// <returns></returns>
        public bool OpenMachine(bool IsOpen)
        {
            ReqSuccessed();
            currentCmd = CurrentCmd.OpenMachine;
            byte[] cmd = new byte[8];
            cmd[0] = DevId;
            cmd[1] = 0x06;      // 读数据
            cmd[2] = 0;
            cmd[3] = 0x13;
            cmd[4] = 0;
            cmd[5] = (byte)(IsOpen ? 0xFF : 0);
            Send(cmd);
            ReqSuccessed();
            return Status.ErrorCode == 0;
        }
        #endregion

        /// <summary>
        /// 读写器发过来的数据处理
        /// </summary>
        /// <param name="data"></param>
        /// <param name="len"></param>
        private void DealData(byte[] msg)
        {
            try
            {
                // 设备号不一致
                if (msg[0] != DevId)
                    return;

                // CRC校验
                var crc = Crc16_ModBus(msg, msg.Length-2);
                if((msg[msg.Length-1] != ((crc >> 8) & 0xff)) || (msg[msg.Length - 2] != (crc & 0xff)))
                {
                    return;
                }

                switch (msg[1])
                {
                    case 0x04:  // 读寄存器
                        ReadReg(msg);
                        break;
                    case 0x06:  // 写寄存器
                        WriteReg(msg);
                        break;
                    default:
                        break;
                }
                bSending = false;
                currentCmd = CurrentCmd.Idle;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }
        }

        private void ReadReg(byte[] msg)
        {
            switch (currentCmd)
            {
                case CurrentCmd.GetWSD:
                    if (msg[2] == 4)
                    {
                        Status.Temp = msg[4];
                        Status.Humid = msg[6];
                        Status.ErrorCode = 0;
                    }
                    break;
                case CurrentCmd.GetWorkState:
                    if (msg[2] == 2)
                    {
                        Status.IsWorking = (msg[4] == 0xff);
                        Status.ErrorCode = 0;
                    }
                    break;
                case CurrentCmd.GetSetHumid:
                    if (msg[2] == 2)
                    {
                        Status.SetHumid = msg[4];
                        Status.ErrorCode = 0;
                    }
                    break;
            }
        }

        private void WriteReg(byte[] msg)
        {
            Status.ErrorCode = 0;
        }
        
        #region Common
        /// <summary>
        /// 阻塞式接收
        /// </summary>
        private void ReqSuccessed()
        {
            int timeout = 100;
            while (bSending)
            {
                Thread.Sleep(10);
                if (timeout-- < 0)
                {                    
                    bSending = false;
                    currentCmd = CurrentCmd.Idle;
                    break;
                }
            }
        }
		
		private void ReqSuccessed(int timeout)
        {
            timeout += 100;
            while (bSending)
            {
                Thread.Sleep(10);
                if (timeout-- < 0)
                {                    
                    bSending = false;
                    currentCmd = CurrentCmd.Idle;
                    break;
                }
            }
        }
        #endregion


        /// <summary>
        /// MODBUS-CRC运算
        /// </summary>
        /// <param name="data"></param>
        /// <param name="len">必须是偶数</param>
        /// <returns></returns>
        public static UInt16 Crc16_ModBus(byte[] data, int lenth)
        {
            ushort crc = 0xffff;
            for (int i = 0; i < lenth; i++)
            {
                crc = (ushort)(data[i] ^ crc);
                for (int j = 0; j < 8; j++)
                {
                    if ((crc & 0x01) == 0x01)
                    {
                        crc = (ushort)(crc >> 1);
                        crc = (ushort)(crc ^ 0xa001);
                    }
                    else
                    {
                        crc = (ushort)(crc >> 1);
                    }
                }
            }
            return crc;
        }

        #region Event
        private void CheckStatusChanged()
        {
            //var currentHashcode = Status.GetHashCode();
            if(!Status.Equals(lastStatusHashcode))
            {
                lastStatusHashcode.Humid = Status.Humid;
                lastStatusHashcode.Temp = Status.Temp;
                lastStatusHashcode.IsWorking = Status.IsWorking;
                lastStatusHashcode.SetHumid = Status.SetHumid;
                lastStatusHashcode.ErrorCode = Status.ErrorCode;
                OnStatusChanged(Status);
            }
        }

        // GPI事件
        public delegate void StatusChanged(MachineStatus args);
        public event StatusChanged OnStatusChanged;
        #endregion
    }
}
