﻿using System;
using System.Text;
using System.Globalization;
using System.Text.RegularExpressions;
using System.Security.Cryptography;
using System.Linq.Expressions;
using System.Collections.Generic;

namespace JmpDehumidifierLib
{
    public class Common
    {
        public Common() { }


        #region Convert
        public static byte[] ToHexByte(string hexString)
        {
            hexString = hexString.Replace(" ", "");
            if ((hexString.Length % 2) != 0)
            {
                hexString = hexString + " ";
            }
            byte[] buffer = new byte[hexString.Length / 2];
            for (int i = 0; i < buffer.Length; i++)
            {
                buffer[i] = Convert.ToByte(hexString.Substring(i * 2, 2), 0x10);
            }
            return buffer;
        }

        public static string ToHexString(byte[] bytes)
        {
            string hexString = string.Empty;
            if (bytes != null)
            {
                StringBuilder strB = new StringBuilder();
                for (int i = 0; i < bytes.Length; i++)
                {
                    strB.Append(bytes[i].ToString("X2"));
                }
                hexString = strB.ToString();
            }

            return hexString;
        }


        /// <summary>
        /// 数组大小端转换
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static byte[] ReserveBytes(byte[] bytes, UInt16 len)
        {
            byte[] ret = new byte[len];
            for(int i =0; i < len; i++)
            {
                ret[len - i - 1] = bytes[i];
            }
            return ret;
        }

        /// <summary>
        /// 数组转字符串
        /// </summary>
        /// <param name="bytes"></param>
        /// <param name="offset"></param>
        /// <param name="lenth"></param>
        /// <returns></returns>
        public static string ToHexString(byte[] bytes, int offset, int lenth)
        {
            string hexString = string.Empty;
            if (bytes != null)
            {
                StringBuilder strB = new StringBuilder();
                for (int i = offset; i < lenth + offset; i++)
                {
                    strB.Append(bytes[i].ToString("X2"));
                }
                hexString = strB.ToString();
            }

            return hexString;
        }

        internal static bool ValidateHex_uint(string input, out uint value)
        {
            if (!uint.TryParse(input, NumberStyles.AllowHexSpecifier, null, out value))
            {
                return false;
            }

            return true;
        }

        internal static bool ValidateHex_ushort(string input, out ushort value)
        {
            if (!ushort.TryParse(input, NumberStyles.AllowHexSpecifier, (IFormatProvider)null, out value))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 是否为十进制字符串
        /// </summary>
        /// <param name="strIn"></param>
        /// <returns></returns>
        internal static bool IsInteger(string strIn)
        {
            bool rtn = true;
            if (strIn != "")
            {
                foreach (char ch in strIn)
                {
                    if (Char.IsNumber(ch))
                    {
                        continue;
                    }
                    else
                    {
                        rtn = false;
                    }
                }
            }
            else
            {
                rtn = false;
            }

            return rtn;
        }
        
        /// <summary>
        /// 是否为16进制字符串
        /// </summary>
        /// <param name="str"></param>
        /// <returns></returns>
        public static bool IsHex(string str)
        {
            const string pattern = @"[A-Fa-f0-9]+$";
            return System.Text.RegularExpressions.Regex.IsMatch(str, pattern);
        }
        #endregion
                
        /// <summary>
        /// 验证是否为数字
        /// </summary>
        /// <param name="number">要验证的数字</param>    
        public static bool IsNumber(string number)
        {
            try
            {
                //如果为空，认为验证不合格
                if (number == null || number == string.Empty)
                {
                    return false;
                }
                //清除要验证字符串中的空格
                number = number.Trim();
                //模式字符串
                string pattern = @"^(-)?[0-9]+[0-9]*[.]?[0-9]*$";
                //验证
                return Regex.IsMatch(number, pattern, RegexOptions.IgnoreCase);
            }
            catch (System.Exception)
            {
                return false;
            }
        }

        /// <summary>
        /// 获取枚举的描述
        /// </summary>
        /// <param name="enumValue">枚举变量</param>
        /// <returns>描述</returns>
        public static string GetEnumDescription(Enum enumValue)
        {
            string str = enumValue.ToString();
            System.Reflection.FieldInfo field = enumValue.GetType().GetField(str);
            object[] objs = field.GetCustomAttributes(typeof(System.ComponentModel.DescriptionAttribute), false);
            if (objs == null || objs.Length == 0) return str;
            System.ComponentModel.DescriptionAttribute da = (System.ComponentModel.DescriptionAttribute)objs[0];
            return da.Description;
        }

        /// <summary>
        /// 返回两个时间差(毫秒数)
        /// </summary>
        /// <param name="DateTime1"></param>
        /// <param name="DateTime2"></param>
        /// <returns></returns>
        public static double DateDiffMillSecond(DateTime DateTime1, DateTime DateTime2)
        {
            return new TimeSpan(DateTime1.Ticks).Subtract(new TimeSpan(DateTime2.Ticks)).Duration().TotalMilliseconds;
        }

        #region 转换
        /// <summary>
        /// byte to ushort
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static UInt64 BytesToUInt64(byte[] bytes, byte pos)
        {
            UInt64 rtn = 0;
            if (bytes != null)
            {
                for (int i = pos; i < pos + 8; i++)
                {
                    rtn = rtn << 8 | bytes[i];
                }

                return rtn;
            }

            return rtn;
        }

        /// <summary>
        /// byte to int
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static Int32 BytesToInt32(byte[] bytes, byte pos)
        {
            Int32 rtn = 0;
            if (bytes != null)
            {
                for (int i = pos; i < pos + 4; i++)
                {
                    rtn = rtn << 8 | bytes[i];
                }

                return rtn;
            }

            return rtn;
        }

        /// <summary>
        /// byte to ushort
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static Int16 BytesToInt16(byte[] bytes, byte pos)
        {
            Int16 rtn = 0;
            if (bytes != null)
            {
                for (int i = pos; i < pos + 2; i++)
                {
                    rtn = (Int16)(rtn << 8 | bytes[i]);
                }
            }

            return rtn;
        }

        /// <summary>
        /// byte to int
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static UInt32 BytesToUInt32(byte[] bytes, byte pos)
        {
            UInt32 rtn = 0;
            if (bytes != null)
            {
                for (int i = pos; i < pos + 4; i++)
                {
                    rtn = rtn << 8 | bytes[i];
                }
            }

            return rtn;
        }

        /// <summary>
        /// byte to ushort
        /// </summary>
        /// <param name="bytes"></param>
        /// <returns></returns>
        public static UInt16 BytesToUInt16(byte[] bytes, byte pos)
        {
            UInt16 rtn = 0;
            if (bytes != null)
            {
                for (int i = pos; i < pos + 2; i++)
                {
                    rtn = (UInt16)(rtn << 8 | bytes[i]);
                }
            }

            return rtn;
        }

        /// <summary>
        /// IP int to string
        /// </summary>
        /// <param name="ipInt"></param>
        /// <returns></returns>
        public static string IntToIp(long ipInt)
        {
            StringBuilder sb = new StringBuilder();
            sb.Append((ipInt >> 24) & 0xFF).Append(".");
            sb.Append((ipInt >> 16) & 0xFF).Append(".");
            sb.Append((ipInt >> 8) & 0xFF).Append(".");
            sb.Append(ipInt & 0xFF);
            return sb.ToString();
        }

        /// <summary>
        /// ip string to int
        /// </summary>
        /// <param name="ip"></param>
        /// <returns></returns>
        public static byte[] IpToBytes(string ip)
        {
            byte[] ipAddr = new byte[4] { 0, 0, 0, 0 };
            char[] separator = new char[] { '.' };
            string[] items = ip.Split(separator);
            ipAddr[0] = Byte.Parse(items[0]);
            ipAddr[1] = Byte.Parse(items[1]);
            ipAddr[2] = Byte.Parse(items[2]);
            ipAddr[3] = Byte.Parse(items[3]);
            return ipAddr;
        }

        public static byte[] UInt32ToBytes(UInt32 val)
        {
            if(val > 0x10000)
            {
                byte[] rtn = new byte[4];
                rtn[0] = (byte)(val >> 24);
                rtn[1] = (byte)(val >> 16);
                rtn[2] = (byte)(val >> 8);
                rtn[3] = (byte)(val);
                return rtn;
            }
            else
            {
                byte[] rtn = new byte[2];
                rtn[0] = (byte)(val >> 8);
                rtn[1] = (byte)(val);
                return rtn;
            }
        }
        #endregion

        /// <summary>
        /// 获取MD5密文
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        public static string GetCiper(string input)
        {
            // Use input string to calculate MD5 hash
            MD5 md5 = System.Security.Cryptography.MD5.Create();
            byte[] inputBytes = System.Text.Encoding.ASCII.GetBytes(input);
            byte[] hashBytes = md5.ComputeHash(inputBytes);

            // Convert the byte array to hexadecimal string
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < hashBytes.Length; i++)
            {
                sb.Append(hashBytes[i].ToString("X2"));
            }
            return sb.ToString();
        }
    }


    public static class TransExpV2<TIn, TOut>
    {

        private static readonly Func<TIn, TOut> cache = GetFunc();
        public static Func<TIn, TOut> GetFunc()
        {
            ParameterExpression parameterExpression = Expression.Parameter(typeof(TIn), "p");
            List<MemberBinding> memberBindingList = new List<MemberBinding>();

            foreach (var item in typeof(TOut).GetProperties())
            {
                if (!item.CanWrite)
                    continue;

                MemberExpression property = Expression.Property(parameterExpression, typeof(TIn).GetProperty(item.Name));
                MemberBinding memberBinding = Expression.Bind(item, property);
                memberBindingList.Add(memberBinding);
            }

            MemberInitExpression memberInitExpression = Expression.MemberInit(Expression.New(typeof(TOut)), memberBindingList.ToArray());
            Expression<Func<TIn, TOut>> lambda = Expression.Lambda<Func<TIn, TOut>>(memberInitExpression, new ParameterExpression[] { parameterExpression });

            return lambda.Compile();
        }

        public static TOut Trans(TIn tIn)
        {
            return cache(tIn);
        }

    }
}
