package com.junmp.jyzb.logManager;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.junmp.jyzb.logManager.service.EsLogService;
import com.junmp.v2.log.api.LogRecordApi;
import com.junmp.v2.log.api.bean.dto.LogRecordDto;
import com.junmp.v2.log.api.thread.LogManagerPool;
import com.junmp.v2.log.db.entity.SysLog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;


public class ESRecordManager implements LogRecordApi {
    private static final Logger log = LoggerFactory.getLogger(com.junmp.v2.log.db.DbLogRecordManager.class);
    private final EsLogService esService;
    private final LogManagerPool logManagerPool;
    private final LogRefreshManager logRefreshManager;

    public ESRecordManager(LogManagerPool logManagerPool, EsLogService esService) {
        this.logManagerPool = logManagerPool;
        this.esService = esService;
        this.logRefreshManager = new ESRecordManager.LogRefreshManager();
        this.logRefreshManager.start();
    }


    public void add(LogRecordDto logRecord) {
        if (null != logRecord) {
            this.addBatch(CollectionUtil.list(false, new LogRecordDto[]{logRecord}));
        }
    }

    public void addAsync(final LogRecordDto logRecord) {
        this.logManagerPool.executeLog(new TimerTask() {
            public void run() {
                ESRecordManager.this.logRefreshManager.putLog(logRecord);
            }
        });
    }

    public void addBatch(List<LogRecordDto> logRecords) {
        if (!ObjectUtil.isEmpty(logRecords)) {
            List<SysLog> sysLogList = (List)logRecords.stream().map((dto) -> {
                SysLog sysLog = new SysLog();
                BeanUtil.copyProperties(dto, sysLog, new String[0]);
                if (StrUtil.isEmpty(sysLog.getLogName())) {
                    sysLog.setLogName("API接口日志记录");
                }

                if (StrUtil.isEmpty(sysLog.getAppName())) {
                    sysLog.setAppName("none-app-name");
                }

                return sysLog;
            }).collect(Collectors.toList());
            if (ObjectUtil.isNotEmpty(sysLogList)) {
//                this.esService.saveBatch(sysLogList);
            }

        }
    }

    class LogRefreshManager extends Thread {
        private final long sleepTime;
        private final int maxCount;
        private final AtomicLong refreshMark = new AtomicLong();
        public AtomicInteger count = new AtomicInteger(0);
        private final Queue<LogRecordDto> queue = new ConcurrentLinkedQueue();

        public LogRefreshManager() {
            this.sleepTime = 3000L;
            this.maxCount = 300;
        }

        public LogRefreshManager(long sleepTime) {
            this.sleepTime = sleepTime;
            this.maxCount = 300;
        }

        public LogRefreshManager(int maxCount) {
            this.sleepTime = 3000L;
            this.maxCount = maxCount;
        }

        public LogRefreshManager(long sleepTime, int maxCount) {
            this.sleepTime = sleepTime;
            this.maxCount = maxCount;
        }

        public void putLog(LogRecordDto record) {
            int queueDataCount = this.count.get();
            if (queueDataCount == 0) {
                this.refreshMark.getAndSet(System.currentTimeMillis());
            }

            if (queueDataCount < this.maxCount * 2) {
                this.queue.offer(record);
                this.count.incrementAndGet();
            }
        }

        private void refresh() {
            this.refreshMark.getAndSet(System.currentTimeMillis());
            int num = this.count.getAndSet(0);
            List<LogRecordDto> cacheAll = new ArrayList(num);

            for(int i = 0; i < num; ++i) {
                LogRecordDto item = (LogRecordDto)this.queue.poll();
                if (null == item) {
                    break;
                }

                cacheAll.add(item);
            }

//            if (null != cacheAll && cacheAll.size() > 0) {
//                this.addBatch(cacheAll);
//            }

        }

        private void timing() {
            long currentTimeMillis = System.currentTimeMillis();
            if (this.refreshMark.get() + this.sleepTime <= currentTimeMillis && this.count.get() > 0) {
                this.refresh();
            }

        }

        private void listener() {
            if (this.count.get() >= this.maxCount) {
                this.refresh();
            }

        }

        public void run() {
            try {
                while(true) {
                    this.listener();
                    this.timing();
                    TimeUnit.MILLISECONDS.sleep(10L);
                }
            } catch (InterruptedException var2) {
                if (ESRecordManager.log.isDebugEnabled()) {
                    var2.printStackTrace();
                }

                ESRecordManager.log.error(var2.getMessage());
            }
        }
    }
}