package com.junmp.jyzb.controller;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.junmp.jyzb.api.bean.dto.ProductBatchDto;
import com.junmp.jyzb.api.bean.dto.ProductProcessDto;
import com.junmp.jyzb.api.bean.req.ProductBatchReq;
import com.junmp.jyzb.api.exception.enums.ProductExceptionEnum;
import com.junmp.jyzb.api.util.GenCodeUtil;
import com.junmp.jyzb.doc.ProductBatchDoc;
import com.junmp.jyzb.entity.ProductBatch;
import com.junmp.jyzb.entity.ProductTag;
import com.junmp.jyzb.service.ProductBatchService;
import com.junmp.jyzb.service.ProductTagService;
import com.junmp.v2.common.bean.request.ValidationApi;
import com.junmp.v2.common.bean.response.ApiRes;
import com.junmp.v2.common.exception.base.ServiceException;
import com.junmp.v2.common.exception.enums.DefaultBizExceptionEnum;
import com.junmp.v2.db.api.page.PageResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <pre>
 *
 * 描述：
 * 版本：1.0.0
 * 日期：2022/8/1 17:59
 * 作者：ningzp@junmp.com.cn
 * <br>修改记录
 * <br>修改日期        修改人          修改内容
 *     20221109       ning           修改初始化重复成品码严重的BUG.
 * </pre>
 */
@RestController
@Slf4j
@RequestMapping("/api")
public class ProductBatchController implements ProductBatchDoc {

    @Resource
    private ProductBatchService productBatchService;

    @Resource
    private ProductTagService productTagService;


    @PostMapping(path = "/batch/produce", name = "生产产品批次")
    public ApiRes<Boolean> makeProduct(@RequestBody @Validated(ValidationApi.detail.class) ProductBatchReq req) {
        if (ObjectUtil.isNotNull(req) && ObjectUtil.isNotNull(req.getBatchId())) {
            ProductBatchDto dto = productBatchService.getBatchDetail(req.getBatchId());
            List<ProductTag> list = batchTagInfo(dto);
            if (null != list && list.size() > 0) {
                productTagService.saveBatch(list);
                //更新已完成
                ProductBatch pb = new ProductBatch();
                pb.setBatchId(req.getBatchId());
                pb.setHasInit(true);
                productBatchService.updateById(pb);

                return ApiRes.success();
            } else {
                return ApiRes.failure("没有可生产数据");
            }
        } else {
            return ApiRes.failure(DefaultBizExceptionEnum.VALID_PARAMS_ERROR);
        }
    }

    @PostMapping(path = "/batch", name = "新增产品批次")
    @Override
    public ApiRes<Boolean> addProductBatch(@RequestBody @Validated(ValidationApi.add.class) ProductBatchReq req) {
        boolean flag = productBatchService.addProductBatch(req);
        if (flag) {
            return ApiRes.success();
        } else {
            log.warn(StrUtil.format("请求的参数：{}", JSONUtil.toJsonStr(req)));
            return ApiRes.failure(StrUtil.format("保存失败"));
        }
    }

    @PutMapping(path = "/batch", name = "更新产品批次")
    @Override
    public ApiRes<Boolean> editProductBatch(@RequestBody @Validated(ValidationApi.edit.class) ProductBatchReq req) {
        boolean flag = productBatchService.editProductBatch(req);
        if (flag) {
            return ApiRes.success();
        } else {
            log.warn(StrUtil.format("请求的参数：{}", JSONUtil.toJsonStr(req)));
            return ApiRes.failure(StrUtil.format("更新失败"));
        }
    }

    @DeleteMapping(path = "/batch", name = "删除产品批次")
    @Override
    public ApiRes<Boolean> delBatch(@RequestBody @Validated(ValidationApi.delete.class) ProductBatchReq req) {
        try {
            boolean flag = productBatchService.del(req);
            if (flag) {
                return ApiRes.success();
            } else {
                return ApiRes.failure("删除失败");
            }
        } catch (ServiceException ex) {
            return ApiRes.failure(ex.getMessage());
        }
    }

    @GetMapping(path = "/batch/page", name = "分页批次数据")
    @Override
    public ApiRes<PageResult<ProductBatchDto>> getPage(ProductBatchReq req) {
        return ApiRes.success(productBatchService.findPage(req));
    }

    @GetMapping(path = "/batch/process/page", name = "分页批次数据")
    @Override
    public ApiRes<PageResult<ProductProcessDto>> getProcessPage(ProductBatchReq req) {
        return ApiRes.success(productBatchService.findProcessPage(req));
    }

    /**
     * 批次信息
     *
     *
     */
    private List<ProductTag> batchTagInfo(ProductBatchDto batch) {
        if (batch.getHasInit()) {
            throw new ServiceException(ProductExceptionEnum.BATCH_PROCESSED_DONE);
        } else {
            List<ProductTag> list = new ArrayList<>();
            Integer total = batch.getTotal();
            //修改已知的BUG  。重复成品码的根源
            Integer initVal = batch.getInitNum() > 0 ? batch.getInitNum() + 1 : 1;
            for (; initVal <= total; initVal++) {
                ProductTag tag = new ProductTag();
                tag.setBatchId(batch.getBatchId());
                tag.setProductId(batch.getProductId());
                tag.setSpecId(batch.getSpecId());
                tag.setProduceTime(new Date());

                tag.setSerialId(initVal);
                //成品码
                StringBuilder sb = new StringBuilder();
                String serialNo = GenCodeUtil.splicingZero(String.valueOf(initVal), 4);
                sb.append(batch.getBatchNo())
                        .append(serialNo);
                tag.setProductCode(sb.toString());


                tag.setHasInit(false);
                tag.setHasChecked(false);
                list.add(tag);
            }
            return list;
        }

    }
}
