package com.junmp.jyzb.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.HexUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.junmp.jyzb.api.bean.dto.ProductTagDto;
import com.junmp.jyzb.api.bean.query.ProductTagReq;
import com.junmp.jyzb.api.bean.vo.TagInfoVo;
import com.junmp.jyzb.api.exception.enums.ProductExceptionEnum;
import com.junmp.jyzb.doc.ProductTagDoc;
import com.junmp.jyzb.entity.ProductTag;
import com.junmp.jyzb.service.ProductTagService;
import com.junmp.v2.common.bean.response.ApiRes;
import com.junmp.v2.common.exception.enums.DefaultBizExceptionEnum;
import com.junmp.v2.db.api.page.PageResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

/**
 * <pre>
 *
 * 描述：
 * 版本：1.0.0
 * 日期：2022/8/1 17:59
 * 作者：ningzp@junmp.com.cn
 * <br>修改记录
 * <br>修改日期        修改人          修改内容
 *
 * </pre>
 */
@RestController
@Slf4j
@RequestMapping("/api")
public class ProductTagController implements ProductTagDoc {

    @Resource
    private ProductTagService productTagService;


    /**
     * 前端参数包括：batchId 和 tagUid
     */
    @PostMapping(path = "/tag/updateUid", name = "更新标签UID")
    @Override
    public ApiRes<TagInfoVo> updateUid(@RequestBody ProductTagReq req) {
        ProductTag info = productTagService.checkTagUid(req);
        info.setTagUid(req.getTagUid());
        info.setHasInit(true);
        boolean flag = productTagService.updateById(info);
        if (flag) {
            String hexStr = HexUtil.encodeHexStr(info.getProductCode());
            return ApiRes.success(TagInfoVo.builder().tagId(info.getTagId()).content(hexStr).build());
        } else {
            return ApiRes.failure("UID数据更新失败");
        }
    }


    /**
     * 前端参数包括：tagId,password,ciphertext
     */
    @PostMapping(path = "/tag/updateContent", name = "更新内容")
    @Override
    public ApiRes<TagInfoVo> updateContent(@RequestBody ProductTagReq req) {
        if (ObjectUtil.isNotNull(req) && ObjectUtil.isNotNull(req.getTagId())) {
            ProductTag info = productTagService.getById(req.getTagId());
            if (null != info) {
                info.setPassword(req.getPassword());
                info.setCiphertext(req.getCiphertext());
                boolean flag = productTagService.updateById(info);
                if (flag) {
                    //查询视图获取数据
                    ProductTagDto dto = productTagService.getTagInfo(req.getTagId());
                    StringBuilder tmp = new StringBuilder();
                    Set<String> params = new LinkedHashSet<>();

                    if (StrUtil.isNotBlank(dto.getCompanyName())) {
                        tmp.append("厂家:{}\n");
                        params.add(dto.getCompanyName());
                    }
                    if (StrUtil.isNotBlank(dto.getProductName())) {
                        tmp.append("名称:{}\n");
                        params.add(dto.getProductName());
                    }
                    if (StrUtil.isNotBlank(dto.getSpec())) {
                        tmp.append("规格:{}\n");
                        params.add(dto.getSpec());
                    }
                    if (StrUtil.isNotBlank(dto.getMaterial())) {
                        tmp.append("材质:{}\n");
                        params.add(dto.getMaterial());
                    }
                    if (StrUtil.isNotBlank(dto.getWeight())) {
                        tmp.append("重量:{}\n");
                        params.add(dto.getWeight());
                    }
                    //厂家:{}\n日期:{}
                    tmp.append("编号:{}\n");
                    params.add(dto.getProductCode());
                    //写入广告语
                    if (StrUtil.isNotBlank(dto.getAdInfo())) {
                        tmp.append("{}\n");
                        params.add(dto.getAdInfo());
                    }
                    log.info(StrUtil.format("待写入的模板:{}", tmp.toString()));
                    //String abc = "名称:{}\n规格:{}\n重量:{}\n厂家:{}\n日期:{}\n编号:{}\n";
                    String content = StrUtil.format(tmp.toString(), params.toArray());
                    log.info(StrUtil.format("写入的内容:{}", content));
                    TagInfoVo infoVo = TagInfoVo.builder()
                            .tagId(req.getTagId())
                            .productCode(dto.getCiphertext())
                            .content(content).build();

                    log.info(StrUtil.format("写入芯片的内容>>", JSONUtil.toJsonStr(info)));

                    return ApiRes.success(infoVo);
                } else {
                    return ApiRes.failure("UID数据更新失败");
                }
            } else {
                return ApiRes.failure(ProductExceptionEnum.TAG_INFO_NOT_EXISTS);
            }
        } else {
            return ApiRes.failure(DefaultBizExceptionEnum.VALID_PARAMS_ERROR);
        }
    }


    /**
     * 前端参数：tagUid,productCode
     */
    @PostMapping(path = "/tag/check", name = "复检接口")
    public ApiRes<Boolean> checkTag(@RequestBody ProductTagReq req) {
        if (ObjectUtil.isNotNull(req) &&
                ObjectUtil.isNotNull(req.getTagUid()) &&
                ObjectUtil.isNotNull(req.getProductCode())) {
            String code = HexUtil.decodeHexStr(req.getProductCode());
            req.setProductCode(code);

            List<ProductTag> list = productTagService.findList(req);
            if (CollectionUtil.isNotEmpty(list) && list.size() == 1) {
                ProductTag tag = list.get(0);
                log.info(StrUtil.format("复检结果为：{}", JSONUtil.toJsonStr(tag)));
                if (tag.getHasChecked()) {
                    log.warn("该产品已完成自检，编号为:{}", req.getTagUid());
                    return ApiRes.success();
                }
                if (ObjectUtil.isNotNull(tag)) {
                    //更新为已经复检
                    tag.setHasChecked(true);
                    tag.setCheckTime(new Date());
                    productTagService.updateById(tag);
                    return ApiRes.success();
                } else {
                    return ApiRes.failure(ProductExceptionEnum.TAG_INFO_NOT_EXISTS);
                }
            } else {
                return ApiRes.failure(ProductExceptionEnum.TAG_INFO_REPEAT_ERROR);
            }
        } else {
            return ApiRes.failure(DefaultBizExceptionEnum.VALID_PARAMS_ERROR);
        }
    }


    @GetMapping(path = "/tag/page", name = "分页标签")
    public ApiRes<PageResult<ProductTag>> getPage(ProductTagReq req) {
        return ApiRes.success(productTagService.findPage(req));
    }
}
