package com.junmp.jyzb.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.junmp.jyzb.api.bean.dto.PoliceFingerDto;
import com.junmp.jyzb.api.bean.dto.PolicemanDto;
import com.junmp.jyzb.api.bean.query.CabinetReq;
import com.junmp.jyzb.api.bean.query.PoliceReassignmentReq;
import com.junmp.jyzb.api.bean.query.PolicemanReq;
import com.junmp.jyzb.api.bean.req.UpdatePolicemanReq;
import com.junmp.jyzb.api.exception.enums.*;
import com.junmp.jyzb.entity.*;
import com.junmp.jyzb.mapper.*;
import com.junmp.jyzb.service.*;
import com.junmp.jyzb.utils.*;
import com.junmp.v2.auth.password.BcryptPasswordStoredEncrypt;
import com.junmp.v2.common.exception.base.ServiceException;
import com.junmp.v2.common.util.BeanPlusUtil;
import com.junmp.v2.db.api.factory.PageFactory;
import com.junmp.v2.db.api.factory.PageResultFactory;
import com.junmp.v2.db.api.page.PageResult;
import com.junmp.v2.sys.api.bean.user.req.SysUserReq;
import com.junmp.v2.sys.api.bean.user.req.UserOrgReq;
import com.junmp.v2.sys.user.entity.SysUser;
import com.junmp.v2.sys.user.service.SysUserOrgService;
import com.junmp.v2.sys.user.service.SysUserService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;


@Service
@Slf4j
public class PolicemanServiceImpl extends ServiceImpl<PolicemanMapper, Policeman> implements PolicemanService {

    @Resource
    private RedisUtils redisUtils;

    @Resource
    private PolicemanMapper policemanMapper;


    @Resource
    private PubOrgService pubOrgService;

    @Resource
    private CabinetService cabinetService;

    @Resource
    private CabinetBoxService cabinetBoxService;

    @Resource
    private PoliceFingerService policeFingerService;

    @Resource
    private SysUserService sysUserService;

    @Resource
    private PoliceEquipmentService policeEquipmentService;



    @Resource
    private SysUserOrgService sysUserOrgService;

    private static final String REDIS_POLICE = "Policeman_";
    private static final String REDIS_ORG = "Organization_";


    /**
     * 添加警员信息
     *
     * @param req
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String AddPoliceman(UpdatePolicemanReq req) {
        //查询传入的组织机构id是否存在
        pubOrgService.PubOrgExist(req.getOrgId());
        //查询是否有重复的policeCode
        List<Policeman> list = list(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(req.getPoliceCode()), Policeman::getPoliceCode, req.getPoliceCode()));
        if (list.size() > 0) {
            throw new ServiceException(PolicemanExceptionEnum.POLICECODE_IS_EXISTS);
        }
        if (!req.getSex().equals("男") &&!req.getSex().equals("女")){
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        Policeman policeman = new Policeman();
        BeanPlusUtil.copyProperties(req, policeman);
        policeman.setState(1);
        save(policeman);
        //判断是否需要同步生成账号，如果为true则生成账号绑定
        if (req.getIsCreatedAccount()==1){
            PolicemanReq policemanReq = new PolicemanReq();
            BeanPlusUtil.copyProperties(req,policemanReq);
            policemanReq.setAccount(req.getPoliceCode());
            PoliceBindUser(policemanReq);
        }
        return policeman.getId();
    }

    // 更新缓存的方法，新增警员信息时调用
    private void updatePoliceIdCache(String orgId, String newPoliceId) {
        // 构建 Redis 缓存键
        String redisKeyFalse = "getAllPoliceId_false_" + orgId;
        String redisKeyTrue = "getAllPoliceId_true_" + orgId;

        // 从 Redis 中获取旧的缓存数据
        List<String> cachedDataFalse = redisUtils.findCachedData(redisKeyFalse);
        List<String> cachedDataTrue = redisUtils.findCachedData(redisKeyTrue);

        //如果缓存中没有数据，则直接查询数据库并更新Redis缓存
        //如果缓存中有数据，就更新数据
        if (cachedDataFalse == null && cachedDataTrue == null) {
            //getAllPoliceId(Map.of("IncludeLowerLevel", false, "orgId", orgId));
            return;
        } else if (cachedDataFalse != null) {
            cachedDataFalse.add(newPoliceId);
            redisUtils.set(redisKeyFalse, cachedDataFalse);
        } else if (cachedDataTrue != null) {
            cachedDataTrue.add(newPoliceId);
            redisUtils.set(redisKeyTrue, cachedDataTrue);
        }
    }

    //删除缓存的方法
    private void deletePoliceIdCache(String orgId) {
        //构建 Redis 缓存键
        String redisKey = "getAllPoliceId_false_" + orgId;
        redisUtils.delete(redisKey);

        //构建 Redis 缓存键
        redisKey = "getAllPoliceId_true_" + orgId;
        redisUtils.delete(redisKey);
    }


    /**
     * @param req
     * @return
     */
    @Override
    public PolicemanDto GetOnePolice(PolicemanReq req) {
        Policeman one = PoliceExist(req.getId());
        PolicemanDto policemanDto = new PolicemanDto();
        BeanPlusUtil.copyProperties(one, policemanDto);
        //将警员的指纹信息查询出来
        List<PoliceFinger> list1 = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                .eq(ObjectUtil.isNotNull(req.getId()), PoliceFinger::getPoliceId, req.getId()));
        List<PoliceFingerDto> fingerDtoList = new ArrayList<>();
        for (PoliceFinger policeFinger : list1) {
            PoliceFingerDto policeFingerDto = new PoliceFingerDto();
            BeanPlusUtil.copyProperties(policeFinger, policeFingerDto);
            fingerDtoList.add(policeFingerDto);
        }
        policemanDto.setFingersList(fingerDtoList);
        return policemanDto;
    }


    /**
     * 删除警员信息
     *
     * @param req
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean DeletePolice(UpdatePolicemanReq req) {
        //通过警员id查询警员信息
        Policeman policeman = PoliceExist(req.getId());
        //如果查询出来的警员信息中没有绑定箱门而参数传递了箱门id，则抛出异常警员绑定箱门错误
        //如果传递的箱门id参数与查询出来的箱门id不一致，则抛出异常警员绑定的箱门与实际绑定不符合
        if ((ObjectUtil.isNull(policeman.getCabinetBoxId()) && ObjectUtil.isNotNull(req.getCabinetBoxId()))) {
            throw new ServiceException(CabinetBoxExceptionEnum.CABINETBOX_ISNOT_EXISTS);
        }
        //通过警员信息中的userId对账号进行删除
        sysUserService.remove(new LambdaQueryWrapper<SysUser>()
                .eq(SysUser::getUserId, policeman.getUserId()));
        //
        return removeById(req.getId());

    }


    /**
     * 修改警员信息
     *
     * @param req
     * @return
     */
    @Override
    public boolean UpdatePolice(UpdatePolicemanReq req) {
        if (!req.getSex().equals("男") &&!req.getSex().equals("女")){
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        Policeman policeman = new Policeman();
        BeanPlusUtil.copyProperties(req, policeman);
        //查询传入的组织机构id是否存在
        pubOrgService.PubOrgExist(req.getOrgId());
        //判断传入的修改的警员编号与本身编号一致，如果一致则无所谓，如果不一致则需要判断修改的警员编号是否已经存在
        //判断警员是否存在
        Policeman one = PoliceExist(req.getId());
        if (!req.getPoliceCode().equals(one.getPoliceCode())) {
            Policeman one1 = getOne(new LambdaQueryWrapper<Policeman>()
                    .eq(ObjectUtil.isNotNull(req.getPoliceCode()), Policeman::getPoliceCode, req.getPoliceCode()));
            if (ObjectUtil.isNotNull(one1)) {
                throw new ServiceException(PolicemanExceptionEnum.POLICECODE_IS_EXISTS);
            }
        }
        //判断是否传递了箱子id，如果传了则判断箱子是否存在
        if (ObjectUtil.isNotNull(req.getCabinetBoxId()) && !req.getCabinetBoxId().equals("")){
            //查询单警柜箱号是否存在,箱门不存在抛异常
            cabinetBoxService.CabinetBoxExist(req.getCabinetBoxId());
        }
        policeman.setUpdateTime(DateTimeUtil.getCurrentDateTime());
        return updateById(policeman);
    }


    /**
     * 修改警员状态
     *
     * @param req
     * @return
     */
    @Override
    public boolean ChangePoliceState(UpdatePolicemanReq req) {
        //通过警员id查询出警员信息
        Policeman one = PoliceExist(req.getId());
        //判断状态码是否是0或1
        if (req.getState()!=1 && req.getState()!=0){
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        one.setState(req.getState());
        return updateById(one);

    }

    public List<Map<String, Object>> getAllPolicemanMsg(Map<String, Object> msg, List<String> allPoliceId) {
        List<Map<String, Object>> allPoliceman = new ArrayList<>();
        // 构建 Redis 缓存键
        String redisKey = REDIS_ORG + msg.get("IncludeLowerLevel").toString() + '_' + msg.get("orgId").toString();
        // 从 Redis 中获取数据
        List<Map<String, Object>> cachedData = redisUtils.findCachedData(redisKey);
        if (cachedData != null) {
            //将Long类型的数据转化为Date
            for (Map<String, Object> one : cachedData) {
                Long dataMsg = (Long) one.get("updateTime");
                one.put("updateTime", redisUtils.getDate(dataMsg));
                allPoliceman.add(one);
            }
            return cachedData;
        }
        for (String policeId : allPoliceId) {
            Map<String, Object> policeMsg = policemanMapper.getOnePolice(policeId);
            allPoliceman.add(policeMsg);
        }

        //按照updateTime降序排序
        DateTimeUtil.timeSort(allPoliceman);

        //将查询结果存入 Redis 中
        redisUtils.set(redisKey, allPoliceman);
        return allPoliceman;
    }

    /**
     * 通过条件查询警员列表(list)
     *
     * @param
     */
    @Override
    public List<PolicemanDto> GetAllPolicemanList(PolicemanReq req) {
        //查询传入的组织机构id是否存在
        PubOrg pubOrg = pubOrgService.PubOrgExist(req.getOrgId());
        //查询出组织机构下的所有警员信息(如果是查看启用的那么state为1)
        List<Policeman> list = list(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(req.getOrgId()), Policeman::getOrgId, req.getOrgId())
                .like(ObjectUtil.isNotEmpty(req.getName()),Policeman::getName,req.getName())
                .like(ObjectUtil.isNotEmpty(req.getPoliceCode()),Policeman::getPoliceCode,req.getPoliceCode())
                .eq(ObjectUtil.isNotEmpty( req.getState()),Policeman::getState, req.getState()));
        List<PolicemanDto> policeList = new ArrayList<>();
        //查询出警员信息下的指纹列表信息
        for (Policeman p : list) {
            PolicemanDto policemanDto = new PolicemanDto();
            BeanPlusUtil.copyProperties(p, policemanDto);
            List<PoliceFinger> list1 = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                    .eq(ObjectUtil.isNotNull(p.getId()), PoliceFinger::getPoliceId, p.getId()));
            List<PoliceFingerDto> policeFingerDtoList = new ArrayList<>();
            for (PoliceFinger policeFinger : list1) {
                PoliceFingerDto policeFingerDto = new PoliceFingerDto();
                BeanPlusUtil.copyProperties(policeFinger, policeFingerDto);
                policeFingerDtoList.add(policeFingerDto);
            }
            policemanDto.setFingersList(policeFingerDtoList);
            policemanDto.setOrgName(pubOrg.getOrgName());
            policeList.add(policemanDto);

        }
        return policeList;
    }

    //通过组织机构id查询警员列表(page)
    @Override
    public PageResult<PolicemanDto> GetAllPolicemanPage(PolicemanReq req) {
        //查询传入的组织机构id是否存在
        PubOrg pubOrg = pubOrgService.PubOrgExist(req.getOrgId());
        //查询出组织机构下的所有警员信息
        LambdaQueryWrapper<Policeman> eq = new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(req.getOrgId()), Policeman::getOrgId, req.getOrgId())
                .like(ObjectUtil.isNotEmpty(req.getName()), Policeman::getName, req.getName())
                .like(ObjectUtil.isNotEmpty(req.getPoliceCode()), Policeman::getPoliceCode, req.getPoliceCode())
                .eq(ObjectUtil.isNotEmpty(req.getState()), Policeman::getState, req.getState());
        long size = list(eq).size();
        Page<Policeman> page = page(PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize()), eq);
        List<Policeman> records = page.getRecords();
        List<PolicemanDto> policeList = new ArrayList<>();
        //查询出警员信息下的指纹列表信息
        for (Policeman p : records) {
            PolicemanDto policemanDto = new PolicemanDto();
            BeanPlusUtil.copyProperties(p, policemanDto);
            List<PoliceFinger> list1 = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                    .eq(ObjectUtil.isNotNull(p.getId()), PoliceFinger::getPoliceId, p.getId()));
            List<PoliceFingerDto> policeFingerDtoList = new ArrayList<>();
            for (PoliceFinger policeFinger : list1) {
                PoliceFingerDto policeFingerDto = new PoliceFingerDto();
                BeanPlusUtil.copyProperties(policeFinger, policeFingerDto);
                policeFingerDtoList.add(policeFingerDto);
            }
            policemanDto.setFingersList(policeFingerDtoList);
            policemanDto.setOrgName(pubOrg.getOrgName());
            policeList.add(policemanDto);

        }
        Page<PolicemanDto> page1 = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        page1.setRecords(policeList);
        page1.setTotal(size);
        return PageResultFactory.createPageResult(page1);
    }


    //根据单警柜id查询警员信息(list)
    @Override
    public List<PolicemanDto> GetPoliceData(CabinetReq req) {
        //判断单警柜id是否正确
        Cabinet cabinet = cabinetService.CabinetExist(req.getId());
        PubOrg pubOrg = pubOrgService.getOne(new LambdaQueryWrapper<PubOrg>()
                        .eq(PubOrg::getOrgId, cabinet.getOrgId()));
        //通过箱门id查询出警员
        List<Policeman> policemenList = policemanMapper.SearchPoliceList(req.getId());
        List<PolicemanDto> list = new ArrayList<>();
        for (Policeman policeman : policemenList) {
            PolicemanDto policemanDto = new PolicemanDto();
            BeanPlusUtil.copyProperties(policeman, policemanDto);
            //指纹信息查找
            List<PoliceFinger> fingerList = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                    .eq(PoliceFinger::getPoliceId, policeman.getId()));
            List<PoliceFingerDto> fingerDtoList = new ArrayList<>();
            for (PoliceFinger policeFinger : fingerList) {
                PoliceFingerDto policeFingerDto = new PoliceFingerDto();
                BeanPlusUtil.copyProperties(policeFinger, policeFingerDto);
                fingerDtoList.add(policeFingerDto);
            }
            policemanDto.setFingersList(fingerDtoList);
            policemanDto.setCabinetNum(cabinet.getCabinetNum());
            policemanDto.setCabinetName(cabinet.getName());
            policemanDto.setOrgName(pubOrg.getOrgName());
            list.add(policemanDto);
        }
        return list;
    }


    //根据单警柜id查询警员信息(page)
    @Override
    public PageResult<PolicemanDto> GetPoliceDataPage(CabinetReq req) {
        //判断单警柜id是否正确
        Cabinet cabinet = cabinetService.CabinetExist(req.getId());
        PubOrg pubOrg = pubOrgService.getOne(new LambdaQueryWrapper<PubOrg>()
                .eq(PubOrg::getOrgId, cabinet.getOrgId()));
        //通过箱门id查询出警员
        Page<Policeman> page = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        long size = policemanMapper.SearchPoliceSize(req.getId());
        IPage<Policeman> policemenList = policemanMapper.SearchPolicePage(page,req.getId());
        List<PolicemanDto> list = new ArrayList<>();
        for (Policeman policeman : policemenList.getRecords()) {
            PolicemanDto policemanDto = new PolicemanDto();
            BeanPlusUtil.copyProperties(policeman, policemanDto);
            //指纹信息查找
            List<PoliceFinger> fingerList = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                    .eq(PoliceFinger::getPoliceId, policeman.getId()));
            List<PoliceFingerDto> fingerDtoList = new ArrayList<>();
            for (PoliceFinger policeFinger : fingerList) {
                PoliceFingerDto policeFingerDto = new PoliceFingerDto();
                BeanPlusUtil.copyProperties(policeFinger, policeFingerDto);
                fingerDtoList.add(policeFingerDto);
            }
            policemanDto.setFingersList(fingerDtoList);
            policemanDto.setCabinetNum(cabinet.getCabinetNum());
            policemanDto.setCabinetName(cabinet.getName());
            policemanDto.setOrgName(pubOrg.getOrgName());
            list.add(policemanDto);
        }
        Page<PolicemanDto> page1 = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        page1.setRecords(list);
        page1.setTotal(size);
        return PageResultFactory.createPageResult(page1);
    }

    //添加人脸信息
    @Override
    public boolean AddFaceInfo(UpdatePolicemanReq req) {
        Policeman police = PoliceExist(req.getId());
        police.setFaceInfo(req.getFaceInfo());
        police.setUpdateTime(DateTimeUtil.getCurrentDateTime());
        return updateById(police);
    }

    /**
     * @param
     * @return
     * @author lxh
     * @description 更新人脸信息\删除人脸信息
     * 删除时：传一个null的人脸信息即可
     * @since 2023/7/18 15:02
     **/
    @Override
    public boolean UpdateFaceInfo(UpdatePolicemanReq req) {
        Policeman police = PoliceExist(req.getId());
        police.setFaceInfo(req.getFaceInfo());
        police.setUpdateTime(DateTimeUtil.getCurrentDateTime());
        return updateById(police);

    }

    /**
     * @param
     * @return
     * @author lxh
     * @description 查询人脸信息
     * @since 2023/7/18 15:02
     **/
    @Override
    public Policeman SearchFaceInfo(PolicemanReq req) {
        Policeman policeman = PoliceExist(req.getId());
        return policeman;
    }


    /**
     * 警员绑定账号
     *
     * @param req
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean PoliceBindUser(PolicemanReq req) {
        //判断警员是否存在并且判断该警员是否已经绑定账号，如果绑定则抛出异常
//        Policeman policeman = PoliceExist(req.getId());
//        if (ObjectUtil.isNotNull(policeman.getUserId())){
//            throw new ServiceException(PolicemanExceptionEnum.POLICEUSER_IS_EXISTS);
//        }
        SysUser user = new SysUser();
        BeanPlusUtil.copyProperties(req, user);
        //判断数据库中是否已经存在账号，如果存在则抛出异常提示账号已存在
        //密码加密，bcrypt加密方式
        BcryptPasswordStoredEncrypt passwordStoredEncrypt = new BcryptPasswordStoredEncrypt();
        String encrypt = passwordStoredEncrypt.encrypt(req.getPassword());
        user.setPassword(encrypt);
        SysUser one = sysUserService.getOne(new LambdaQueryWrapper<SysUser>()
                .eq(SysUser::getAccount, user.getAccount())
                .eq(SysUser::getPassword, user.getPassword()));
        if (ObjectUtil.isNotNull(one)){
            throw new ServiceException(UserExceptionEnum.USER_IS_EXIST);
        }

        //存储到数据库
        //判断传入的sex是什么，需要做处理
        if (req.getSex().equals("男")) {
            user.setSex("M");
        } else {
            user.setSex("F");
        }
        user.setRealName(req.getName());
        user.setNickName(req.getName());
        sysUserService.save(user);
        UserOrgReq userOrgReq = new UserOrgReq();
        BeanPlusUtil.copyProperties(user,userOrgReq);
        userOrgReq.setOrgId(req.getOrgId());
        this.sysUserOrgService.add(userOrgReq);
        //通过警员id将查询到的警员信息进行修改设置
        Policeman policeman = new Policeman();
        policeman.setUserId(user.getUserId());
        policeman.setIsCreatedAccount(1);
        policeman.setPassword(encrypt);
        return updateById(policeman);
    }

    //人员绑定箱门
    @Override
    public boolean PoliceBindBox(PolicemanReq req) {
        Policeman policeman = PoliceExist(req.getId());
        BeanPlusUtil.copyProperties(req, policeman);
        //判断箱子是否存在
        cabinetBoxService.CabinetBoxExist(req.getCabinetBoxId());
        return updateById(policeman);
    }

    /**
     * 通过警员id判断警员是否存在，不存在则抛出异常
     */
    @Override
    public Policeman PoliceExist(String policeId) {
        Policeman policeman = getOne(new LambdaQueryWrapper<Policeman>()
                .eq(Policeman::getId,policeId));
        if (ObjectUtil.isNull(policeman)) {
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        return policeman;
    }

    //删除账号时同时对警员的信息进行更改操作
    @Override
    public boolean UpdatePoliceInfo(UserOrgReq req) {
        Policeman police = getOne(new LambdaQueryWrapper<Policeman>().eq(Policeman::getUserId,req.getUserId()));
        if (ObjectUtil.isNull(police)) {
            //如果查询不到，表示该账号不是警员的，直接删除就行，不需要做任何操作
            return true;
        }
        police.setIsCreatedAccount(0);
        police.setUserId(null);
        police.setPassword(null);
        return updateById(police);
    }

    //警员调岗
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean changePoliceOrg(PoliceReassignmentReq req) {
        //通过警员id查询出该警员的基本信息
        Policeman policeman = getById(req.getId());
        //如果警员绑定账号，将警员的账号进行删除
        if (ObjectUtil.isNotNull(policeman.getUserId())){
            policeman.setUserId(null);
            policeman.setPassword(null);
            sysUserService.removeById(policeman.getUserId());
        }
        //修改警员的个人信息(删除指纹信息，修改组织机构信息)
        List<PoliceFinger> list = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                .eq(PoliceFinger::getPoliceId, policeman.getId()));
        if (list.size()!=0){
            policeFingerService.removeBatchByIds(list);
        }
        policeman.setOrgId(req.getGoalOrgId());
        //如果绑定箱门，进行解绑
        if (ObjectUtil.isNotNull(policeman.getCabinetBoxId())){
            policeman.setCabinetBoxId(null);
        }
        //对装备进行相关操作
        //判断前端传递的装备列表是否为空，如果为空则表示该警员不拿走任何装备信息，将警员装备表中的数据删除即可
        if (req.getEqsReqList().size()==0){
            policeEquipmentService.remove(new LambdaQueryWrapper<PoliceEquipment>()
                    .eq(PoliceEquipment::getPoliceId, policeman.getId()));
        }else{
            //警员带走的装备
            
        }
        return updateById(policeman);
    }

    //获取警员下面的装备
    @Override
    public PageResult<PoliceEquipment> GetPoliceEqs(PolicemanReq req) {
        LambdaQueryWrapper<PoliceEquipment> eq = new LambdaQueryWrapper<PoliceEquipment>()
                .eq(PoliceEquipment::getPoliceId, req.getId());
        Page<PoliceEquipment> page = policeEquipmentService
                .page(PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize()), eq);
        return PageResultFactory.createPageResult(page);
    }


}
