package com.junmp.jyzb.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;

import com.junmp.jyzb.api.bean.dto.*;
import com.junmp.jyzb.api.bean.query.*;
import com.junmp.jyzb.api.bean.req.BatchEditingInvsReq;
import com.junmp.jyzb.api.bean.req.EquipmentBindReq;
import com.junmp.jyzb.api.bean.vo.InventoryVo;
import com.junmp.jyzb.api.exception.enums.CabinetBoxExceptionEnum;
import com.junmp.jyzb.api.exception.enums.CabinetExceptionEnum;
import com.junmp.jyzb.api.exception.enums.InventoryExceptionEnum;
import com.junmp.jyzb.entity.*;

import com.junmp.jyzb.service.*;
import com.junmp.v2.common.exception.base.ServiceException;
import com.junmp.v2.common.util.BeanPlusUtil;
import com.junmp.v2.common.util.HttpServletUtil;
import com.junmp.v2.db.api.factory.PageFactory;
import com.junmp.v2.db.api.factory.PageResultFactory;
import com.junmp.v2.db.api.page.PageResult;
import com.junmp.v2.office.api.OfficeExcelApi;
import com.junmp.v2.office.api.bean.ExcelExportParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import com.junmp.jyzb.mapper.InventoryMapper;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;


@Service
@Slf4j
public class InventoryServiceImpl extends ServiceImpl<InventoryMapper, Inventory> implements InventoryService {
    @Resource
    private InventoryMapper inventoryMapper;
    @Resource
    private EquipmentTypeService equipmentTypeService;
    @Resource
    private EquipmentSizeService equipmentSizeService;
    @Resource
    private WarehouseService warehouseService;
    @Resource
    private PubOrgService PubOrgService;

    @Resource
    private CabinetBoxService cabinetBoxService;

    @Resource
    private PubOrgService pubOrgService;

    @Resource
    private CabinetService cabinetService;

    @Resource
    private SupplierService supplierService;
    @Resource
    private ShelfService shelfService;
    @Resource
    private WarehouseAreaService warehouseAreaService;

    @Resource
    private LogSummaryService logSummaryService;
    @Resource
    private LogDetailService logDetailService;

    @Resource
    private OfficeExcelApi officeExcelApi;

    @Resource
    private OrderMainService orderMainService;

    @Resource
    private InventorySummaryService inventorySummaryService;


    //根据装备id查询出装备的详细信息
    @Override
    public InventoryDto getInventoryDetail(InventoryReq req) {
        Inventory inventory = InventoryExist(req.getId());
        InventoryDto bean = BeanUtil.toBean(inventory, InventoryDto.class);
        // 查询与库存表关联的其他数据
        String orgName= PubOrgService.getById(inventory.getOrgId()).getOrgName();
        String TypeName = equipmentTypeService.getById(inventory.getTypeId()).getName();
        String SizeName = equipmentSizeService.getById(inventory.getSizeId()).getName();
        // 创建并填充对象
        bean.setSizeName(SizeName);
        bean.setTypeName(TypeName);
        bean.setOrgName(orgName);
        return bean;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean setInventoryMsg(){
        //先查询inventory_summary表中的物资阈值字段是否为空，如果为空则默认他是默认值，如果不是默认值，将其存储，再重新统计后进行赋值
        List<InventorySummary> list = inventorySummaryService.list(new LambdaQueryWrapper<InventorySummary>()
                .isNotNull(InventorySummary::getThreshold)
                .eq(InventorySummary::getLocationType,0));

        //清空数据库中已存在的信息
        inventoryMapper.deleteAllMsg();
        //放入单警柜信息汇总
        boolean b = inventoryMapper.SetCabinetInventory();
        //放入仓库区域信息汇总
        boolean b1 = inventoryMapper.SetWarehouseInventory();
        boolean b2=true;
        if (list.size()>0){
            for (InventorySummary inventorySummary:list) {
                InventorySummary one = inventorySummaryService.getOne(new LambdaQueryWrapper<InventorySummary>()
                        .eq(InventorySummary::getLocationId, inventorySummary.getLocationId())
                        .eq(InventorySummary::getLocationType, 0)
                        .eq(InventorySummary::getSizeId, inventorySummary.getSizeId())
                        .eq(InventorySummary::getTypeId, inventorySummary.getTypeId())
                        .eq(InventorySummary::getOrgId, inventorySummary.getOrgId()));
                one.setThreshold(inventorySummary.getThreshold());
                b2 = inventorySummaryService.updateById(one);
            }
        }
        if (b && b1 && b2){
            return true;
        }else{
            return false;
        }
    }



    //设置仓库库存数量（总数，在库数，出库数，价格）
    @Override
    public boolean UpdateWarehouseInsNum(WarehouseReq req) {
        Warehouse warehouse = warehouseService.WarehoustExist(req.getId());
        boolean b = warehouseService.SetInventoryInfo(warehouse.getId());
        return b;
    }

    //设置单警柜库存数量（总数，在库数，出库数，价格）
    @Override
    public boolean UpdateCabinetInsNum(CabinetReq req) {
        Cabinet cabinet = cabinetService.CabinetExist(req.getId());
        boolean b = cabinetService.SetInventoryInfo(cabinet.getId());
        return b;
    }

    //装备绑定箱门
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean InventoryBind(EquipmentBindReq req) {
        //判断箱门是否存在
        CabinetBox cabinetBox = cabinetBoxService.CabinetBoxExist(req.getCabinetBoxId());
        if (Objects.isNull(cabinetBox)){
            throw new ServiceException(CabinetBoxExceptionEnum.CABINETBOX_ISNOT_EXISTS);
        }
        List<String> equipmentList = req.getEquipmentList();
        if (equipmentList.size()==0){
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        List<Inventory> inventoryList=new ArrayList<>();
        for (String id: equipmentList) {
            Inventory inventory = getById(id);
            //locationId:根据单警柜的不同存放的内容不同，单警柜存放箱门id，仓库存放仓库id
            inventory.setLocationId(req.getCabinetBoxId());
            inventoryList.add(inventory);
        }
        return updateBatchById(inventoryList);
    }

    //根据Epc查询装备信息
    @Override
    public List<InventoryDto> GetInvInfoByEpc(InventoryReq req) {
        //判断传入的是否为空
        List<String> epcList = req.getEpcList();
        if (epcList.size()==0){
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        List<InventoryDto> inventoryDtoList=new ArrayList<>();
        List<Inventory> inventoryList = list(new LambdaQueryWrapper<Inventory>().in(Inventory::getEpc, epcList));
        for (Inventory inventory:inventoryList) {
            InventoryDto inventoryDto = new InventoryDto();
            BeanPlusUtil.copyProperties(inventory, inventoryDto);
            //设置型号名称，类别名称，供应商名称
            String sizeName = equipmentSizeService.getById(inventory.getSizeId()).getName();
            String typeName = equipmentTypeService.getById(inventory.getTypeId()).getName();
            if (inventory.getSupplierId() != null) {
                String supplierName = supplierService.getById(inventory.getSupplierId()).getName();
                inventoryDto.setSupplierName(supplierName);
            }
            inventoryDto.setSizeName(sizeName);
            inventoryDto.setTypeName(typeName);
            inventoryDtoList.add(inventoryDto);
        }
        return inventoryDtoList;
    }

    //根据组织机构/单警柜id/仓库id查询每件装备细信的详息（返回结果是List）
    @Override
    public List<InventoryDto> EqsBySearchType(InventoryReq req) {
        //判断传入的queryType参数是否是正确的
        if (!req.getQueryType().equals("org") && !req.getQueryType().equals("warehouse") && !req.getQueryType().equals("cabinet")){
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        //判断如果传递了类别，判断传递的值是否为空或者是否不存在
        if (ObjectUtil.isNotNull(req.getOrgId())){
            pubOrgService.PubOrgExist(req.getOrgId());
        } else if (ObjectUtil.isNotNull(req.getWarehouseId())){
            warehouseService.WarehoustExist(req.getWarehouseId());
        } else if (ObjectUtil.isNotNull(req.getCabinetId())){
            cabinetService.CabinetExist(req.getCabinetId());
        }else {
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        return inventoryMapper.EqsBySearchType(req);
    }

    //根据组织机构/单警柜id/仓库id查询每件装备细信的详息（返回结果是page）
    public PageResult<InventoryDto> EqsByTypePage(InventoryReq req) {
        //判断传入的queryType参数是否是正确的
        if (!req.getQueryType().equals("org") && !req.getQueryType().equals("warehouse") && !req.getQueryType().equals("cabinet")){
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        //判断如果传递了类别，判断传递的值是否为空或者是否不存在
        if (ObjectUtil.isNotNull(req.getOrgId())){
            pubOrgService.PubOrgExist(req.getOrgId());
        } else if (ObjectUtil.isNotNull(req.getWarehouseId())){
            warehouseService.WarehoustExist(req.getWarehouseId());
        } else if (ObjectUtil.isNotNull(req.getCabinetId())){
            cabinetService.CabinetExist(req.getCabinetId());
        }else {
            throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
        }
        Page<InventoryDto> page = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        IPage<InventoryDto> pages= inventoryMapper.EqsByTypePage(page,req.getQueryType(),req.getOrgId(),req.getWarehouseId(),req.getCabinetId());
        page.setRecords(pages.getRecords());
        return PageResultFactory.createPageResult(page);
    }

    //查询箱门（cabinetBox下）所有装备简要信息列表（传递cabinetBoxId）
    @Override
    public List<EqsBriefDto> searchEqsByBoxId(CabinetBoxReq req) {
        List<Inventory> list = list(new LambdaQueryWrapper<Inventory>()
                .eq(Inventory::getLocationId, req.getId())
                .orderByDesc(Inventory::getTypeId)
                .orderByDesc(Inventory::getSizeId));
        if (list.size()==0){
            throw new ServiceException(CabinetBoxExceptionEnum.CABINETBOX_ISNOT_EXISTS);
        }
        List<EqsBriefDto> eqsBriefDtoList=new ArrayList<>();
        Iterator<Inventory> iterator = list.iterator();
        while (iterator.hasNext()){
            Inventory next = iterator.next();
            EqsBriefDto eqsBriefDto = new EqsBriefDto();
            BeanPlusUtil.copyProperties(next,eqsBriefDto);
            Map<String, String> map = setNames( eqsBriefDto.getTypeId(),eqsBriefDto.getSizeId(), eqsBriefDto.getSupplierId());
            //设置typeName
            eqsBriefDto.setTypeName(map.get("typeName"));
            //设置sizeName
            eqsBriefDto.setSizeName(map.get("sizeName"));
            //如果有供应商id的前提下，设置supplierName
            if (ObjectUtil.isNotNull(eqsBriefDto.getSupplierId())){
                eqsBriefDto.setSupplierName(map.get("supplierName"));
            }
            eqsBriefDtoList.add(eqsBriefDto);
        }
        return eqsBriefDtoList;
    }

    //查询箱门（cabinetBox下）所有装备简要信息列表page（传递cabinetBoxId）
    @Override
    public PageResult<EqsBriefDto> searchEqsByBoxIdPage(CabinetBoxReq req) {
//        List<EqsBriefDto> eqsBriefDtoList = searchEqsByBoxId(req);
        LambdaQueryWrapper<Inventory> eq = new LambdaQueryWrapper<Inventory>()
                .eq(Inventory::getLocationId, req.getId())
                .orderByDesc(Inventory::getTypeId)
                .orderByDesc(Inventory::getSizeId);
        List<Inventory> list = list(eq);
        long size = list.size();
        Page<Inventory> page = page(PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize()), eq);
        List<Inventory> records = page.getRecords();
        List<EqsBriefDto> eqsBriefDtoList=records.parallelStream()
                .map(inventory -> {
            EqsBriefDto eqsBriefDto = new EqsBriefDto();
            BeanPlusUtil.copyProperties(inventory,eqsBriefDto);
            Map<String, String> map = setNames(eqsBriefDto.getTypeId(), eqsBriefDto.getSizeId(), eqsBriefDto.getSupplierId());
            //设置typeName
            eqsBriefDto.setTypeName(map.get("typeName"));
            //设置sizeName
            eqsBriefDto.setSizeName(map.get("sizeName"));
            //如果有供应商id的前提下，设置supplierName
            if (ObjectUtil.isNotNull(eqsBriefDto.getSupplierId())){
                eqsBriefDto.setSupplierName(map.get("supplierName"));
            }
            return eqsBriefDto;
        }).collect(Collectors.toList());
        Page<EqsBriefDto> page1 = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        page1.setRecords(eqsBriefDtoList);
        page1.setTotal(size);
        return PageResultFactory.createPageResult(page1);
    }

    //根据条件查询仓库下的所有装备信息
    @Override
    public PageResult<InventoryDto> GetDetailByTerms(InventoryReq req) {
        //判断组织机构是否存在
        pubOrgService.PubOrgExist(req.getOrgId());
        //根据条件查询仓库下的所有装备
        LambdaQueryWrapper<Inventory> eq = new LambdaQueryWrapper<Inventory>()
                .eq(Inventory::getOrgId, req.getOrgId())
                .eq(Inventory::getLocationType, 0)
                .eq(ObjectUtil.isNotNull(req.getLocationId()) && !req.getLocationId().trim().isEmpty(), Inventory::getLocationId, req.getLocationId())
                .eq(ObjectUtil.isNotNull(req.getLocationState()) && !req.getLocationState().trim().isEmpty(), Inventory::getLocationState, req.getLocationState())
                .eq(ObjectUtil.isNotNull(req.getBussinessState()) && !req.getBussinessState().trim().isEmpty(), Inventory::getBussinessState, req.getBussinessState())
                .eq(ObjectUtil.isNotNull(req.getTermState()), Inventory::getTermState, req.getTermState())
                .eq(ObjectUtil.isNotNull(req.getSizeId()) && !req.getSizeId().trim().isEmpty(),Inventory::getSizeId,req.getSizeId())
                .eq(ObjectUtil.isNotNull(req.getTypeId()) && !req.getTypeId().trim().isEmpty(),Inventory::getTypeId,req.getTypeId())
                .eq(ObjectUtil.isNotNull(req.getPrice()),Inventory::getPrice,req.getPrice());

        long size = list(eq).size();
        Page<Inventory> page = page(PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize()), eq);
        List<Inventory> records = page.getRecords();
        List<InventoryDto> collect = records.parallelStream()
                .map(inventory -> {
                    InventoryDto inventoryDto = new InventoryDto();
                    BeanPlusUtil.copyProperties(inventory, inventoryDto);
                    Map<String, String> map = setNames(inventory.getTypeId(), inventory.getSizeId(), inventory.getSupplierId());
                    //设置typeName
                    inventoryDto.setTypeName(map.get("typeName"));
                    //设置sizeName
                    inventoryDto.setSizeName(map.get("sizeName"));
                    //如果有供应商id的前提下，设置supplierName
                    if (ObjectUtil.isNotNull(inventory.getSupplierId())) {
                        inventoryDto.setSupplierName(map.get("supplierName"));
                    }
                    String orgName = pubOrgService.getOne(new LambdaQueryWrapper<PubOrg>().eq(PubOrg::getOrgId, inventory.getOrgId())).getOrgName();
                    inventoryDto.setOrgName(orgName);
                    //查询图片并返回
                    String photo = equipmentTypeService.getById(inventory.getTypeId()).getPhoto();
                    inventoryDto.setPhoto(photo);
                    return inventoryDto;
                }).collect(Collectors.toList());
        Page<InventoryDto> page1 = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        page1.setRecords(collect);
        page1.setTotal(size);
        return PageResultFactory.createPageResult(page1);
    }

    //批量进行调整
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean BatchEditingInvsInfo(BatchEditingInvsReq req) {
        //查询inventory表中数据，对响应数据进行修改
        //判断是对整一类的装备进行修改还是说对个别几个装备进行批量修改
        if (req.getInventoryIdList().size()!=0){
            List<Inventory> list = list(new LambdaQueryWrapper<Inventory>().in(Inventory::getId, req.getInventoryIdList()));
            if (list.size()==0){
                throw new ServiceException(CabinetExceptionEnum.PARAMETER_ERROR);
            }else {
                //批量修改个别装备
                List<Inventory> collect = list.stream().map(inventory -> {
                    BeanPlusUtil.copyProperties(req,inventory );
                    if (ObjectUtil.isNotNull(req.getShelfColumns()) && !req.getShelfColumns().toString().trim().isEmpty()
                            && ObjectUtil.isNotNull(req.getShelfRanges()) && !req.getShelfRanges().toString().trim().isEmpty()
                            && ObjectUtil.isNotNull(req.getShelfRows()) && !req.getShelfRows().toString().trim().isEmpty()) {
                        inventory.setShelfLocation(req.getShelfRanges() + "/" + req.getShelfColumns() + "/" + req.getShelfRows());
                    }
                    return inventory;
                }).collect(Collectors.toList());
                updateBatchById(collect);
                //修改汇总信息
                return setInventorySummary(req);
            }
        }else {
            List<Inventory> list = list(createWrapper(req));
            //直接将一类的数据进行修改
            List<Inventory> collect = list.stream().map(inventory -> {
                BeanPlusUtil.copyProperties(req, inventory);
                if (ObjectUtil.isNotNull(req.getShelfColumns()) && !req.getShelfColumns().toString().trim().isEmpty()
                        && ObjectUtil.isNotNull(req.getShelfRanges()) && !req.getShelfRanges().toString().trim().isEmpty()
                        && ObjectUtil.isNotNull(req.getShelfRows()) && !req.getShelfRows().toString().trim().isEmpty()) {
                    inventory.setShelfLocation(req.getShelfRanges() + "/" + req.getShelfColumns() + "/" + req.getShelfRows());
                }
                return inventory;
            }).collect(Collectors.toList());
            updateBatchById(collect);
            return setInventorySummary(req);
        }
    }

    //修改库存信息汇总
    public boolean setInventorySummary(BatchEditingInvsReq req){
        //先将其删除再全部重新添加计算
        List<InventorySummary> list = inventorySummaryService.list(new LambdaQueryWrapper<InventorySummary>()
                .eq(ObjectUtil.isNotNull(req.getOrgId()),InventorySummary::getOrgId,req.getOrgId())
                .eq(ObjectUtil.isNotNull(req.getLocationId()),InventorySummary::getLocationId,req.getLocationId())
                .eq(ObjectUtil.isNotNull(req.getSizeId()),InventorySummary::getSizeId,req.getSizeId())
                .eq(ObjectUtil.isNotNull(req.getTypeId()),InventorySummary::getTypeId,req.getTypeId()));
        boolean b = inventorySummaryService.removeBatchByIds(list);
        boolean a = inventoryMapper.insertInventorySummary(req);
        return (a && b);
    }

    //根据货架id获取装备列表汇总信息(list)
    @Override
    public List<InventorySummary> ShelfInventoryList(ShelfReq req) {
        //判断货架是否存在
        Shelf shelf = shelfService.ShelfExist(req.getShelfId());
        warehouseService.WarehoustExist(shelf.getWarehouseId());
        //通过货架id获取装备列表汇总
        return inventoryMapper.getSumByShelf(req.getShelfId());
    }

    //根据货架id获取装备列表汇总信息(page)
    @Override
    public PageResult<InventorySummary> ShelfInventoryPage(ShelfReq req) {
        //判断货架是否存在
        Shelf shelf = shelfService.ShelfExist(req.getShelfId());
        warehouseService.WarehoustExist(shelf.getWarehouseId());
        Page<InventorySummary> page = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        IPage<InventorySummary> pages = inventoryMapper.ShelfInventoryPage(page,req.getShelfId());
        page.setRecords(pages.getRecords());
        return PageResultFactory.createPageResult(page);
    }

    //通过货架id查询装备简要信息（list）
    @Override
    public List<EqsBriefDto> ShelfInventoryDetail(InventoryReq req) {
        //判断货架是否存在
        Shelf shelf = shelfService.ShelfExist(req.getShelfId());
        warehouseService.WarehoustExist(shelf.getWarehouseId());
        List<Inventory> list = list(new LambdaQueryWrapper<Inventory>()
                .eq(Inventory::getShelfId, req.getShelfId())
                .orderByDesc(Inventory::getTypeId)
                .orderByDesc(Inventory::getSizeId));
        List<EqsBriefDto> eqsBriefDtoList=list.parallelStream()
                .map(inventory -> {
                    EqsBriefDto eqsBriefDto = new EqsBriefDto();
                    BeanPlusUtil.copyProperties(inventory,eqsBriefDto);
                    Map<String, String> map = setNames( eqsBriefDto.getTypeId(),eqsBriefDto.getSizeId(), eqsBriefDto.getSupplierId());
                    //设置typeName
                    eqsBriefDto.setTypeName(map.get("typeName"));
                    //设置sizeName
                    eqsBriefDto.setSizeName(map.get("sizeName"));
                    //如果有供应商id的前提下，设置supplierName
                    if (ObjectUtil.isNotNull(eqsBriefDto.getSupplierId())){
                        eqsBriefDto.setSupplierName(map.get("supplierName"));
                    }
                    return eqsBriefDto;
                }).collect(Collectors.toList());
        return eqsBriefDtoList;
    }

    //通过货架id查询装备简要信息（page）
    @Override
    public PageResult<EqsBriefDto> ShelfInventoryPage(InventoryReq req) {
        Shelf shelf = shelfService.ShelfExist(req.getShelfId());
        warehouseService.WarehoustExist(shelf.getWarehouseId());
        LambdaQueryWrapper<Inventory> eq = new LambdaQueryWrapper<Inventory>()
                .eq(Inventory::getShelfId, req.getShelfId())
                .orderByDesc(Inventory::getTypeId)
                .orderByDesc(Inventory::getSizeId);
        List<Inventory> list = list(eq);
        long size = list.size();
        Page<Inventory> page = page(PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize()), eq);
        List<Inventory> records = page.getRecords();
        List<EqsBriefDto> eqsBriefDtoList=records.parallelStream()
            .map(inventory -> {
                EqsBriefDto eqsBriefDto = new EqsBriefDto();
                BeanPlusUtil.copyProperties(inventory,eqsBriefDto);
                Map<String, String> map = setNames( eqsBriefDto.getTypeId(),eqsBriefDto.getSizeId(), eqsBriefDto.getSupplierId());
                //设置typeName
                eqsBriefDto.setTypeName(map.get("typeName"));
                //设置sizeName
                eqsBriefDto.setSizeName(map.get("sizeName"));
                //如果有供应商id的前提下，设置supplierName
                if (ObjectUtil.isNotNull(eqsBriefDto.getSupplierId())){
                    eqsBriefDto.setSupplierName(map.get("supplierName"));
                }
                return eqsBriefDto;
            }).collect(Collectors.toList());
        Page<EqsBriefDto> page1 = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        page1.setRecords(eqsBriefDtoList);
        page1.setTotal(size);
        return PageResultFactory.createPageResult(page1);
    }


    /**
     * 通过区域id查询装备汇总信息
     * @param req
     * @return
     */
    @Override
    public List<InventorySummary> AreaInventoryList(WarehouseAreaReq req) {
        //判断区域是否存在
        WarehouseArea warehouseArea = warehouseAreaService.WarehouseAreaExist(req.getId());
        warehouseService.WarehoustExist(warehouseArea.getWarehouseId());
        //通过区域id获取装备列表汇总
        return inventoryMapper.getSumByArea(req.getId());
    }

    @Override
    public PageResult<InventorySummary> AreaInventoryPage(WarehouseAreaReq req) {
        //判断区域是否存在
        WarehouseArea warehouseArea = warehouseAreaService.WarehouseAreaExist(req.getId());
        warehouseService.WarehoustExist(warehouseArea.getWarehouseId());
        Page<InventorySummary> page = PageFactory.getDefaultPage(req.getPageNo(), req.getPageSize());
        IPage<InventorySummary> pages= inventoryMapper.AreaInventoryPage(page,req.getId());
        page.setRecords(pages.getRecords());
        return PageResultFactory.createPageResult(page);
    }


    //通过单警柜箱门id获取箱门下的装备汇总信息(list)
    @Override
    public List<InventorySummary> EqsByCabinetBoxId(CabinetBoxReq req) {
        cabinetBoxService.CabinetBoxExist(req.getId());
        return inventoryMapper.getSumByCabinetBox(req.getId());
    }

    //通过单警柜箱门id获取箱门下的装备汇总信息(page)
    @Override
    public PageResult<InventorySummary> EqsByCabinetBoxIdPage(CabinetBoxReq req) {
        cabinetBoxService.CabinetBoxExist(req.getId());
        Map<String, Long> map = orderMainService.getPage(req.getPageNo(), req.getPageSize());
        Long pageNo = map.get("pageNo");
        Long pageSize = map.get("pageSize");
        Page<InventorySummary> page=new Page<>(pageNo,pageSize);
        IPage<InventorySummary> pages = inventoryMapper.EqsByCabinetBoxIdPage(page,req.getId());
        page.setRecords(pages.getRecords());
        return PageResultFactory.createPageResult(page);
    }

    //通过组织机构id将该组织下的装备进行导出
    @Override
    public void ExportInventoryExcel(InventoryReq req){
        //判断组织机构是否存在
        pubOrgService.PubOrgExist(req.getOrgId());
        //查询该组织机构下的所有装备信息
        List<InventoryVo> list = inventoryMapper.selectListByOrg(req.getOrgId());
        HttpServletResponse response = HttpServletUtil.getResponse();
        ExcelExportParam param = new ExcelExportParam();
        param.setDataList(list);
        param.setClazz(Inventory.class);
        param.setResponse(response);
        param.setFileName("装备列表.xls");
        //对数据进行导出
        officeExcelApi.easyExportDownload(param);
    }

    //获取装备的出入库信息
    @Override
    public List<InOutRecordsDto> getInOutRecords(InventoryReq req) {
        //通过装备id获取出入库记录(判断id是否存在)
        InventoryExist(req.getId());
        List<LogDetail> list = logDetailService.list(new LambdaQueryWrapper<LogDetail>()
                .eq(LogDetail::getInventoryId, req.getId()));
        List<InOutRecordsDto> inOutRecordsDtoList=new ArrayList<>();
        for (LogDetail logDetail: list) {
            //获取
            LogSummary logSummary = logSummaryService.getById(logDetail.getSummaryId());
            InOutRecordsDto inOutRecordsDto = new InOutRecordsDto();
            BeanPlusUtil.copyProperties(logSummary,inOutRecordsDto);
            inOutRecordsDtoList.add(inOutRecordsDto);
        }
        return inOutRecordsDtoList;
    }

    //查询装备数量报表
    @Override
    public List<EqsSumDto> GetListEquipment(InventoryReq req) {
        //通过获取传递的查询条件进行查询，得到最终的装备数量报表
        return inventoryMapper.GetListEquipment(req);
    }



    //通过id判断装备是否存在
    @Override
    public Inventory InventoryExist(String id) {
        Inventory invMap=this.getById(id);
        if (ObjectUtil.isNull(invMap)){
            throw new ServiceException(InventoryExceptionEnum.INVENTORY_NOT_EXIST);
        }
        return invMap;
    }

    //通用方法，用于设置对象的typeName属性，sizeName属性和supplierName属性
    public Map<String,String> setNames (String typeId,String sizeId,String supplierId){
        Map<String,String> map=new HashMap<>();
        String sizeName = equipmentSizeService.getOne(new LambdaQueryWrapper<EquipmentSize>()
                .eq(EquipmentSize::getId, sizeId)).getName();

        String typeName = equipmentTypeService.getOne(new LambdaQueryWrapper<EquipmentType>()
                .eq(EquipmentType::getId, typeId)).getName();
        //如果有供应商id的前提下，设置supplierName
        if (ObjectUtil.isNotNull(supplierId)){
            String supplierName = supplierService.getOne(new LambdaQueryWrapper<Supplier>()
                    .eq(Supplier::getId, supplierId)).getName();
            map.put("supplierName",supplierName);
        }
        map.put("sizeName",sizeName);
        map.put("typeName",typeName);
        return map;
    }

    //创建条件查询
    public LambdaQueryWrapper<Inventory> createWrapper(BatchEditingInvsReq req){
        LambdaQueryWrapper<Inventory> wrapper = new LambdaQueryWrapper<>();
        if (ObjectUtil.isEmpty(req)){
            return wrapper;
        }
        wrapper.eq(ObjectUtil.isNotNull(req.getOrgId()),Inventory::getOrgId,req.getOrgId());
        wrapper.eq(ObjectUtil.isNotNull(req.getLocationId()),Inventory::getLocationId,req.getLocationId());
        wrapper.eq(ObjectUtil.isNotNull(req.getSizeId()),Inventory::getSizeId,req.getSizeId());
        wrapper.eq(ObjectUtil.isNotNull(req.getTypeId()),Inventory::getTypeId,req.getTypeId());
        return wrapper;
    }

}