package com.junmp.jyzb.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.junmp.jyzb.api.bean.dto.PoliceDto;
import com.junmp.jyzb.api.bean.dto.PoliceFingerDto;
import com.junmp.jyzb.api.bean.dto.PolicemanDto;
import com.junmp.jyzb.api.bean.query.PolicemanReq;
import com.junmp.jyzb.api.bean.req.UpdatePolicemanReq;
import com.junmp.jyzb.entity.*;
import com.junmp.jyzb.exception.enums.CabinetBoxExceptionEnum;
import com.junmp.jyzb.exception.enums.CabinetExceptionEnum;
import com.junmp.jyzb.exception.enums.PolicemanExceptionEnum;
import com.junmp.jyzb.exception.enums.PubOrgExceptionEnum;
import com.junmp.jyzb.mapper.*;
import com.junmp.jyzb.service.*;
import com.junmp.jyzb.utils.*;
import com.junmp.v2.auth.password.BcryptPasswordStoredEncrypt;
import com.junmp.v2.common.exception.base.ServiceException;
import com.junmp.v2.common.util.BeanPlusUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;


import static com.junmp.jyzb.utils.CheckBlank.checkNotBlank;

@Service
@Slf4j
public class PolicemanServiceImpl extends ServiceImpl<PolicemanMapper, Policeman> implements PolicemanService {

    @Resource
    private RedisUtils redisUtils;

    @Resource
    private PolicemanMapper policemanMapper;

    @Resource
    private PoliceFingerMapper policeFingerMapper;

    @Resource
    private CabinetBoxMapper cabinetBoxMapper;

    @Resource
    private CabinetMapper cabinetMapper;

    @Resource
    private PubOrgMapper pubOrgMapper;

    @Resource
    private PubOrgService pubOrgService;

    @Resource
    private CabinetService cabinetService;

    @Resource
    private PolicemanService policemanService;

    @Resource
    private CabinetBoxService cabinetBoxService;

    @Resource
    private PoliceFingerService policeFingerService;

    @Resource
    private SysuserService sysUserService;

    private static final String REDIS_POLICE = "Policeman_";
    private static final String REDIS_ORG = "Organization_";


    /**
     * 添加警员信息
     * @param req
     * @return
     */
    @Override
    public String AddPoliceman(UpdatePolicemanReq req) {
        //查询传入的组织机构id是否存在
        PubOrg puborg = pubOrgService.getOne(new LambdaQueryWrapper<PubOrg>()
                .eq(ObjectUtil.isNotEmpty(req.getOrgId()),PubOrg::getOrgId,req.getOrgId()));
        if (ObjectUtil.isEmpty(puborg)) {
            //抛出组织机构不存在异常
            throw new ServiceException(PubOrgExceptionEnum.PUBORG_NOT_EXIST);
        }
        //查询是否有重复的policeCode
        List<Policeman> list = policemanService.list(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(req.getPoliceCode()), Policeman::getPoliceCode, req.getPoliceCode()));
        if (list.size()>0){
            throw new ServiceException(PolicemanExceptionEnum.POLICECODE_IS_EXISTS);
        }
        Policeman policeman = new Policeman();
        BeanPlusUtil.copyProperties(req,policeman);
        //添加警员绑定用户信息
        policemanService.save(policeman);
        return policeman.getId();
    }

    // 更新缓存的方法，新增警员信息时调用
    private void updatePoliceIdCache(String orgId, String newPoliceId) {
        // 构建 Redis 缓存键
        String redisKeyFalse = "getAllPoliceId_false_" + orgId;
        String redisKeyTrue = "getAllPoliceId_true_" + orgId;

        // 从 Redis 中获取旧的缓存数据
        List<String> cachedDataFalse = redisUtils.findCachedData(redisKeyFalse);
        List<String> cachedDataTrue = redisUtils.findCachedData(redisKeyTrue);

        //如果缓存中没有数据，则直接查询数据库并更新Redis缓存
        //如果缓存中有数据，就更新数据
        if (cachedDataFalse == null && cachedDataTrue == null) {
            //getAllPoliceId(Map.of("IncludeLowerLevel", false, "orgId", orgId));
            return;
        }else if(cachedDataFalse!=null){
            cachedDataFalse.add(newPoliceId);
            redisUtils.set(redisKeyFalse, cachedDataFalse);
        }else if(cachedDataTrue!=null){
            cachedDataTrue.add(newPoliceId);
            redisUtils.set(redisKeyTrue, cachedDataTrue);
        }
    }

    //删除缓存的方法
    private void deletePoliceIdCache(String orgId){
        //构建 Redis 缓存键
        String redisKey = "getAllPoliceId_false_" + orgId;
        redisUtils.delete(redisKey);

        //构建 Redis 缓存键
        redisKey = "getAllPoliceId_true_" + orgId;
        redisUtils.delete(redisKey);
    }


    /**
     *
     * @param req
     * @return
     */
    @Override
    public PolicemanDto GetOnePolice(PolicemanReq req) {
        Policeman one = getOne(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotNull(req.getId()), Policeman::getId, req.getId()));
        if (ObjectUtil.isNull(one)){
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        PolicemanDto policemanDto = new PolicemanDto();
        BeanPlusUtil.copyProperties(one,policemanDto);
        //将警员的指纹信息查询出来
        List<PoliceFinger> list1 = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                .eq(ObjectUtil.isNotNull(req.getId()), PoliceFinger::getPoliceId, req.getId()));
        List<PoliceFingerDto> fingerDtoList=new ArrayList<>();
        for (PoliceFinger policeFinger:list1) {
            PoliceFingerDto policeFingerDto = new PoliceFingerDto();
            BeanPlusUtil.copyProperties(policeFinger,policeFingerDto);
            fingerDtoList.add(policeFingerDto);
        }
        policemanDto.setFingersList(fingerDtoList);
        return policemanDto;
    }


    /**
     * 删除警员信息
     * @param req
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean DeletePolice(UpdatePolicemanReq req) {
        //通过警员id查询警员信息
        Policeman policeman = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotNull(req.getId()), Policeman::getId, req.getId()));
        //查询不到警员信息则抛异常
        if (ObjectUtil.isNull(policeman)){
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        //如果查询出来的警员信息中没有绑定箱门而参数传递了箱门id，则抛出异常警员绑定箱门错误
        //如果传递的箱门id参数与查询出来的箱门id不一致，则抛出异常警员绑定的箱门与实际绑定不符合
        if ((ObjectUtil.isNull(policeman.getCabinetBoxId()) && ObjectUtil.isNotNull(req.getCabinetBoxId()))){
            throw new ServiceException(CabinetBoxExceptionEnum.CABINETBOX_ISNOT_EXISTS);
        }
        //通过警员信息中的userId对账号进行删除
        sysUserService.remove(new LambdaQueryWrapper<User>()
                .eq(User::getUserId,policeman.getUserId()));
         return policemanService.removeById(policeman);

    }


    /**
     * 修改警员信息
     * @param req
     * @return
     */
    @Override
    public boolean UpdatePolice(UpdatePolicemanReq req) {
        Policeman policeman = new Policeman();
        BeanPlusUtil.copyProperties(req,policeman);
        //查询传入的组织机构id是否存在
        PubOrg puborg = pubOrgService.getOne(new LambdaQueryWrapper<PubOrg>()
                .eq(ObjectUtil.isNotEmpty(req.getOrgId()),PubOrg::getOrgId,req.getOrgId()));
        if (ObjectUtil.isEmpty(puborg)) {
            //抛出组织机构不存在异常
            throw new ServiceException(PubOrgExceptionEnum.PUBORG_NOT_EXIST);
        }
        //判断传入的修改的警员编号与本身编号一致，如果一致则无所谓，如果不一致则需要判断修改的警员编号是否已经存在
        Policeman one = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotNull(req.getId()), Policeman::getId, req.getId()));
        if (!req.getPoliceCode().equals(one.getPoliceCode())){
            Policeman one1 = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
                    .eq(ObjectUtil.isNotNull(req.getPoliceCode()), Policeman::getPoliceCode, req.getPoliceCode()));
            if (ObjectUtil.isNotNull(one1)){
                throw new ServiceException(PolicemanExceptionEnum.POLICECODE_IS_EXISTS);
            }
        }

        //查询单警柜箱号是否存在,箱门不存在抛异常
        CabinetBox cabinetBox = cabinetBoxService.getOne(new LambdaQueryWrapper<CabinetBox>()
                .eq(CabinetBox::getId, req.getCabinetBoxId()));
        if (ObjectUtil.isNull(cabinetBox)){
            throw new ServiceException(CabinetBoxExceptionEnum.CABINETBOX_ISNOT_EXISTS);
        }
        policeman.setUpdateTime(DateTimeUtil.getCurrentDateTime());
        return policemanService.updateById(policeman);
    }

    @Override
    public List<String> getAllPoliceId(Map<String, Object> msg) {

        //构建 Redis 缓存键
        String redisKey = "getAllPoliceId_"+msg.get("IncludeLowerLevel")+'_'+msg.get("orgId");
        // 从 Redis 中获取数据
        List<String> cachedData = redisUtils.findCachedData(redisKey);
        if (cachedData!=null){
            return cachedData;
        }
        List<String> policeId = new ArrayList<>();
        if (msg.get("IncludeLowerLevel").equals("false")){
            policeId = policemanMapper.getAllPoliceId(msg.get("orgId").toString());
        }else if(msg.get("IncludeLowerLevel").equals("true")){
            //查询某组织机构的本级及下级
            List<String> orgIds = getSubIds(msg.get("orgId").toString()); //调用本级及下级的所有组织机构的id
            for (String orgId:orgIds){
                //String redisKey = REDIS_ORG + orgId;
                List<String> allPoliceId = policemanMapper.getAllPoliceId(orgId);
                policeId.addAll(allPoliceId);
            }
        }else {
            //填入的IncludeLowerLevel不正确
            return null;
        }
        //将查询结果存入 Redis 中
        redisUtils.set(redisKey, policeId);
        return policeId;
    }

    //查询某组织机构的所有子机构
    public List<String> getSubIds(String id) {
        List<String> subIds = new ArrayList<>();
        subIds.add(id);//先往List中放入父级id
        List<String> subMenus = policemanMapper.getChildren(id);//查询该id的子类id
        if (subMenus != null) {
            for (String subMenu : subMenus) {
                subIds.addAll(getSubIds(subMenu));
            }
        }
        return subIds;
    }


    @Override
    public List<String> getAllPoliceIdByCabinet(Map<String, Object> cabinetId) {
        //根据单警柜id找到所有的箱号id
        List<String> msg = policemanMapper.getAllBoxIdByCabinet(cabinetId.get("cabinetId").toString());
        //遍历所有的箱号id,根据箱号id查询所有的警员信息
        List<String> boxId = new ArrayList<>();
        for (String id:msg){
            policemanMapper.getAllPoliceIdByBoxId(id);
        }
        return boxId;
    }



    /**
     * 修改警员状态
     * @param req
     * @return
     */
    @Override
    public boolean ChangePoliceState(UpdatePolicemanReq req) {
        //通过警员id查询出警员信息
        Policeman one = getOne(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotNull(req.getId()), Policeman::getId, req.getId()));
        if (ObjectUtil.isNull(one)){
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        one.setState(req.getState());
        return policemanService.updateById(one);

    }

    public List<Map<String, Object>> getAllPolicemanMsg(Map<String, Object> msg,List<String> allPoliceId) {
        List<Map<String, Object>> allPoliceman = new ArrayList<>();
        // 构建 Redis 缓存键
        String redisKey = REDIS_ORG + msg.get("IncludeLowerLevel").toString() +'_'+ msg.get("orgId").toString();
        // 从 Redis 中获取数据
        List<Map<String, Object>> cachedData = redisUtils.findCachedData(redisKey);
        if (cachedData != null) {
            //将Long类型的数据转化为Date
            for (Map<String,Object> one:cachedData){
                Long dataMsg = (Long) one.get("updateTime");
                one.put("updateTime", redisUtils.getDate(dataMsg));
                allPoliceman.add(one);
            }
            return cachedData;
        }
        for (String policeId : allPoliceId) {
            Map<String, Object> policeMsg = policemanMapper.getOnePolice(policeId);
            allPoliceman.add(policeMsg);
        }

        //按照updateTime降序排序
        DateTimeUtil.timeSort(allPoliceman);

        //将查询结果存入 Redis 中
        redisUtils.set(redisKey, allPoliceman);
        return allPoliceman;
    }

    @Override
    public ResponseResult getAllPoliceman(Map<String, Object> msg) {
        try {
            checkNotBlank(msg.get("orgId"), "orgId不能为空");
        } catch (IllegalArgumentException e) {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,e.getMessage());
        }
        List<Map<String, Object>> allPoliceman = new ArrayList<>();

        //如果组织机构是浙江省公安厅
        if (msg.get("orgId").toString().equals("1369509498032808905") && msg.get("IncludeLowerLevel").toString().equals("true")){
            allPoliceman = policemanMapper.getAllPolice();
            return new ResponseResult(HttpStatus.SUCCESS,ReturnMsg.PASS,allPoliceman);
        }

        //获取组织机构Id
        List<String> allOrgId =getAllOrgId(msg);
        if (allOrgId.isEmpty()){
            return new ResponseResult(HttpStatus.ERROR,ReturnMsg.ERROR,"未查询到相关信息");
        }

        //根据获取的组织机构查询警员信息
        allPoliceman = getAllPolicemanByOrg(msg,allOrgId);

        return new ResponseResult(HttpStatus.SUCCESS,"操作成功",allPoliceman);

    }

    /**
     * 通过组织机构id查询警员列表
     * @param orgId
     */
    @Override
    public PoliceDto GetAllPolicemanList(Long orgId) {
        //查询传入的组织机构id是否存在
        PubOrg puborg = pubOrgService.getOne(new LambdaQueryWrapper<PubOrg>()
                .eq(ObjectUtil.isNotEmpty(orgId),PubOrg::getOrgId,orgId));
        if (ObjectUtil.isNull(puborg)) {
            //抛出组织机构不存在异常
            throw new ServiceException(PubOrgExceptionEnum.PUBORG_NOT_EXIST);
        }
        //查询出组织机构下的所有警员信息
        List<Policeman> list = list(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(orgId), Policeman::getOrgId, orgId)
                .eq(Policeman::getState,1));
        List<PolicemanDto> policeList=new ArrayList<>();
        //查询出警员信息下的指纹列表信息
        for (Policeman p:list) {
            PolicemanDto policemanDto = new PolicemanDto();
            BeanPlusUtil.copyProperties(p,policemanDto);
            List<PoliceFinger> list1 = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                    .eq(ObjectUtil.isNotNull(p.getId()), PoliceFinger::getPoliceId, p.getId()));
            List<PoliceFingerDto> policeFingerDtoList=new ArrayList<>();
            for (PoliceFinger policeFinger:list1) {
                PoliceFingerDto policeFingerDto = new PoliceFingerDto();
                BeanPlusUtil.copyProperties(policeFinger,policeFingerDto);
                policeFingerDtoList.add(policeFingerDto);
            }
            policemanDto.setFingersList(policeFingerDtoList);
            policeList.add(policemanDto);

        }
        PoliceDto policeDto = new PoliceDto();
        policeDto.setPolicemanlist(policeList);
        //设置组织机构名
        policeDto.setOrgName(puborg.getOrgName());
        policeDto.setOrgId(orgId);
        return policeDto;
    }

    private List<Map<String, Object>> getAllPolicemanByOrg(Map<String, Object> msg, List<String> allOrgId) {
        List<Map<String, Object>> allPoliceman = new ArrayList<>();
        allPoliceman = policemanMapper.getAllPolicemanByOrgList(allOrgId);
        return allPoliceman;
    }

    private List<String> getAllOrgId(Map<String, Object> msg) {
        List<String> allOrg = new ArrayList<>();
        if (msg.get("IncludeLowerLevel").equals("false")){
            allOrg.add((String) msg.get("orgId"));
        }else if(msg.get("IncludeLowerLevel").equals("true")){
            //查询某组织机构的本级及下级
            allOrg = pubOrgService.getLowerOrg(msg.get("orgId").toString());
        }
        return allOrg;
    }


    //根据单警柜id查询警员信息
    @Override
    public PoliceDto GetPoliceData(String id) {
        PoliceDto policeDto = new PoliceDto();
        //判断单警柜id是否正确
        Cabinet cabinet = cabinetService.getOne(new LambdaQueryWrapper<Cabinet>()
                .eq(Cabinet::getId, id));
        if (ObjectUtil.isNull(cabinet)){
            throw new ServiceException(CabinetExceptionEnum.CABINET_ISNOT_EXISTS);
        }
        //通过箱门id查询出警员
        List<Policeman> policemenList = policemanMapper.SearchPoliceList(id);
        List<PolicemanDto> list = new ArrayList<>();
        for (Policeman policeman:policemenList) {
            PolicemanDto policemanDto = new PolicemanDto();
            BeanPlusUtil.copyProperties(policeman,policemanDto);
            //指纹信息查找
            List<PoliceFinger> fingerList = policeFingerService.list(new LambdaQueryWrapper<PoliceFinger>()
                    .eq(PoliceFinger::getPoliceId, policeman.getId()));
            List<PoliceFingerDto> fingerDtoList=new ArrayList<>();
            for (PoliceFinger policeFinger:fingerList) {
                PoliceFingerDto policeFingerDto = new PoliceFingerDto();
                BeanPlusUtil.copyProperties(policeFinger,policeFingerDto);
                fingerDtoList.add(policeFingerDto);
            }
            policemanDto.setFingersList(fingerDtoList);
            list.add(policemanDto);
        }
        policeDto.setPolicemanlist(list);
        policeDto.setCabinetNum(cabinet.getCabinetNum());
        policeDto.setCabinetName(cabinet.getName());
        PubOrg pubOrg = pubOrgService.getOne(new LambdaQueryWrapper<PubOrg>()
                .eq(PubOrg::getOrgId, cabinet.getOrgId()));
        policeDto.setOrgId(pubOrg.getOrgId());
        policeDto.setOrgName(pubOrg.getOrgName());
        return policeDto;
    }

    //添加人脸信息
    @Override
    public boolean AddFaceInfo(PolicemanReq req) {
        //面部信息为null或者为空
        if (ObjectUtil.isNull(req.getFaceInfo()) || ObjectUtil.isEmpty(req.getFaceInfo())){
            return false;
        }
        Policeman police = getOne(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(req.getId()),Policeman::getId,req.getId()));
        if (ObjectUtil.isNull(police)){
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        police.setFaceInfo(req.getFaceInfo());
        police.setUpdateTime(DateTimeUtil.getCurrentDateTime());
        return policemanService.updateById(police);
    }

    /**
         * @author lxh
         * @description 更新人脸信息\删除人脸信息
         * 删除时：传一个null的人脸信息即可
         * @since 2023/7/18 15:02
         * @param
         * @return
         **/
    @Override
    public boolean UpdateFaceInfo(UpdatePolicemanReq req) {
        //警员id不能为空
        if (ObjectUtil.isEmpty(req.getId()) || ObjectUtil.isNull(req.getId())){
            return false;
        }
        Policeman police = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(req.getId()), Policeman::getId, req.getId()));
        police.setFaceInfo(req.getFaceInfo());
        police.setUpdateTime(DateTimeUtil.getCurrentDateTime());
        return policemanService.updateById(police);

    }

    /**
         * @author lxh
         * @description 查询人脸信息
         * @since 2023/7/18 15:02
         * @param
         * @return
         **/
    @Override
    public Policeman SearchFaceInfo(PolicemanReq req) {
        //警员id不能为空
        if (ObjectUtil.isEmpty(req.getId()) || ObjectUtil.isNull(req.getId())){
            return null;
        }
        Policeman policeman = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
                .eq(ObjectUtil.isNotEmpty(req.getId()), Policeman::getId, req.getId()));
        if (ObjectUtil.isNull(policeman)){
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        return policeman;
    }


    /**
     * 警员绑定账号
     * @param req
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean PoliceBindUser(PolicemanReq req) {
        User user = new User();
        BeanPlusUtil.copyProperties(req,user);
        //密码加密，依赖注入错误
        BcryptPasswordStoredEncrypt passwordStoredEncrypt = new BcryptPasswordStoredEncrypt();
        String encrypt = passwordStoredEncrypt.encrypt(req.getPassword());
        user.setPassword(encrypt);
        //存储到数据库
        //判断传入的sex是什么，需要做处理
        if (req.getSex().equals("男")){
            user.setSex("M");
        }else{
            user.setSex("F");
        }
        user.setRealName(req.getName());
        user.setNickName(req.getName());
        sysUserService.save(user);

        //通过警员id将查询到的警员信息进行修改设置
        Policeman policeman = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
                .eq(Policeman::getId, req.getId()));
        if (ObjectUtil.isNull(policeman)){
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        policeman.setUserId(user.getUserId());
        policeman.setIsCreatedAccount(true);
        policeman.setPassword(encrypt);
        return policemanService.updateById(policeman);
    }

    //人员绑定箱门
    @Override
    public boolean PoliceBindBox(PolicemanReq req) {
        Policeman policeman = new Policeman();
        BeanPlusUtil.copyProperties(req,policeman);
        Policeman one = policemanService.getOne(new LambdaQueryWrapper<Policeman>()
                .eq(Policeman::getId, req.getId()));
        if (ObjectUtil.isNull(one)){
            throw new ServiceException(PolicemanExceptionEnum.POLICEMAN_NOT_EXIST);
        }
        //判断箱子是否存在
        CabinetBox cabinetBox = cabinetBoxService.getOne(new LambdaQueryWrapper<CabinetBox>()
                .eq(CabinetBox::getId, req.getCabinetBoxId()));
        if (ObjectUtil.isNull(cabinetBox)){
            throw new ServiceException(CabinetBoxExceptionEnum.CABINETBOX_ISNOT_EXISTS);
        }
        return policemanService.updateById(policeman);
    }


}
