package com.junmp.jyzb.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;

import com.junmp.jyzb.api.bean.dto.InventoryDto;
import com.junmp.jyzb.api.bean.req.InventoryReq;
import com.junmp.jyzb.entity.Inventory;

import com.junmp.jyzb.service.*;
import com.junmp.jyzb.utils.HttpStatus;
import com.junmp.jyzb.utils.ResponseResult;
import com.junmp.jyzb.utils.ReturnData;
import com.junmp.jyzb.utils.ReturnMsg;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import javax.annotation.Resource;
import com.junmp.jyzb.mapper.InventoryMapper;

import java.math.BigDecimal;
import java.util.*;

import static com.junmp.jyzb.utils.CheckBlank.checkNotBlank;

@Service
@Slf4j
public class InventoryServiceImpl extends ServiceImpl<InventoryMapper, Inventory> implements InventoryService {

    @Resource
    private InventoryMapper inventoryMapper;
    @Resource
    private EquipmentTypeService equipmentTypeService;
    @Resource
    private EquipmentSizeService equipmentSizeService;
    @Resource
    private WarehouseService warehouseService;
    @Resource
    private PubOrgService PubOrgService;

    @Resource
    private CabinetService cabinetService;
    @Override
    public InventoryDto getInventoryDetail(InventoryReq req) {

        Inventory invMap=this.getById(req.getID());
        if (invMap == null) {
            return null;
        }
        InventoryDto bean = BeanUtil.toBean(invMap, InventoryDto.class);
        // 查询与库存表关联的其他数据
        String orgName= PubOrgService.getById(invMap.getOrgId()).getOrgName();
        String TypeName = equipmentTypeService.getById(invMap.getTypeId()).getName();
        String SizeName = equipmentSizeService.getById(invMap.getSizeId()).getName();
        // 创建并填充对象
        bean.setSizeName(SizeName);
        bean.setTypeName(TypeName);
        bean.setOrgName(orgName);

        // 填充其他关联信息的字段和对象引用
        return bean;

    }

    @Override
    public ResponseResult getEquipmentInfo(Map<String, Object> inMsg) {
        //参数校验
        try {
            checkNotBlank(inMsg.get("queryType"), "queryType不能为空");
            checkNotBlank(inMsg.get("typeId"), "typeId不能为空");
        } catch (IllegalArgumentException e) {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,e.getMessage());
        }

        //解析json中的信息
        String queryType = inMsg.get("queryType").toString();
        String idMsg = inMsg.get("typeId").toString();

        //根据传入的queryType来辨别要查询什么信�
        if (queryType.equals("org") || queryType.equals("warehouse") || queryType.equals("cabinet")){
            Map<String,Object> returnMsg = getLocationEquipmentInfo(queryType,idMsg);
            return new ResponseResult(HttpStatus.SUCCESS, ReturnMsg.PASS,returnMsg);
        }else {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,"输入的queryType信息有误");
        }

    }

    private Map<String, Object> getLocationEquipmentInfo(String queryType, String idMsg) {
        List<Map<String, Object>> allMsg = inventoryMapper.getLocationInventory(queryType,idMsg);

        Map<String, Object> resultMap = new HashMap<>();
        for (Map<String, Object> msg : allMsg) {
            String locationId = msg.get("locationId").toString();
            String locationName = msg.get("locationName").toString();
            String typeId = msg.get("typeId").toString();
            String typeName = msg.get("typeName").toString();
            String sizeId = msg.get("sizeId").toString();
            String sizeName = msg.get("sizeName").toString();
            int number = (int) msg.get("number");
            BigDecimal priceBigDecimal = (BigDecimal) msg.get("price");
            float price = priceBigDecimal.floatValue();

            //检查resultMap中是否存在orgId
            if (resultMap.containsKey(locationId)) {
                // 获取现有组织级数�
                Map<String, Object> orgData = (Map<String, Object>) resultMap.get(locationId);
                List<Map<String, Object>> orgTypeList = (List<Map<String, Object>>) orgData.get("typeList");

                // 检查组织类型列表中是否存在typeId
                boolean typeExist = false;
                for (Map<String, Object> typeData : orgTypeList) {
                    if (typeId.equals(typeData.get("typeId"))) {
                        // 获取现有的类型级别数�
                        List<Map<String, Object>> sizeList = (List<Map<String, Object>>) typeData.get("sizeList");
                        sizeList.add(createSizeData(sizeId,sizeName, number,price)); //向sizeList中记录number的信�
                        int typeNumber = (int) typeData.get("typeNum");
                        float typePrice = (float) typeData.get("typeTotalPrice");
                        typeData.put("typeNum", typeNumber + number); //计算装备数量
                        typeData.put("typeTotalPrice", typePrice + price);
                        typeExist = true;
                        break;
                    }
                }

                // 如果 orgTypeList 中不包含 typeId，则创建新的类型级别数据
                if (!typeExist) {
                    Map<String, Object> typeData = new HashMap<>();
                    typeData.put("typeId", typeId);
                    typeData.put("typeName", typeName);
                    typeData.put("typeNum", number);
                    typeData.put("typeTotalPrice", price);
                    List<Map<String, Object>> sizeList = new ArrayList<>();
                    sizeList.add(createSizeData(sizeId,sizeName, number,price));
                    typeData.put("sizeList", sizeList);
                    orgTypeList.add(typeData);
                }

                // 更新组织级别数据
                int orgNumber = (int) orgData.get("locationNum");
                float orgPrice = (float) orgData.get("totalPrice");
                orgData.put("locationNum", orgNumber + number);
                orgData.put("totalPrice", orgPrice + price);
            } else {
                // 创建新的组织级别数据
                Map<String, Object> orgData = new HashMap<>();
                orgData.put("locationType", queryType);
                orgData.put("locationId", locationId);
                orgData.put("locationName", locationName);
                orgData.put("locationNum", number);
                orgData.put("totalPrice", price);
                List<Map<String, Object>> orgTypeList = new ArrayList<>();
                Map<String, Object> typeData = new HashMap<>();
                typeData.put("typeId", typeId);
                typeData.put("typeName", typeName);
                typeData.put("typeNum", number);
                typeData.put("typeTotalPrice", price);
                List<Map<String, Object>> sizeList = new ArrayList<>();
                sizeList.add(createSizeData(sizeId,sizeName, number,price));
                typeData.put("sizeList", sizeList);
                orgTypeList.add(typeData);
                orgData.put("typeList", orgTypeList);
                resultMap.put(locationId, orgData);
            }
        }
        Map<String,Object> returnMsg = (Map<String, Object>) resultMap.get(idMsg);
        return returnMsg;
    }

    private static Map<String, Object> createSizeData(String sizeId,String sizeName, int number,float price) {
        Map<String, Object> sizeData = new HashMap<>();
        sizeData.put("sizeId", sizeId);
        sizeData.put("sizeName", sizeName);
        sizeData.put("sizeNum", number);
        sizeData.put("sizeTotalPrice", price);
        return sizeData;
    }

    @Override
    public ResponseResult setInventoryMsg(){
        inventoryMapper.deleteAllMsg(); //清空数据库中已存在的信息

        inventoryMapper.setOrgInventory(); //放入组织机构信息汇�

        inventoryMapper.setCabinetInventory(); //放入组织机构信息汇�

        inventoryMapper.setWarehouseInventory(); //放入组织机构信息汇�
        return new ResponseResult(HttpStatus.SUCCESS, ReturnMsg.PASS);
    }

    @Override
    public ResponseResult updateInventoryNum(Map<String, Object> msg) {
        //参数校验
        try {
            checkNotBlank(msg.get("locationType"), "locationType不能为空");
        } catch (IllegalArgumentException e) {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,e.getMessage());
        }
        String locationType = msg.get("locationType").toString();
        if (!locationType.equals("warehouse") && !locationType.equals("cabinet")){
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,"输入的locationType值有�");
        }else if(locationType.equals("warehouse")){
            warehouseService.setWarehouseInventory(msg);
        }else if(locationType.equals("cabinet")){
            cabinetService.setCabinetInventory(msg);
        }
        return new ResponseResult(HttpStatus.SUCCESS, ReturnMsg.PASS);
    }

    @Override
    public ResponseResult getInventoryList(Map<String, Object> msg) {
        //参数校验
        try {
            checkNotBlank(msg.get("queryType"), "queryType不能为空");
            checkNotBlank(msg.get("typeCode"), "typeCode不能为空");
            checkNotBlank(msg.get("typeId"), "typeId不能为空");
            checkNotBlank(msg.get("sizeId"), "sizeId不能为空");
        } catch (IllegalArgumentException e) {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,e.getMessage());
        }

        //解析json中的信息
        String queryType = msg.get("queryType").toString();
        String typeCode = msg.get("typeCode").toString();
        String typeId = msg.get("typeId").toString();
        String sizeId = msg.get("sizeId").toString();

        //根据传入的queryType来辨别要查询什么信�
        if (queryType.equals("org") || queryType.equals("warehouse") || queryType.equals("cabinet")){
            List<Map<String,Object>> returnMsg = inventoryMapper.getInventoryList(queryType,typeCode,typeId,sizeId);
            return new ResponseResult(HttpStatus.SUCCESS, ReturnMsg.PASS,returnMsg);
        }else {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,"输入的queryType信息有误");
        }
    }

    @Override
    public ResponseResult getInventoryByOrgId(Map<String, Object> msg) {
        //参数校验
        try {
            checkNotBlank(msg.get("orgId"), "orgId不能为空");
        } catch (IllegalArgumentException e) {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,e.getMessage());
        }
        String orgId = msg.get("orgId").toString();
        List<Map<String, Object>> returnMsg = inventoryMapper.getInventoryByOrgId(orgId);
        return new ResponseResult(HttpStatus.SUCCESS, ReturnMsg.PASS,returnMsg);
    }

    @Override
    public ResponseResult getInventoryDevelop() {
        return null;
    }

    @Override
    public ResponseResult InvertoryBindBox(Map<String, Object> msg) {
        //参数校验
        try {
            checkNotBlank(msg.get("inventoryId"), "inventoryId不能为空");
            checkNotBlank(msg.get("locationId"), "locationId不能为空");
        } catch (IllegalArgumentException e) {
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR,e.getMessage());
        }
        //将装备和箱门进行绑定（简单来 说就是更新数据库中的单条记录信息）
        Inventory inventory = inventoryMapper.selectOne(new LambdaQueryWrapper<Inventory>()
                .eq(Inventory::getID, msg.get("inventoryId"))
                .eq(Inventory::getLocationId, msg.get("locationId")));
        if (Objects.isNull(inventory)){
            return new ResponseResult(HttpStatus.ERROR, ReturnMsg.ERROR, ReturnData.NO_DATA);
        }
        inventory.setLocationId((String) msg.get("locationId"));
        inventoryMapper.updateById(inventory);
        return new ResponseResult(HttpStatus.SUCCESS, ReturnMsg.PASS,"绑定成功");
    }


    /**
     * @Author: Zhaojw
     * @Description: 生成lambda表达式
     * @DateTime: 2023/7/20 13:08
     * @Params:
     * @Return
     */
    private LambdaQueryWrapper<Inventory> createWrapper(InventoryReq req) {
        LambdaQueryWrapper<Inventory> wrapper = new LambdaQueryWrapper<>();
        if (ObjectUtil.isEmpty(req)) {
            return wrapper;
        }
        //根据库存ID查询
        wrapper.eq(ObjectUtil.isNotEmpty(req.getID()), Inventory::getID, req.getID());
        //根据epc模糊查询
        wrapper.like(ObjectUtil.isNotEmpty(req.getEpc()), Inventory::getEpc, req.getEpc());
        //根据位置
        wrapper.eq(ObjectUtil.isNotEmpty(req.getLocationId()), Inventory::getLocationId, req.getLocationId());
        wrapper.orderByDesc(Inventory::getUpdateTime);
        return wrapper;
    }
}