﻿using Common.Global;
using RabbitMQ.Client;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Common.Utility.RabbitMQ
{
    /// <summary>
    /// RabbitMQ工具类
    /// </summary>
    public class RabbitMQContext
    {
        /// <summary>
        /// 私有化构造函数
        /// 用于单例模式
        /// </summary>
        private RabbitMQContext() { }

        /// <summary>
        /// Lazy对象
        /// </summary>
        private static readonly Lazy<IConnection> LazyConnection = new Lazy<IConnection>(() =>
        {
            ConnectionFactory factory = null;
            IConnection connection = null;

            #region 初始工厂类

            if (AdminGlobalContext.RabbitMQConfig.HostName.Contains(","))
            {
                //创建连接对象工厂
                factory = new ConnectionFactory()
                {
                    UserName = AdminGlobalContext.RabbitMQConfig.UserName,
                    Password = AdminGlobalContext.RabbitMQConfig.Password,
                    AutomaticRecoveryEnabled = true,//如果connection挂掉是否重新连接
                    TopologyRecoveryEnabled = true//连接恢复后，连接的交换机，队列等是否一同恢复
                };
                //创建连接对象
                connection = factory.CreateConnection(AdminGlobalContext.RabbitMQConfig.HostName.Split(','));
            }
            else
            {
                //创建连接对象工厂
                factory = new ConnectionFactory()
                {
                    UserName = AdminGlobalContext.RabbitMQConfig.UserName,
                    Password = AdminGlobalContext.RabbitMQConfig.Password,
                    HostName = AdminGlobalContext.RabbitMQConfig.HostName,
                    Port = AdminGlobalContext.RabbitMQConfig.Port,
                    RequestedHeartbeat = TimeSpan.FromSeconds(30), //心跳包
                    AutomaticRecoveryEnabled = true, //自动重连
                    TopologyRecoveryEnabled = true, //拓普重连
                    NetworkRecoveryInterval = TimeSpan.FromSeconds(10)
                };
                //创建连接对象
                connection = factory.CreateConnection();
            }

            #endregion


            return connection;
        });

        /// <summary>
        /// 单例对象
        /// </summary>
        public static IConnection ConnectionInstance { get { return LazyConnection.Value; } }

        /// <summary>
        /// 是否已创建
        /// </summary>
        public static bool IsConnectionInstanceCreated { get { return LazyConnection.IsValueCreated; } }

        /// <summary>
        /// 发送消息
        /// </summary>
        /// <param name="message">消息体</param>
        public static void SendMessage(string message)
        {
            IConnection connection = ConnectionInstance;
            //定义通道
            var channel = connection.CreateModel();
            //如果只有生产者，则使用以下注释内容，作为单例使用
            //定义交换机
            //channel.ExchangeDeclare(AdminGlobalContext.RabbitMQConfig.ExchangeName, ExchangeType.Fanout, true, false);
            ////定义队列
            //channel.QueueDeclare(AdminGlobalContext.RabbitMQConfig.QueueName, false, false, false, null);
            ////将队列绑定到交换机
            //channel.QueueBind(AdminGlobalContext.RabbitMQConfig.QueueName, AdminGlobalContext.RabbitMQConfig.ExchangeName, "", null);
            //发布消息
            channel.BasicPublish(AdminGlobalContext.RabbitMQConfig.ExchangeName, "", null, Encoding.Default.GetBytes(message));
        }

        /// <summary>
        /// 异步发送消息
        /// </summary>
        /// <param name="message">消息体</param>
        /// <returns></returns>
        public static async Task SendMessageAsynce(string message)
        {
            await Task.Run(() => SendMessage(message));
        }
    }
}
