﻿using APIs.Common;
using APIs.Dto;
using APIs.Req;
using Autofac.Core;
using AutoMapper;
using Common;
using Common.Global;
using Common.Utility.Model;
using Common.Utility.RabbitMQ;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Filters;
using Microsoft.AspNetCore.Mvc.ModelBinding;
using Microsoft.CodeAnalysis;
using Microsoft.IdentityModel.Tokens;
using Models.SqlModel;
using Models.Table;
using Models.ToolsModel;
using Newtonsoft.Json;
using NuGet.Protocol;
using Services;
using Services.Interface;
using SqlSugar;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Diagnostics.Metrics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Linq;
using System.Linq.Expressions;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Security.Cryptography.Xml;
using System.Text;
using static Common.HttpHelper;
using static Dm.net.buffer.ByteArrayBuffer;
using static Microsoft.AspNetCore.Razor.Language.TagHelperMetadata;
using static System.Net.Mime.MediaTypeNames;
using static JmpDehumidifierLib.Dehumidifier;
using JmpDehumidifierLib;
using NLog.Fluent;
using System.Security.Policy;

namespace APIs.Controllers
{
    [Route("api/[controller]/[action]")]
    [ApiController]
    public class BaseInfoController : ControllerBase
    {
        public IMapper Mapper { get; }

        private readonly IPoliceService _policeService;
        private readonly IEquipmentTypeService _equipmentTypeService;
        private readonly IEquipmentSizeService _equipmentSizeService;
        private readonly ICarService _carService;
        private readonly IInventoryService _inventoryService;
        private readonly IInvService _invService;
        private readonly IUsersService _usersService;
        private readonly ILogService _logService;
        private readonly IDevHistoryService _devHistoryService;
        private readonly IDevService _devService;
        private readonly IWarehouseService _warehouseService;
        private readonly IOrderService _orderService;
        private readonly ISupplierService _supplierService;
        private readonly IThisInfoService _thisInfoService;
        private readonly IPrintService _printService;
        private readonly IPoliceFingerService _policeFingerService;
        private readonly IPoliceInfoService _policeInfoService;
        private readonly IBussinessInventoryService _bussinessInventoryService;
        private readonly IBussinessInventoryDetailService _bussinessInventoryDetailService;
        private readonly IChannelService _channelService;
        private readonly IViewWarrantyWarnService _viewWarrantyWarnService;

        private readonly DehumidifierHelper _dehumidifierHelper;

        public BaseInfoController(IMapper mapper, DehumidifierHelper dehumidifierHelper, IPoliceService policeService, IEquipmentTypeService equipmentTypeService,
            IEquipmentSizeService equipmentSizeService,ICarService carService, IInventoryService inventoryService,
            IInvService invService, IUsersService usersService, ILogService logService, IDevHistoryService devHistoryService,
            IDevService devService, IWarehouseService warehouseService, IOrderService orderService, ISupplierService supplierService,
            IThisInfoService thisInfoService, IPrintService printService, IPoliceFingerService policeFingerService, IPoliceInfoService policeInfoService,
            IBussinessInventoryService bussinessInventoryService, IBussinessInventoryDetailService bussinessInventoryDetailService, IChannelService channelService,
            IViewWarrantyWarnService viewWarrantyWarnService)
        {
            Mapper = mapper;
            _dehumidifierHelper = dehumidifierHelper;
            _carService = carService;
            _equipmentSizeService = equipmentSizeService;
            _equipmentTypeService = equipmentTypeService;
            _policeService = policeService;
            _inventoryService = inventoryService;
            _invService = invService;
            _usersService = usersService;
            _logService = logService;
            _devHistoryService = devHistoryService;
            _devService = devService;
            _warehouseService = warehouseService;
            _orderService = orderService;
            _supplierService = supplierService;
            _thisInfoService = thisInfoService;
            _printService = printService;
            _policeFingerService = policeFingerService;
            _policeInfoService = policeInfoService;
            _bussinessInventoryService = bussinessInventoryService;
            _bussinessInventoryDetailService = bussinessInventoryDetailService;
            _channelService = channelService;
            _viewWarrantyWarnService = viewWarrantyWarnService;
        }

        /// <summary>
        /// 设定除湿机
        /// </summary>
        /// <param name="req"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> SetDehumidifier([FromBody] SetDehumidifierReq req)
        {
            try
            {
                if (string.IsNullOrEmpty(req.devId) ||
                    req.setState == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "参数不正确",
                    };
                }

                Dev myDev = await _devService.QueryOne(s => s.id.Equals(req.devId));
                if (myDev != null)
                {
                    if (!string.IsNullOrEmpty(req.setHumid))
                    {
                        myDev.setSd = req.setHumid;
                    }
                    if (req.setState != null)
                    {
                        myDev.setState = req.setState;
                    }
                    myDev.updateTime = DateTime.Now;
                }
                else
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无设备信息",
                    };
                }

                _dehumidifierHelper.SetDehumidifier(myDev);
                await _devService.Update(myDev);

                return new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                };
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 手持机获取盘点单据
        /// </summary>
        /// <param name="req"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> GetInventoryOrder([FromBody] GetInventoryOrderReq req)
        {
            try
            {
                #region
                if (string.IsNullOrEmpty(req.warehouseId))
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "参数不正确",
                    };
                }
                if(await _orderService.QueryOne(s => s.orderState != 1 && s.bussinessType.Equals("purchase")) != null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "存在未记账结算的采购单据",
                    };
                }

                if (await _orderService.QueryOne(s => s.orderState == 2) != null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "存在未关闭的单据",
                    };
                }
                #endregion

                var BIOrderMain = await _bussinessInventoryService.QueryOneOrderDesc(s => s.warehouseId.Equals(req.warehouseId) && s.inventoryState.Equals("waitting"), c => c.createTime);
                BussinessInventory? rs = await _bussinessInventoryService.GenerateInventoryOrder(BIOrderMain);

                if(rs == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "子单生成失败",
                    };
                }

                #region 上报平台单据已开始
                string param = JsonConvert.SerializeObject(new 
                {
                    orderId = rs.id,
                    detailList = rs.DetailList
                });

                var json =  HttpHelper.HttpPost(AdminGlobalContext.jyzbConfig.Url + "/BussinessInventory/Start", param);
                if (string.IsNullOrEmpty(json))
                {
                    await _bussinessInventoryService.OrderRollBack(rs);
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无法连接至平台",
                    };
                }

                var httpRes = JsonConvert.DeserializeObject<res<string>>(json);
                if (httpRes == null || httpRes.code != "99200")
                {
                    await _bussinessInventoryService.OrderRollBack(rs);
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "上报平台失败",
                    };
                }
                #endregion


                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = JsonConvert.SerializeObject(rs),
                };

                return src;

            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 手持机上传单据操作结果
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> HandUploadResult([FromBody] HandResultReq req)
        {
            try
            {
                if (await _bussinessInventoryService.QueryOne(s => s.inventoryState.Equals("running")) != null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "正在盘点 禁止出入库",
                    };
                }

                if (req == null || req.order == null || req.logList == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "参数错误",
                    };
                }

                var rs = await _logService.HandUploadResult(Convert.ToInt64(AdminGlobalContext.jyzbConfig.OrgId), req);


                return rs ? new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                }: new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = "数据更新失败",
                };
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 手持机获取单据
        /// </summary>
        /// <param name="req"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> HandGetOrder([FromBody] HandGetOrderReq req)
        {
            try
            {
                if (string.IsNullOrEmpty(req.warehouseId))
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "参数不正确",
                    };
                }

                var orders = await _orderService.QueryHandOrder(req.warehouseId);

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = orders,
                };

                return src;

            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 手持机获取装备类别号型信息
        /// </summary>
        /// <param name="req"></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<ApiResult> HandEquBaseInfo()
        {
            try
            {
                var types = await _equipmentTypeService.Query();
                var sizes = await _equipmentSizeService.Query();

                var rs = sizes.Select(s => new { s.id, s.typeId, s.name, s.code, typeName = types.FirstOrDefault(x => s.typeId.Equals(x.id))?.name}).ToList();

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = rs,
                };

                return src;

            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 更新警员人脸图片地址
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> UpdateHKPicture([FromBody] UpdateHKPictureReq req)
        {
            try
            {
                if (req == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = "无效参数",
                    };
                }else if (string.IsNullOrEmpty(req.policeId))
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = "警员ID为空",
                    };
                }

                var policeinfo = await _policeInfoService.QueryOne(s => s.id.Equals(req.policeId));

                if (policeinfo != null)
                {
                    using (HttpClient client = new HttpClient())
                    {
                        var postContent = new MultipartFormDataContent();
                        string boundary = string.Format("--{0}", DateTime.Now.Ticks.ToString("x"));
                        postContent.Headers.Add("ContentType", $"multipart/form-data, boundary={boundary}");
                        postContent.Add(new ByteArrayContent(req.picture), "picture", Guid.NewGuid().ToString());
                        postContent.Add(new StringContent(req.policeId), "policeId");
                        using (HttpResponseMessage response = client.PostAsync(AdminGlobalContext.jyzbConfig.Url + "/api/uploadPolicePhoto", postContent).Result)
                        {
                            var str = response.Content.ReadAsStringAsync().Result;
                            var res = JsonConvert.DeserializeObject<HttpHelper.res<dynamic>>(str);

                            if (res != null && res.code.Equals("99200"))
                            {
                                policeinfo.photo = res.data.photo;
                                var rs = await _policeInfoService.Update(policeinfo);

                                return rs ? new ApiResult
                                {
                                    code = ResultCode.OPERATE_SUCCESS.Code,
                                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                                } : new ApiResult
                                {
                                    code = ResultCode.OPERATE_FAILED.Code,
                                    msg = "更新地址失败",
                                };
                            }
                            else
                            {
                                return new ApiResult
                                {
                                    code = ResultCode.OPERATE_FAILED.Code,
                                    msg = "上传平台失败",
                                };
                            }
                        }
                    }
                }
                else
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无匹配警员信息",
                    };
                }

            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 通道获取配置
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> GetWarehouseConfig([FromBody] GetWarehouseConfigReq req)
        {
            try
            {
                if (req == null || (string.IsNullOrEmpty(req.deviceId) && string.IsNullOrEmpty(req.warehouseId)))
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "参数错误",
                    };
                }

                Channel? config = null;
                if (!string.IsNullOrEmpty(req.deviceId))
                {
                    config = await _channelService.QueryOne(s => s.id.Equals(req.deviceId));
                }
                else
                {
                    config = await _channelService.QueryOne(s => s.warehouseId.Equals(req.warehouseId));
                }

                if (config != null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = ResultCode.OPERATE_SUCCESS.Msg,
                        data = config.config,
                    };
                }
                else
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无通道配置",
                    };
                }
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 人脸机通过人员标识获取警员信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> GetPoliceInfoByHkDevice([FromBody] GetPoliceInfoByHkDeviceReq req)
        {
            try
            {
                if(req == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = "无效参数",
                    };
                }

                var baseinfo = await _policeInfoService.QueryOne(s => s.HkFaceDevice == req.hkDevice);

                if (baseinfo != null)
                {
                    var info = new
                    {
                        policeId = baseinfo.id,
                        policeName = baseinfo.name,
                        orgId = baseinfo.orgId,
                    };

                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = ResultCode.OPERATE_SUCCESS.Msg,
                        data = info
                    };
                }
                else
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无匹配警员信息",
                    };
                }

            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 获取警员信息
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        public async Task<ApiResult> GetPoliceInfo()
        {
            try
            {
                var fingers = await _policeFingerService.Query();

                var info = (await _policeInfoService.Query())?.Select(s => new
                {
                    id = s.id,
                    name = s.name,
                    policeCode = s.HkFaceDevice,
                    photo = s.photo,
                    carNo = s.HkFaceDevice,
                    createTime = s.createTime,
                    updateTime = s.updateTime,
                    fingerList = fingers.Where(c => c.policeId.Equals(s.id)).Select(s => new { fingerInfo = s.fingerInfo, fingerNum = s.fingerNum }).ToList(),
                }).ToList();

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = info
                };

                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 通道上传出入库信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> UploadRFID([FromBody] ChannelLogReq req)
        {
            try
            {
                #region 合法判断
                if (req.Epc == null || req.Epc.Count() == 0)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "EPC为空",
                    };
                }

                if (req.state != 0 && req.state != 1)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "出入库状态错误",
                    };
                }

                if(req.warehouseId == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "仓库为空",
                    };
                }
                #endregion

                if(await _bussinessInventoryService.QueryOne(s => s.inventoryState.Equals("running")) != null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "正在盘点 禁止出入库",
                    };
                }

                //还原非单标签EPC
                List<string> PendingEpc = new List<string>();
                foreach (var item in req.Epc)
                {
                    var info = EpcConvert.EpcAnlysing(EpcConvert.ToHexByte(item));
                    if (info.EType != 0 && info.EType != 1)//如果不是单标签则还原
                    {
                        PendingEpc.Add(EpcConvert.RestoreEpc(item));
                    }
                    else
                    {
                        PendingEpc.Add(item);
                    }
                }

                var myOrder = await _orderService.QueryOrder(s => s.orderState == 2);//单据
                var warehouse = await _warehouseService.QueryOne(s => s.id.Equals(req.warehouseId));
                var warehouseInvtory = await _inventoryService.Query(s => !s.bussinessState.Equals("destruction") //已销毁
                && !s.state.Equals("2") //采购入库未结算
                && !s.state.Equals("3") //日常出库转领出
                && !(s.bussinessState.Equals("quick") && s.state.Equals("1")) //移库出库未入库
                && s.locationId.Equals(req.warehouseId));//仓库装备
                //&& !((myOrder != null && (myOrder.bussinessType.Equals("repair") && myOrder.orderType.Equals("in"))) && (s.bussinessState.Equals("repair") && s.state.Equals("1")))
                if (myOrder == null)
                {
                    warehouseInvtory = warehouseInvtory.Where(s => !(s.bussinessState.Equals("repair") && s.state.Equals("1"))).ToList();
                }
                else if(!(myOrder.bussinessType.Equals("repair") && myOrder.orderType.Equals("in")))
                {
                    warehouseInvtory = warehouseInvtory.Where(s => !(s.bussinessState.Equals("repair") && s.state.Equals("1"))).ToList();
                }

                myOrder = (myOrder != null && !myOrder.orderType.Equals(req.state == 0 ? "out" : "in")) ? null : myOrder;//如果出入状态与单据不匹配则作为无单据处理

                List<string> newEquOrderType = new List<string>() { "purchase", "allocate", "gift", "return" };//入库单据为这些种类则会有新装备

                List<string> epcList = new List<string>();//经过过滤后的EPC

                //初步过滤
                if (myOrder == null || !(newEquOrderType.Contains(myOrder.bussinessType) && myOrder.orderType.Equals("in")))//日常和无新装备单据的出入库过滤非本仓库物资
                {
                    if (warehouseInvtory == null || warehouseInvtory.Count() == 0)
                    {
                        return new ApiResult
                        {
                            code = ResultCode.OPERATE_FAILED.Code,
                            msg = "仓库无装备"
                        };
                    }

                    var removeEpcs = PendingEpc.Except(warehouseInvtory.Select(s => s.epc).ToList()).ToList();
                    epcList = PendingEpc.Where(s => !removeEpcs.Contains(s)).ToList();
                }
                else//有新装备单据过滤本仓库物资
                {
                    if (warehouseInvtory == null || warehouseInvtory.Count() == 0)
                    {
                        epcList = PendingEpc;
                    }
                    else
                    {
                        var keepEpcs = PendingEpc.Except(warehouseInvtory.Select(s => s.epc).ToList()).ToList();
                        epcList = PendingEpc.Where(s => keepEpcs.Contains(s)).ToList();
                    }
                }

                var deatilDic = myOrder?.DetailList.Where(s => s.warehouseId.Equals(req.warehouseId))
                    .Select(s =>s.equipmentSize)
                    .ToList();//通道所在仓库子单

                //创建出入库记录
                List<Inventory> newEquList = new List<Inventory>();
                List<LogDetail> logDetailList = new List<LogDetail>();

                var policeName = string.IsNullOrEmpty(req.policeId) ? "" : (await _policeInfoService.QueryOne(s => s.id.Equals(req.policeId)))?.name;

                var orgInfo = await _thisInfoService.QueryOne(s => true);

                var warrantyList = (await _viewWarrantyWarnService.Query()).Select(s => s.epc).ToList();//超期提醒
                Dictionary<string, int> warrantyDic = new Dictionary<string, int>();

                foreach (var epc in epcList)
                {
                    var epc_info = EpcConvert.EpcAnlysing(EpcConvert.ToHexByte(epc));
                    string typecode = epc_info.Wzdm.ToString("X");
                    string sizecode = epc_info.Hxdm.ToString("X");
                    string suppliercode = epc_info.OrganizationCode.ToString().ToLower();

                    var mytype = await _equipmentTypeService.QueryOne(s => s.id.Equals(typecode));
                    var mysize = await _equipmentSizeService.QueryOne(s => s.typeId.Equals(typecode) && s.code.Equals(sizecode));
                    var mysupplier = await _supplierService.QueryOne(s => s.code.ToLower().Equals(suppliercode.ToLower()));

                    if (mytype != null && mysize != null)
                    {
                        if(myOrder != null && deatilDic != null)//单据出入库则过滤非单据装备
                        {
                            if(!deatilDic.Contains(mysize.id))
                            {
                                continue;
                            }
                            else if(newEquOrderType.Contains(myOrder.bussinessType) && myOrder.orderType.Equals("in"))//有新装备单据生成库存记录
                            {
                                var periodInfo = (await _printService.QueryOne(s => s.id.Equals(myOrder.id)))?.DetailList.FirstOrDefault(s => s.sizeId.Equals(mysize.id));
                                newEquList.Add(new Inventory 
                                {
                                    id = Guid.NewGuid().ToString(),
                                    state = "2",
                                    epc = epc,
                                    typeId = mytype.id ?? "",
                                    sizeId = mysize.id ?? "",
                                    typeName = mytype.name ?? "",
                                    sizeName = mysize.name ?? "",
                                    bussinessState = "normal",
                                    equState = "normal",
                                    locationId = req.warehouseId,
                                    locationName = warehouse?.name,
                                    maintenancePeriod = Convert.ToString(Convert.ToInt32(periodInfo?.maintenancePeriod)),
                                    warrantyPeriod = Convert.ToString(Convert.ToInt32(periodInfo?.warrantyPeriod)),
                                    supplierId = mysupplier?.id ?? "",
                                    orgId = Convert.ToInt64(orgInfo.orgId),
                                    lostFlag = "0",
                                    equipmentType = "0",
                                    locationType = "0",
                                    supplierName = mysupplier?.name ?? "",
                                    productionDate = periodInfo?.productionDate,
                                    createTime = DateTime.Now,
                                    updateTime = DateTime.Now,
                                    price = myOrder.DetailList?.FirstOrDefault(s => s.equipmentSize.Equals(mysize.id))?.price ?? 0,
                                });
                            }
                        }

                        //出入库子单

                        var inv = await _inventoryService.QueryOne(s => s.epc.Equals(epc));
                        logDetailList.Add(new LogDetail() 
                        {
                            inventoryId = inv?.id,
                            epc = epc,
                            equipmentName = mytype?.name ?? "",
                            equipmentSize = mysize.name ?? "",
                            equipmentTypeId = mytype?.id ?? "",
                            equipmentSizeId = mysize.id ?? "",
                            errorState = 0,
                            state = req.state == 1 ? 0 : 1,
                            equipmentType = 0,
                            supplierId = mysupplier?.id ?? "",
                            supplierName = mysupplier?.name ?? "",
                            price = inv == null ? (myOrder.DetailList?.FirstOrDefault(s => s.equipmentSize.Equals(mysize.id))?.price ?? 0) : (inv?.price ?? 0),
                            createTime = DateTime.Now,
                            updateTime = DateTime.Now,
                        });

                        if(warrantyList != null && warrantyList.Contains(epc))
                        {
                            string thisKey = $"{mysize.id}||{mytype?.name}({mysize.name})";
                            if (warrantyDic.TryGetValue(thisKey, out int va))
                            {
                                warrantyDic[thisKey] += 1;
                            }
                            else
                            {
                                warrantyDic.Add(thisKey, 1);
                            }
                        }
                    }
                    else
                    {
                        continue;
                    }
                }

                //生成主单
                LogSummary summary = new LogSummary();
                if (logDetailList.Count() > 0)
                {
                    var names = logDetailList.GroupBy(s => s.equipmentName).Select(s => s.Key).ToList();

                    string equipments = "";
                    for (int i = 0; i < names.Count; i++)
                    {
                        equipments = (i == names.Count - 1) ? equipments + $"{names[i]}" : equipments + $"{names[i]},";
                    }

                    summary = new LogSummary()
                    {
                        locationId = req.warehouseId,
                        locationName = warehouse.name,
                        userName = policeName,
                        useTime = DateTime.Now,
                        createTime = DateTime.Now,
                        updateTime = DateTime.Now,
                        bussinessType = myOrder?.bussinessType ?? "normal",
                        deviceType = 2,
                        number = logDetailList.Count(),
                        outInState = req.state == 0 ? "out" : "in",
                        equipments = equipments,
                        orderId = myOrder?.id,
                        orderCode = myOrder?.orderCode,
                        isUpload = 0,
                        price = logDetailList.Sum(s => s.price),
                        DetailList = logDetailList,
                        picture = req.picture,
                    };
                }

                var rs = await _logService.AddChannelLog(myOrder, summary, newEquList);

                if (rs)
                {
                    string msg = $"{(req.state == 0 ? "出库" : "入库")}成功{summary.number}件";
                    if (warrantyDic.Any())
                    {
                        msg += ",即将超期或已超期:";
                        foreach (var item in warrantyDic)
                        {
                            var name = item.Key.Split(new string[] { "||" }, StringSplitOptions.None).Skip(1).FirstOrDefault();
                            msg += $"{name}{item.Value}件,";
                        };
                        msg = msg.Substring(0, msg.Length - 1);
                    }

                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = msg
                    };
                }
                else
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = ResultCode.OPERATE_FAILED.Msg
                    };
                }

            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 更新警员指纹
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> UpdateFinger([FromBody] UpdateFingerReq req)
        {
            try
            {
                if(req == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "参数错误",
                    };
                }else if(string.IsNullOrEmpty(req.policeId))
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "警员编号为空",
                    };
                }else if(req.fingerList == null || req.fingerList.Count() == 0)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "指纹信息为空",
                    };
                }

                var police = await _policeInfoService.QueryOne(s => s.id.Equals(req.policeId));
                if(police == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无警员信息",
                    };
                }else
                {
                    var send = new
                    {
                        policeId = req.policeId,
                        fingerList = req.fingerList.Select(s => new { fingerInfo = s.fingerInfo, fingerNum = s.fingerNum })
                    };
                    var json = HttpHelper.HttpPost(/*AdminGlobalContext.jyzbConfig.Url*/"http://192.168.2.14:10030" + "/PolicemanFinger/insertOrUpdateFingerInfo", JsonConvert.SerializeObject(send));
                    var httprs = JsonConvert.DeserializeObject<HttpHelper.res<string>>(json);

                    if(httprs != null && httprs.code == "99200")
                    {
                        var rs = await _policeFingerService.UpdateFinger(req.policeId, req.fingerList);

                        return new ApiResult
                        {
                            code = ResultCode.OPERATE_FAILED.Code,
                            msg = ResultCode.OPERATE_FAILED.Msg,
                        };
                    }
                    else
                    {
                        return new ApiResult
                        {
                            code = ResultCode.OPERATE_FAILED.Code,
                            msg = "上传平台失败",
                        };
                    }
                    
                }
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 获取装备类别
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        public async Task<ApiResult> GetEquipmentType()
        {
            try
            {
                var types = (await _equipmentTypeService.Query())?.Select(s => new
                {
                    id = s.id,
                    name = s.name,
                    code = s.id,
                }).ToList();


                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = types
                };

                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 获取本组织机构所有仓库
        /// </summary>
        /// <param name=""></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<ApiResult> GetWarehouse()
        {
            try
            {
                var warehouses = (await _warehouseService.Query())?.Select(s => new 
                { 
                    id = s.id,
                    name = s.name,
                }).ToList();


                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = warehouses
                };

                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }


        /// <summary>
        /// 通道出入库照片上传
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> UploadLogPicture([FromForm] LogPictureReq req)
        {
            try
            {
                if(req.picture == null)
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无图片",
                    };
                }

                if (string.IsNullOrEmpty(req.fileName))
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无图片名称",
                    };
                }

                if (string.IsNullOrEmpty(req.directoy))
                {
                    return new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "无文件夹名称",
                    };
                }

                using (MemoryStream ms = new MemoryStream())
                {
                    //拷贝文件到内存流
                    req.picture.CopyTo(ms);
                    var tmp = ms.GetBuffer();
                    var fileType = tmp[0].ToString() + tmp[1].ToString();

                    var path = Path.Combine("LogPictures", req.directoy);

                    Directory.CreateDirectory(path);
                    var filePath = Path.Combine(path, req.fileName);

                    System.Drawing.Image.FromStream(ms).Save(filePath);


                    var src = new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = ResultCode.OPERATE_SUCCESS.Msg,
                    };

                    return src;
                }
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 获取出入库照片
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> GetLogPicture([FromBody] GetLogPictureReq req)
        {
            try
            {
                //读取图片
                string imageBytes = "";

                string url = $"LogPictures/{req.pid}";
                using (FileStream fileStream = new FileStream(url, FileMode.Open))
                {
                    using (BinaryReader reader = new BinaryReader(fileStream))
                    {
                        imageBytes = Convert.ToBase64String(reader.ReadBytes((int)fileStream.Length));
                    }
                }

                var res = new DownloadPicRes() { picture = imageBytes };

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = res
                };

                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 获取账号
        /// </summary>
        /// <param name=""></param>
        /// <returns></returns>
        [HttpGet]
        public virtual async Task<ApiResult> GetAccount()
        {
            try
            {
                var accountlist = await _usersService.Query();

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = accountlist
                };

                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 手持机出入库记录上传海南版
        /// </summary>
        /// <param name=""></param>
        /// <returns></returns>
        [HttpPost]
        public virtual async Task<ApiResult> UploadHandRecordSP([FromBody] HandRecordsReq req)
        {
            try
            {
                var rs = await _logService.AddHandLogs(req);

                var src = rs ? new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                } : new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ResultCode.OPERATE_FAILED.Msg,
                };
                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 盘库记录上传
        /// </summary>
        /// <param name=""></param>
        /// <returns></returns>
        [HttpPost]
        public virtual async Task<ApiResult> UploadInv([FromBody] Models.ReqModel.InvReq req)
        {
            try
            {
                Expression<Func<Inventory, bool>> expression = s => s.lostFlag == "1";
                var inv = await _inventoryService.Query(expression);

                var lostInv = inv?.Select(s => new Inventory()
                {
                    id = s.id,
                    epc = s.epc,
                    typeName = s.typeName,
                    sizeName = s.sizeName,
                    lostFlag = "0",
                    updateTime = DateTime.Now,
                }).ToList();

                var rs = _invService.AddInvLogs(req, lostInv).Result;

                var src = rs ? new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                } : new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ResultCode.OPERATE_FAILED.Msg,
                };
                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 获取所有装备信息
        /// </summary>
        /// <param name=""></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<ApiResult> GetInventory()
        {
            try
            {
                Expression<Func<Inventory, bool>> expression = s => s.locationType != "1" && !s.bussinessState.Equals("destruction");
                var invlist = await _inventoryService.Query(expression);

                if (invlist == null)
                {
                    var error = new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "库存为空",
                    };
                    return error;
                }

                var baseinv = invlist.Select(s => new
                {
                    typeId = s.typeId,
                    typeName = s.typeName,
                    sizeId = s.sizeId,
                    sizeName = s.sizeName,
                    equipId = s.id,
                    epc = s.epc,
                    state = s.state,
                }).ToList();

                var inv = baseinv.GroupBy(x => new { x.typeId, x.typeName, x.sizeName, x.sizeId })
                    .Select(s => new
                    {
                        typeId = s.Key.typeId,
                        typeName = s.Key.typeName,
                        sizeId = s.Key.sizeId,
                        sizeName = s.Key.sizeName,
                        epcList = baseinv.Where(a => a.sizeId.Equals(s.Key.sizeId))
                        .Select(b => new
                        {
                            equipId = b.equipId,
                            epc = b.epc,
                            state = b.state,
                        }).ToList()
                    }).ToList();

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = inv
                };
                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 根据人员id删除照片及照片数据库信息
        /// </summary>
        /// <param name=""></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> DeletePictureById([FromForm] DelPicReq req)
        {
            try
            {
                var police = await _policeService.QueryOne(p => p.id.Equals(req.id));

                if(police == null)
                {
                    var error = new ApiResult
                    {
                        code = ResultCode.POLICE_ERROR.Code,
                        msg = ResultCode.POLICE_ERROR.Msg,
                    };
                    return error;
                }

                string delurl = police.picUrl;

                if (string.IsNullOrEmpty(delurl))
                {
                    var error = new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "照片信息不存在",
                    };
                    return error;
                }

                //删除数据库照片信息
                police.faceInfo = "";
                police.picUrl = "";
                police.picUpdateTime = DateTime.Now;

                bool rs = await _policeService.Update(police);

                if (!rs)
                {
                    var error = new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "照片信息删除失败",
                    };
                    return error;
                }

                //删除照片
                FileInfo mypic = new FileInfo(delurl);
                mypic.Delete();

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg
                };
                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 人员照片下载
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> DownloadPicture([FromForm] DownloadPicReq req)
        {
            try
            {
                var pic = await _policeService.QueryOne(p => !string.IsNullOrEmpty(p.picUrl) && p.picUrl.Equals(req.picUrl));

                //不存在
                if (pic == null)
                {
                    var error = new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "照片不存在",
                    };
                    return error;
                }

                //读取图片
                string imageBytes = "";

                string url = pic.picUrl;
                using (FileStream fileStream = new FileStream(url, FileMode.Open))
                {
                    using (BinaryReader reader = new BinaryReader(fileStream))
                    {
                        imageBytes = Convert.ToBase64String(reader.ReadBytes((int)fileStream.Length));
                    }
                }

                var res = new DownloadPicRes() { picture = imageBytes };

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = res
                };

                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }

        /// <summary>
        /// 人员照片上传
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public async Task<ApiResult> UploadPicture([FromForm] PictureReq req)
        {
            try
            {
                var police = await _policeService.QueryOne(p => p.id == req.policeId);

                //警员不存在
                if (police == null)
                {
                    var error = new ApiResult
                    {
                        code = ResultCode.POLICE_ERROR.Code,
                        msg = "警员不存在",
                    };
                    return error;
                }

                //存储图片
                var ext = Path.GetExtension(req.picture.FileName).ToLower();

                using (MemoryStream ms = new MemoryStream())
                {
                    //拷贝文件到内存流
                    req.picture.CopyTo(ms);
                    var tmp = ms.GetBuffer();
                    var fileType = tmp[0].ToString() + tmp[1].ToString();

                    var path = "Pictures";
                    var fileName = $"{req.policeId}{Path.GetExtension(req.picture.FileName)}";

                    Directory.CreateDirectory(path);
                    var filePath = Path.Combine(path, fileName);

                    System.Drawing.Image.FromStream(ms).Save(filePath);

                    //更新警员照片地址
                    police.picUrl = filePath;
                    police.updateTime = DateTime.Now;
                    police.picUpdateTime = police.updateTime;
                    bool rs = await _policeService.Update(police);

                    var reply = new UploadPictureReply() { picUrl = filePath, updateTime = (DateTime)police.picUpdateTime }; 

                    var src = rs ? new ApiResult
                    {
                        code = ResultCode.OPERATE_SUCCESS.Code,
                        msg = ResultCode.OPERATE_SUCCESS.Msg,
                        data = reply
                    } : new ApiResult
                    {
                        code = ResultCode.OPERATE_FAILED.Code,
                        msg = "更新照片失败"
                    };

                    return src;

                }
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                };
                return error;
            }
        }
        class UploadPictureReply 
        {
            public string picUrl {  get; set; }
            public DateTime updateTime { get; set; }
        }


        /// <summary>
        /// 新增基础信息
        /// </summary>
        /// <param name="req"></param>
        /// <returns></returns>
        [HttpPost]
        public virtual async Task<ApiResult> UploadInfo([FromBody] InfoListReq req)
        {

            try
            {
                string idlist = string.Join(",", req.infoList.Select(d => d.policeId).ToArray());
                var list = idlist.Split(",");
                var police = await _policeService.Query(p => list.Contains(p.id));
                if (police != null)
                {
                    foreach (var item in police)
                    {
                        var itemnew = req.infoList.Where(p => p.policeId.Equals(item.id)).FirstOrDefault();
                        if (itemnew != null)
                        {
                            item.fingerInfo = itemnew.fingerInfo;
                            item.faceInfo = itemnew.faceInfo;
                        }
                        
                    }
               
                    int result=  _policeService.UpdateObjs(police);
                    if (result > 0)
                    {
                        var src = new ApiResult
                        {
                            code = ResultCode.OPERATE_SUCCESS.Code,
                            msg = ResultCode.OPERATE_SUCCESS.Msg,
                            data = police
                        };
                        return src;
                    }
                    else {
                        var src = new ApiResult
                        {
                            code = ResultCode.POLICE_ERROR.Code,
                            msg = ResultCode.POLICE_ERROR.Msg,
                            data = ""
                        };
                        return src;
                    }
                  
                }
                else
                {
                    var src = new ApiResult
                    {
                        code = ResultCode.POLICE_ERROR.Code,
                        msg = ResultCode.POLICE_ERROR.Msg,
                        data = ""
                    };
                    return src;
                }
                
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,

                };
                return error;
            }

        }

        /// <summary>
        /// 查询人员信息
        /// </summary>
        /// <param name="req"></param>
        /// <returns></returns>
        [HttpPost]
        public virtual async Task<ApiResult> GetAllPersonnel([FromBody] CommonReq req)
        {

            try
            {
                DateTime reqTime = DateTime.MinValue;
                if (!string.IsNullOrEmpty(req.updateTime))
                {
                    reqTime = DateTime.Parse(req.updateTime);
                }
                var police = await _policeService.Query(p=>p.updateTime>= reqTime);
                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = police
                };
                return src;
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,

                };
                return error;
            }

        }

        /// <summary>
        /// 查询装备类型信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]

        public  async Task<string> GetEquipmentType([FromBody] CommonReq req)
        {

            try
            {
                DateTime reqTime = DateTime.MinValue;
                if (!string.IsNullOrEmpty(req.updateTime))
                {
                    reqTime = DateTime.Parse(req.updateTime);
                }
                var typeQuery = await _equipmentTypeService.Query(p => p.updateTime >= reqTime);
                List<EquipmentTypeDto> typeDto = new List<EquipmentTypeDto>() ;
                Mapper.Map(typeQuery, typeDto);
                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data= typeDto
                };
                return JsonConvert.SerializeObject(src);
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,
                  
                };
                return JsonConvert.SerializeObject(error);
            }

        }

        /// <summary>
        /// 查询装备号型信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]

        public async Task<string> GetEquipmentSize([FromBody] CommonReq req)
        {

            try
            {
                DateTime reqTime= DateTime.MinValue;
                if (!string.IsNullOrEmpty(req.updateTime))
                {
                    reqTime = DateTime.Parse(req.updateTime);
                }
                var sizeQuery = await _equipmentSizeService.Query(p => p.updateTime >= reqTime);
                List<EquipmentSizeDto> sizeDto = new List<EquipmentSizeDto>();
                Mapper.Map(sizeQuery, sizeDto);
                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = sizeDto
                };
               
                return JsonConvert.SerializeObject(src);
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,

                };
                return JsonConvert.SerializeObject(error); 
            }

        }
        /// <summary>
        /// 查询车辆信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]

        public async Task<string> GetCar([FromBody] CommonReq req)
        {

            try
            {
                DateTime reqTime = DateTime.MinValue;
                if (!string.IsNullOrEmpty(req.updateTime))
                {
                    reqTime = DateTime.Parse(req.updateTime);
                }
                var carQuery = await _carService.Query(p => p.updateTime >= reqTime);
                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = carQuery
                };

                return JsonConvert.SerializeObject(src);
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,

                };
                return JsonConvert.SerializeObject(error);
            }

        }

        /// <summary>
        /// 查询当前时间信息
        /// </summary>
        /// <returns></returns>
        [HttpGet]

        public async Task<string> GetTime()
        {
            try
            {

                var src = new ApiResult
                {
                    code = ResultCode.OPERATE_SUCCESS.Code,
                    msg = ResultCode.OPERATE_SUCCESS.Msg,
                    data = DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")
            };

                return JsonConvert.SerializeObject(src);
            }
            catch (Exception ex)
            {
                var error = new ApiResult
                {
                    code = ResultCode.OPERATE_FAILED.Code,
                    msg = ex.Message,

                };
                return JsonConvert.SerializeObject(error);
            }
         
        }
        /// <summary>
        /// 发送消息队列测试
        /// </summary>
        /// <returns></returns>
        [HttpPost]

        public async Task<string> SendMQ([FromBody] RecordsReq req)
        {
            
            await RabbitMQContext.SendMessageAsynce(JsonConvert.SerializeObject(req));
            var src = new ApiResult
            {
                code = ResultCode.OPERATE_SUCCESS.Code,
                msg = ResultCode.OPERATE_SUCCESS.Msg
                
            };
            return JsonConvert.SerializeObject(src);
        }
    }
}
